# 機能設計書 53-通知

## 概要

本ドキュメントは、VS Codeの通知機能に関する設計書である。ユーザーへのメッセージ表示、プロンプト、ステータスメッセージなど、非モーダルな情報伝達手段を提供する。トースト通知や通知センターを通じて、重要な情報をユーザーに伝達する。

### 本機能の処理概要

**業務上の目的・背景**：
VS Codeは多くのイベント（ファイル保存、拡張機能更新、エラー発生など）をユーザーに通知する必要がある。モーダルダイアログと異なり、通知は作業を中断せずにユーザーに情報を伝達でき、ユーザー体験を向上させる。緊急度に応じた優先度制御や「今後表示しない」オプションにより、通知疲れを防止する。

**機能の利用シーン**：
- 拡張機能のインストール・更新完了時
- ファイル保存やビルド完了時のフィードバック
- エラーや警告の通知
- ユーザーへの確認・選択肢の提示
- 長時間処理の進捗表示

**主要な処理内容**：
1. 情報/警告/エラー通知の表示（info, warn, error）
2. 選択肢付きプロンプトの表示（prompt）
3. ステータスバーへのメッセージ表示（status）
4. 通知の優先度制御（DEFAULT, OPTIONAL, SILENT, URGENT）
5. 「今後表示しない」オプションの管理
6. 通知フィルターの設定と適用
7. 通知の進捗表示とメッセージ更新

**関連システム・外部連携**：
- ストレージサービス（IStorageService）: 「今後表示しない」設定の永続化
- ダイアログサービス（IDialogService）: モーダル確認が必要な場合の連携

**権限による制御**：
通知ソースごとにフィルター設定可能。「おやすみモード」相当の機能でURGENT以外の通知を抑制可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | ワークベンチ | 補助機能 | 通知センター、トースト通知の統合管理 |
| 6 | ステータスバー | 補助機能 | 通知ステータスアイコンの表示 |
| 8 | バナー | 主機能 | 信頼性や更新情報等のバナー通知表示 |
| 45 | 通知センター | 主機能 | 通知メッセージのセンター形式表示 |
| 50 | ウェルカムバナー | 主機能 | エディタ領域のウェルカムバナー表示 |

## 機能種別

UI表示 / メッセージング / ユーザーインタラクション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| message | NotificationMessage | Yes | 通知メッセージ（文字列またはError） | リンク形式[text](link)対応 |
| severity | Severity | No | 重要度（Info/Warning/Error） | Info/Warning/Errorのいずれか |
| choices | IPromptChoice[] | No | 選択肢（prompt用） | 配列 |
| options | INotificationProperties | No | 追加オプション（sticky, priority等） | - |

### 入力データソース

- プログラムコードからの直接呼び出し
- 拡張機能からのAPI経由
- 内部イベント（ファイル監視、ビルド完了等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| handle | INotificationHandle | 通知制御用ハンドル |
| statusHandle | IStatusHandle | ステータスメッセージ制御用ハンドル |

### 出力先

- トースト通知（画面右下）
- 通知センター（ステータスバーアイコンから開く）
- ステータスバー（短いメッセージ）

## 処理フロー

### 処理シーケンス

```
1. 通知メソッド呼び出し（notify/info/warn/error/prompt）
   └─ メッセージとオプションを受け取る
2. 通知フィルターチェック
   └─ ソースごとのフィルター設定を確認
3. 優先度チェック
   └─ おやすみモード時はURGENTのみ表示
4. 重複チェック
   └─ 同一IDの通知がある場合はスキップまたは更新
5. 通知の表示
   └─ トースト通知または通知センターに追加
6. アクション待機
   └─ ユーザーの選択またはクローズを待機
7. コールバック実行
   └─ 選択されたアクションのrun()を実行
```

### フローチャート

```mermaid
flowchart TD
    A[notify呼び出し] --> B{フィルターチェック}
    B -->|フィルター適用| C[通知抑制]
    B -->|フィルターなし| D{優先度チェック}
    D -->|おやすみモード & 非URGENT| C
    D -->|表示OK| E{重複チェック}
    E -->|重複あり| F[既存通知を更新]
    E -->|重複なし| G[新規通知作成]
    G --> H{sticky?}
    H -->|Yes| I[永続通知として表示]
    H -->|No| J[一定時間後に自動非表示]
    I --> K[アクション待機]
    J --> K
    K --> L{ユーザー操作}
    L -->|アクション選択| M[コールバック実行]
    L -->|クローズ| N[onDidClose発火]
    M --> N
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-53-01 | 優先度制御 | URGENTはおやすみモードでも表示 | 常時 |
| BR-53-02 | Sticky通知 | エラー+プライマリアクション、進捗表示は自動非表示しない | 該当条件時 |
| BR-53-03 | 今後表示しない | NeverShowAgainオプションで永続的に非表示可能 | ユーザー選択時 |
| BR-53-04 | 重複防止 | 同一IDの通知は重複表示しない | id指定時 |
| BR-53-05 | WCAG準拠 | 重要度に応じたプレフィックス付与 | 常時 |

### 計算ロジック

```typescript
// 重要度プレフィックス付与（WCAG 4.1.3準拠）
function withSeverityPrefix(label: string, severity: Severity): string {
  if (severity === Severity.Error) {
    return `Error: ${label}`;
  }
  if (severity === Severity.Warning) {
    return `Warning: ${label}`;
  }
  return `Info: ${label}`;
}

// 通知優先度
enum NotificationPriority {
  DEFAULT,   // 通常：おやすみモードで非表示
  OPTIONAL,  // オプション：通知センターのみ
  SILENT,    // サイレント：通知センターのみ
  URGENT     // 緊急：常に表示
}
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 今後表示しない保存 | storage | INSERT/UPDATE | NeverShowAgainの設定保存 |
| フィルター設定保存 | storage | INSERT/UPDATE | ソースごとのフィルター設定 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | アクションエラー | 選択アクションのrun()でエラー | エラー通知を表示 |

### リトライ仕様

該当なし（通知は即座に処理）

## トランザクション仕様

該当なし

## パフォーマンス要件

- 通知表示はUIスレッドで即座に実行
- 大量の通知が同時に発生した場合のスロットリング

## セキュリティ考慮事項

- メッセージ内のリンクは安全な形式のみ許可
- 拡張機能からの通知はソース情報を表示

## 備考

- モーダルな確認が必要な場合はIDialogServiceを使用
- 通知メッセージはMarkdown形式のリンク[text](link)をサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

通知のインターフェースと型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | notification.ts | `src/vs/platform/notification/common/notification.ts` | INotificationService、INotification、関連型定義 |

**読解のコツ**:
- Severityは共通のbase/common/severity.tsからインポート
- NotificationPriorityで表示優先度を制御
- NeverShowAgainScopeでスコープを指定（WORKSPACE/PROFILE/APPLICATION）

**主要処理フロー**:
1. **14行目**: INotificationServiceのdecorator登録
2. **18-39行目**: NotificationPriority enum定義
3. **41-62行目**: INotificationProperties - sticky、priority、neverShowAgain
4. **64-82行目**: NeverShowAgainScope - 保存スコープ定義
5. **128-172行目**: INotification - 通知本体の型定義
6. **174-190行目**: INotificationActions - プライマリ/セカンダリアクション
7. **233-274行目**: INotificationHandle - 通知制御ハンドル
8. **371-454行目**: INotificationService - サービスインターフェース

#### Step 2: 便利メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | notification.ts | `src/vs/platform/notification/common/notification.ts` | info/warn/error/prompt/statusメソッド |

**主要処理フロー**:
- **418-430行目**: info/warn/error - 簡易通知メソッド
- **443-453行目**: prompt - 選択肢付き通知
- **477-491行目**: withSeverityPrefix - WCAG準拠のプレフィックス付与

#### Step 3: プロンプト関連の型を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | notification.ts | `src/vs/platform/notification/common/notification.ts` | IPromptChoice、IPromptOptions |

**主要処理フロー**:
- **284-301行目**: IBasePromptChoice - 基本選択肢
- **303-310行目**: IPromptChoice - プライマリ/セカンダリ区分
- **312-323行目**: IPromptChoiceWithMenu - メニュー付き選択肢
- **325-332行目**: IPromptOptions - onCancelコールバック

### プログラム呼び出し階層図

```
INotificationService (サービスインターフェース)
    │
    ├─ notify() - 詳細な通知
    │      │
    │      ├─ INotification (通知データ)
    │      │      ├─ severity
    │      │      ├─ message
    │      │      ├─ actions
    │      │      └─ progress
    │      │
    │      └─ INotificationHandle (制御ハンドル)
    │             ├─ onDidClose
    │             ├─ updateMessage()
    │             ├─ updateSeverity()
    │             └─ close()
    │
    ├─ info() / warn() / error() - 簡易通知
    │      └─ notify()に委譲
    │
    ├─ prompt() - 選択肢付き通知
    │      │
    │      ├─ IPromptChoice[]
    │      │      ├─ label
    │      │      ├─ run()
    │      │      └─ isSecondary
    │      │
    │      └─ INotificationHandle
    │
    └─ status() - ステータスバーメッセージ
           └─ IStatusHandle
                  └─ close()
```

### データフロー図

```
[入力]               [処理]                    [出力]

notify()
    │
    ↓
┌───────────────┐
│ フィルター    │
│  チェック     │──────→ 抑制 → 終了
└───────────────┘
    │OK
    ↓
┌───────────────┐
│ 優先度チェック │──────→ 非表示 → 通知センターのみ
│ (おやすみモード)│
└───────────────┘
    │表示OK
    ↓
┌───────────────┐
│ 重複チェック  │──────→ 重複 → 既存更新
└───────────────┘
    │新規
    ↓
┌───────────────┐
│ 通知作成      │
│ (INotification)│
└───────────────┘
    │
    ├──────────────────┐
    ↓                  ↓
┌──────────┐    ┌──────────┐
│トースト   │    │通知センター│
│(右下)     │    │  (一覧)   │
└──────────┘    └──────────┘
    │                │
    ↓                ↓
[ユーザー操作]  [ユーザー操作]
    │                │
    └────────────────┘
           │
           ↓
    ┌──────────┐
    │アクション │
    │ 実行     │
    └──────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| notification.ts | `src/vs/platform/notification/common/notification.ts` | ソース | インターフェース定義 |
| testNotificationService.ts | `src/vs/platform/notification/test/common/testNotificationService.ts` | テスト | テスト用モック実装 |
| severity.ts | `src/vs/base/common/severity.ts` | ソース | Severity enum定義 |
