# 機能設計書 56-ウィンドウ管理

## 概要

本ドキュメントは、VS Codeのウィンドウ管理機能に関する設計書である。アプリケーションウィンドウの作成・管理、レイアウト制御、タイトルバースタイル、ウィンドウサイズ・位置の管理などを提供する。

### 本機能の処理概要

**業務上の目的・背景**：
VS Codeはマルチウィンドウ対応のアプリケーションであり、複数のワークスペースを別々のウィンドウで開くことができる。ウィンドウの状態（サイズ、位置、最大化状態など）を適切に管理し、ユーザーの作業環境を維持することが重要である。また、プラットフォーム固有のウィンドウ外観（タイトルバー、メニュー等）をカスタマイズ可能にする。

**機能の利用シーン**：
- 新しいウィンドウでフォルダやワークスペースを開く
- ウィンドウの最小化・最大化・復元
- タイトルバースタイルの変更（ネイティブ/カスタム）
- ウィンドウ間でのフォルダ追加・削除
- セッション復元時のウィンドウ状態の再現

**主要な処理内容**：
1. ウィンドウの作成・オープン（openWindow）
2. ウィンドウ状態の管理（サイズ、位置、最大化）
3. タイトルバースタイルの制御（native/custom）
4. メニューバーの表示制御
5. ウィンドウコントロールスタイル（native/custom/hidden）
6. ズームレベルの管理
7. 補助ウィンドウの管理

**関連システム・外部連携**：
- Electron: ネイティブウィンドウAPI
- 設定サービス（IConfigurationService）: ウィンドウ設定の読み込み

**権限による制御**：
特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | ワークベンチ | 主機能 | ワークベンチウィンドウの初期化、レイアウト管理、パーツの配置 |
| 2 | タイトルバー | 補助機能 | ウィンドウコントロール（最小化、最大化、閉じる） |
| 53 | スプラッシュ画面 | 主機能 | 起動時のスプラッシュスクリーン表示 |

## 機能種別

ウィンドウ管理 / UI制御 / 状態管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| uriToOpen | IWindowOpenable[] | No | 開くURIリスト | workspace/folder/file URI |
| options | IOpenWindowOptions | No | オープンオプション | - |
| forceNewWindow | boolean | No | 新規ウィンドウ強制 | - |
| remoteAuthority | string | No | リモート接続先 | - |

### 入力データソース

- コマンドライン引数
- ファイル/フォルダのドロップ
- メニュー操作
- プログラムコードからの呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| windowId | number | 作成されたウィンドウID |
| openedWindow | IOpenedMainWindow | 開かれたウィンドウ情報 |

### 出力先

- 新規ウィンドウの表示
- 既存ウィンドウへのフォーカス

## 処理フロー

### 処理シーケンス

```
1. openWindow呼び出し
   └─ オプションとURIを受け取る
2. ウィンドウ再利用判定
   └─ forceReuseWindow/forceNewWindowを確認
3. 既存ウィンドウ検索
   └─ 同一ワークスペースのウィンドウを検索
4. ウィンドウ作成/再利用
   └─ 新規作成または既存ウィンドウを使用
5. 設定の適用
   └─ タイトルバー、メニュースタイル等
6. コンテンツのロード
   └─ ワークスペース/フォルダを開く
```

### フローチャート

```mermaid
flowchart TD
    A[openWindow] --> B{forceNewWindow?}
    B -->|Yes| C[新規ウィンドウ作成]
    B -->|No| D{forceReuseWindow?}
    D -->|Yes| E[既存ウィンドウ再利用]
    D -->|No| F{同一ワークスペース?}
    F -->|Yes| E
    F -->|No| G{設定確認}
    G -->|newWindow| C
    G -->|reuseWindow| E
    C --> H[設定適用]
    E --> H
    H --> I[コンテンツロード]
    I --> J[ウィンドウ表示]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-56-01 | 最小サイズ | 幅400px、高さ270px以上 | 常時 |
| BR-56-02 | パネル付き最小幅 | 垂直パネル時は幅600px以上 | パネル表示時 |
| BR-56-03 | タイトルバー選択 | nativeTabsやsimpleFullScreen時はnative | macOS |
| BR-56-04 | デフォルトタイトルバー | Web以外ではcustomがデフォルト | デスクトップ |
| BR-56-05 | ズームレベル計算 | scale = 1.2 ^ level | ズーム時 |

### 計算ロジック

```typescript
// ウィンドウ最小サイズ
const WindowMinimumSize = {
  WIDTH: 400,
  WIDTH_WITH_VERTICAL_PANEL: 600,
  HEIGHT: 270
};

// ズームレベルからズーム倍率への変換
function zoomLevelToZoomFactor(zoomLevel = 0): number {
  return 1.2 ** zoomLevel;
}

// デフォルトウィンドウサイズ
const DEFAULT_EMPTY_WINDOW_SIZE = { width: 1200, height: 800 };
const DEFAULT_WORKSPACE_WINDOW_SIZE = { width: 1440, height: 900 };
const DEFAULT_AUX_WINDOW_SIZE = { width: 1024, height: 768 };

// タイトルバースタイル判定
function getTitleBarStyle(configurationService): TitlebarStyle {
  if (isWeb) {
    return TitlebarStyle.CUSTOM;
  }
  // nativeTabsやsimpleFullScreen時はNATIVE
  if (nativeTabs || !nativeFullScreen) {
    return TitlebarStyle.NATIVE;
  }
  return configuration.titleBarStyle ?? TitlebarStyle.CUSTOM;
}
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ウィンドウ状態保存 | storage | UPDATE | サイズ、位置、最大化状態の保存 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ウィンドウ作成失敗 | リソース不足等 | エラーログ出力 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- ウィンドウ作成は可能な限り高速に
- スプラッシュスクリーンで体感速度を向上

## セキュリティ考慮事項

- リモート接続時のremoteAuthority検証
- 信頼されていないワークスペースの警告

## 備考

- macOSではネイティブタブがサポートされている
- Window Controls Overlayはデスクトップのみ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ウィンドウ関連の型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | window.ts | `src/vs/platform/window/common/window.ts` | ウィンドウ設定、オープンオプション、タイトルバースタイル |

**読解のコツ**:
- TitlebarStyle、WindowControlsStyleはenumで定義
- IOpenWindowOptionsでウィンドウオープン時のオプションを指定

**主要処理フロー**:
1. **22-26行目**: WindowMinimumSize - ウィンドウ最小サイズ定義
2. **38-56行目**: IBaseOpenWindowsOptions、IOpenWindowOptions - オープンオプション
3. **130-139行目**: MenuSettings、MenuStyleConfiguration - メニュー設定
4. **220-234行目**: TitleBarSetting、TitlebarStyle、WindowControlsStyle enum
5. **242-280行目**: getTitleBarStyle関数 - タイトルバースタイル判定
6. **296行目**: DEFAULT_CUSTOM_TITLEBAR_HEIGHT = 35
7. **473-479行目**: zoomLevelToZoomFactor、デフォルトサイズ定義

#### Step 2: ウィンドウ設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | window.ts | `src/vs/platform/window/common/window.ts` | IWindowSettings、IWindowsConfiguration |

**主要処理フロー**:
- **188-214行目**: IWindowSettings - すべてのウィンドウ設定項目
- **141-174行目**: hasNativeContextMenu、hasNativeMenu関数

#### Step 3: プラットフォーム固有の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | window.ts | `src/vs/platform/window/electron-main/window.ts` | メインプロセス側実装 |
| 3-2 | window.ts | `src/vs/platform/window/electron-browser/window.ts` | レンダラー側実装 |

### プログラム呼び出し階層図

```
Window Management
    │
    ├─ IWindowOpenable (開くもの)
    │      ├─ IWorkspaceToOpen (workspaceUri)
    │      ├─ IFolderToOpen (folderUri)
    │      └─ IFileToOpen (fileUri)
    │
    ├─ IOpenWindowOptions (オプション)
    │      ├─ forceNewWindow
    │      ├─ forceReuseWindow
    │      ├─ remoteAuthority
    │      └─ addMode / removeMode
    │
    ├─ TitlebarStyle
    │      ├─ NATIVE
    │      └─ CUSTOM
    │
    ├─ WindowControlsStyle
    │      ├─ NATIVE
    │      ├─ CUSTOM
    │      └─ HIDDEN
    │
    └─ IWindowSettings
           ├─ titleBarStyle
           ├─ menuBarVisibility
           ├─ zoomLevel
           ├─ restoreWindows
           └─ newWindowDimensions
```

### データフロー図

```
[入力]               [処理]                    [出力]

openWindow()
    │
    ↓
┌───────────────┐
│オプション解析 │
│ - forceNew    │
│ - forceReuse  │
└───────────────┘
    │
    ↓
┌───────────────┐
│ウィンドウ検索 │
│ (既存確認)    │
└───────────────┘
    │
    ├──────────────────┐
    ↓                  ↓
┌──────────┐    ┌──────────┐
│新規作成   │    │再利用    │
└──────────┘    └──────────┘
    │                │
    └────────────────┘
           │
           ↓
    ┌──────────────┐
    │設定適用      │
    │ - titleBar   │
    │ - menuBar    │
    │ - zoom       │
    └──────────────┘
           │
           ↓
    ┌──────────────┐
    │コンテンツ    │
    │ロード        │
    └──────────────┘
           │
           ↓
    [ウィンドウ表示]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| window.ts | `src/vs/platform/window/common/window.ts` | ソース | 共通型定義・ユーティリティ |
| window.ts | `src/vs/platform/window/electron-main/window.ts` | ソース | メインプロセス実装 |
| window.ts | `src/vs/platform/window/electron-browser/window.ts` | ソース | レンダラー実装 |
