# 機能設計書 60-更新管理

## 概要

本ドキュメントは、VS Codeの更新管理機能に関する設計書である。アプリケーションの自動更新チェック、ダウンロード、インストールを管理し、ユーザーに最新バージョンを提供する。

### 本機能の処理概要

**業務上の目的・背景**：
VS Codeは定期的にアップデートがリリースされ、新機能、バグ修正、セキュリティパッチが提供される。ユーザーが手動でダウンロードせずに最新バージョンに更新できる仕組みが必要である。状態マシンパターンで更新プロセスを管理する。

**機能の利用シーン**：
- 起動時の自動更新チェック
- 手動での更新確認（ヘルプメニュー）
- 更新のダウンロードと適用
- 再起動による更新インストール

**主要な処理内容**：
1. 更新チェック（checkForUpdates）
2. 更新ダウンロード（downloadUpdate）
3. 更新適用（applyUpdate）
4. 終了してインストール（quitAndInstall）
5. 状態遷移管理（State Machine）

**関連システム・外部連携**：
- 更新サーバー（update.code.visualstudio.com）
- Electron autoUpdater（win32/darwin）
- apt/snapパッケージマネージャ（Linux）

**権限による制御**：
- 管理者権限での実行時は更新無効化（Windows）
- 設定による更新の有効/無効制御

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | バックグラウンド処理（通知経由で表示） |

## 機能種別

更新管理 / 状態マシン / バックグラウンド処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| explicit | boolean | Yes(check) | 明示的チェックか | - |
| packagePath | string | Yes(apply) | パッケージパス | ファイル存在確認 |

### 入力データソース

- 更新サーバーからのレスポンス
- ユーザー操作（メニュー、通知）
- 自動更新タイマー

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| state | State | 現在の更新状態 |
| update | IUpdate | 更新情報（version, url等） |

### 出力先

- 状態変更イベント（onStateChange）
- 通知サービス経由の表示
- ステータスバー

## 処理フロー

### 処理シーケンス

```
1. 初期化
   └─ Uninitialized → Idle
2. 更新チェック
   └─ Idle → CheckingForUpdates
3. 更新発見
   └─ CheckingForUpdates → AvailableForDownload (Linux)
   └─ CheckingForUpdates → Downloading (win32/darwin)
4. ダウンロード完了
   └─ Downloading → Downloaded
5. 更新適用
   └─ Downloaded → Updating → Ready
6. 再起動後インストール
   └─ Ready → (restart) → 新バージョン起動
```

### フローチャート

```mermaid
flowchart TD
    A[Uninitialized] --> B[Idle]
    B --> C[CheckingForUpdates]
    C --> D{更新あり?}
    D -->|No| B
    D -->|Yes Linux| E[AvailableForDownload]
    D -->|Yes Win/Mac| F[Downloading]
    E --> F
    F --> G[Downloaded]
    G --> H[Updating]
    H --> I[Ready]
    I --> J[quitAndInstall]
    J --> K[再起動・インストール]

    B2[Disabled] -.-> B2

    subgraph エラー時
        C -->|Error| B
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-60-01 | 状態マシン | 定義された状態遷移に従う | 常時 |
| BR-60-02 | 管理者無効化 | 管理者実行時は更新無効 | Windows + Admin |
| BR-60-03 | 明示的チェック | explicit=trueで手動チェック | ユーザー操作時 |
| BR-60-04 | バージョン形式 | Win/Linux:CommitHash, Mac:ProductVersion | プラットフォーム別 |
| BR-60-05 | 無効化理由 | DisablementReasonで理由を記録 | Disabled状態時 |

### 計算ロジック

```typescript
// 更新状態の型定義
export const enum StateType {
  Uninitialized = 'uninitialized',
  Idle = 'idle',
  Disabled = 'disabled',
  CheckingForUpdates = 'checking for updates',
  AvailableForDownload = 'available for download',
  Downloading = 'downloading',
  Downloaded = 'downloaded',
  Updating = 'updating',
  Ready = 'ready',
}

// 更新タイプ
export const enum UpdateType {
  Setup,   // Windows Setup.exe
  Archive, // Archive/zip
  Snap     // Linux Snap
}

// 無効化理由
export const enum DisablementReason {
  NotBuilt,              // 更新機能未ビルド
  DisabledByEnvironment, // 環境変数による無効化
  ManuallyDisabled,      // 設定による無効化
  MissingConfiguration,  // 設定不足
  InvalidConfiguration,  // 設定不正
  RunningAsAdmin,        // 管理者実行
}

// 更新情報
interface IUpdate {
  version: string;       // CommitHash or ProductVersion
  productVersion?: string;
  timestamp?: number;
  url?: string;
  sha256hash?: string;
}
```

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（更新サーバーとの通信のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ネットワークエラー | 更新サーバー接続失敗 | Idle状態に戻り、error記録 |
| - | ダウンロードエラー | ダウンロード中断 | Idle状態に戻り、再試行可能 |
| - | 検証エラー | sha256hash不一致 | ダウンロード再試行 |
| - | 無効化 | 各種理由 | Disabled状態、理由表示 |

### リトライ仕様

自動更新チェックは定期的に再試行される

## トランザクション仕様

該当なし

## パフォーマンス要件

- 更新チェックはバックグラウンドで実行
- ダウンロードは低優先度で実行
- UIをブロックしない

## セキュリティ考慮事項

- HTTPS通信必須
- sha256hashによる検証
- 管理者権限時の更新制限
- 信頼された更新サーバーのみ使用

## 備考

- Windows: Squirrel/NuGetベースの更新
- macOS: Electron autoUpdater
- Linux: apt/snap/手動ダウンロード

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

更新関連の型定義と状態マシンを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | update.ts | `src/vs/platform/update/common/update.ts` | StateType、State、IUpdateService |

**読解のコツ**:
- 状態マシンパターンで更新プロセスを管理
- 各状態には関連データが含まれる（update情報など）

**主要処理フロー**:
1. **10-17行目**: IUpdate - 更新情報インターフェース
2. **19-35行目**: 状態マシンの説明コメント
3. **37-47行目**: StateType enum - 状態の種類
4. **49-53行目**: UpdateType enum - 更新タイプ
5. **55-62行目**: DisablementReason enum - 無効化理由
6. **64-74行目**: 各状態の型定義
7. **76-86行目**: State ファクトリ関数
8. **88-93行目**: IAutoUpdater - Electron autoUpdater
9. **95行目**: IUpdateService decorator登録
10. **97-110行目**: IUpdateService - サービスインターフェース

### プログラム呼び出し階層図

```
IUpdateService
    │
    ├─ state: State
    │      │
    │      ├─ Uninitialized
    │      ├─ Idle (updateType, error?)
    │      ├─ Disabled (reason: DisablementReason)
    │      ├─ CheckingForUpdates (explicit)
    │      ├─ AvailableForDownload (update: IUpdate)
    │      ├─ Downloading
    │      ├─ Downloaded (update: IUpdate)
    │      ├─ Updating (update: IUpdate)
    │      └─ Ready (update: IUpdate)
    │
    ├─ onStateChange: Event<State>
    │
    ├─ checkForUpdates(explicit)
    │      └─ Idle → CheckingForUpdates
    │
    ├─ downloadUpdate()
    │      └─ AvailableForDownload → Downloading → Downloaded
    │
    ├─ applyUpdate()
    │      └─ Downloaded → Updating → Ready
    │
    ├─ quitAndInstall()
    │      └─ Ready → アプリ終了・更新インストール
    │
    └─ isLatestVersion()
           └─ 最新バージョンか確認
```

### データフロー図

```
[入力]               [処理]                    [出力]

起動 / 手動チェック
    │
    ↓
┌───────────────┐
│Uninitialized  │
│(初期状態)      │
└───────────────┘
    │
    ↓
┌───────────────┐
│Idle           │
│(待機)         │
└───────────────┘
    │
    ↓
┌───────────────┐
│CheckingFor    │───→ onStateChange
│Updates        │
└───────────────┘
    │
    ├──────────────────┐
    ↓                  ↓
┌───────────┐   ┌────────────────┐
│No Update  │   │AvailableFor   │
│→ Idle     │   │Download (Linux)│
└───────────┘   └────────────────┘
                       │
                       ↓
                ┌───────────────┐
                │Downloading    │
                └───────────────┘
                       │
                       ↓
                ┌───────────────┐
                │Downloaded     │
                └───────────────┘
                       │
                       ↓
                ┌───────────────┐
                │Updating       │
                └───────────────┘
                       │
                       ↓
                ┌───────────────┐
                │Ready          │
                └───────────────┘
                       │
                       ↓
                [quitAndInstall]
                       │
                       ↓
                [再起動・更新完了]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| update.ts | `src/vs/platform/update/common/update.ts` | ソース | 共通型定義・インターフェース |
| updateService.ts | `src/vs/platform/update/electron-main/updateService.ts` | ソース | メインプロセス実装 |
| abstractUpdateService.ts | `src/vs/platform/update/electron-main/abstractUpdateService.ts` | ソース | 抽象基底クラス |
| updateService.darwin.ts | `src/vs/platform/update/electron-main/updateService.darwin.ts` | ソース | macOS固有実装 |
| updateService.win32.ts | `src/vs/platform/update/electron-main/updateService.win32.ts` | ソース | Windows固有実装 |
| updateService.linux.ts | `src/vs/platform/update/electron-main/updateService.linux.ts` | ソース | Linux固有実装 |
