# 機能設計書 81-タスク自動検出（Jake）

## 概要

本ドキュメントは、VS Codeの「タスク自動検出（Jake）」機能の設計内容を記載する。JakeはJavaScript用のビルドツールであり、本機能はJakefileからタスクを自動検出してVS Codeのタスクシステムに統合する。

### 本機能の処理概要

Jake拡張機能は、ワークスペース内のJakefileまたはJakefile.jsファイルを検出し、定義されているタスクを自動的にVS Codeのタスクリストに追加する機能を提供する。

**業務上の目的・背景**：開発者がJakeビルドツールを使用する際、手動でtasks.jsonにタスクを定義する手間を省き、プロジェクト内のJakeタスクを自動的に認識して実行可能にする。これにより、開発ワークフローの効率化と、プロジェクトへの参入障壁の低下を実現する。

**機能の利用シーン**：
- 新しいJakeプロジェクトをVS Codeで開いたとき、自動的にタスクが認識される
- タスクパレット（Ctrl+Shift+B等）からJakeタスクを直接実行する
- build、compile、watch、testなどのタスクを適切なタスクグループに分類して表示する

**主要な処理内容**：
1. ワークスペースフォルダの監視とJakefileの検出
2. `jake --tasks`コマンドによるタスク一覧の取得
3. タスク定義の解析と適切なタスクグループへの分類
4. VS Codeタスクプロバイダーへの登録

**関連システム・外部連携**：VS Codeタスクシステム、Node.js/Jake CLI

**権限による制御**：信頼されていないワークスペースでも動作可能（untrustedWorkspaces: supported）

## 関連画面

本機能は画面機能マッピングに明示的に記載されていないが、以下の画面で利用される。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | タスクパネル | 主画面 | Jakeタスクの一覧表示と実行 |
| - | コマンドパレット | 参照画面 | 「Tasks: Run Task」でJakeタスク選択 |

## 機能種別

タスク自動検出 / コマンド実行 / ファイル監視

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| jake.autoDetect | string | No | 自動検出の有効/無効（"on"/"off"） | enum: on, off |
| task | string | Yes（タスク実行時） | 実行するJakeタスク名 | - |
| file | string | No | Jakefileのパス | ファイル存在確認 |

### 入力データソース

- ワークスペースフォルダ内のJakefileまたはJakefile.js
- ワークスペース設定（jake.autoDetect）
- node_modules/.bin/jakeコマンド

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| tasks | vscode.Task[] | 検出されたJakeタスクの配列 |
| taskName | string | タスク名 |
| taskGroup | TaskGroup | Build/Testグループへの分類 |

### 出力先

VS Codeタスクシステム（vscode.tasks.registerTaskProvider）

## 処理フロー

### 処理シーケンス

```
1. 拡張機能のアクティベーション
   └─ onTaskType:jakeイベントで起動

2. TaskDetectorの初期化
   └─ ワークスペースフォルダごとにFolderDetectorを作成

3. Jakefileの検出
   └─ Jakefile または Jakefile.js の存在確認

4. Jakeコマンドの特定
   └─ ローカル（node_modules/.bin/jake）またはグローバルのjakeを使用

5. タスク一覧の取得
   └─ `jake --tasks` コマンドを実行し出力を解析

6. タスク定義の作成
   └─ 各タスクをvscode.Taskとして構築

7. タスクグループの分類
   └─ build/compile/watchを含むタスクはBuildグループ
   └─ testを含むタスクはTestグループ

8. タスクプロバイダーへの登録
   └─ vscode.tasks.registerTaskProvider('jake', ...)
```

### フローチャート

```mermaid
flowchart TD
    A[拡張機能起動] --> B{ワークスペースあり?}
    B -->|Yes| C[FolderDetector作成]
    B -->|No| Z[終了]
    C --> D{autoDetect=on?}
    D -->|Yes| E[ファイルウォッチャー開始]
    D -->|No| Z
    E --> F{Jakefile存在?}
    F -->|Yes| G[jake --tasks実行]
    F -->|No| Z
    G --> H{実行成功?}
    H -->|Yes| I[タスク一覧解析]
    H -->|No| J[エラー出力]
    I --> K[タスクグループ分類]
    K --> L[タスクプロバイダー登録]
    L --> Z
    J --> Z
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-81-1 | Buildタスク判定 | タスク名にbuild/compile/watchが含まれる場合Buildグループ | 常時 |
| BR-81-2 | Testタスク判定 | タスク名にtestが含まれる場合Testグループ | 常時 |
| BR-81-3 | Jakeコマンド優先順位 | ローカルインストール優先、なければグローバル | 常時 |

### 計算ロジック

タスク名からグループを判定するロジック：
```typescript
const buildNames = ['build', 'compile', 'watch'];
const testNames = ['test'];

function isBuildTask(name: string): boolean {
    return buildNames.some(n => name.indexOf(n) !== -1);
}

function isTestTask(name: string): boolean {
    return testNames.some(n => name.indexOf(n) !== -1);
}
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 警告 | jake --tasks実行失敗 | 出力チャンネルにエラー表示、ユーザーに通知 |
| - | 情報 | Jakefileが存在しない | 空のタスク配列を返却 |

### リトライ仕様

リトライは行わない。ファイル変更時に再検出が実行される。

## トランザクション仕様

該当なし

## パフォーマンス要件

- タスク一覧はキャッシュされ、Jakefileの変更時のみ再取得
- ファイルウォッチャーによる変更検知で効率的な更新

## セキュリティ考慮事項

- 信頼されていないワークスペースでもサポート（外部コマンド実行に注意）
- 仮想ワークスペースは非サポート（virtualWorkspaces: false）

## 備考

- Jake公式サイト: https://jakejs.com/
- Gulp/Grunt/npmタスク検出と同様のアーキテクチャ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | package.json | `extensions/jake/package.json` | タスク定義スキーマ（taskDefinitions）と設定項目 |

**読解のコツ**: `contributes.taskDefinitions`でjakeタスクの構造を定義している。`task`と`file`プロパティが使用される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | main.ts | `extensions/jake/src/main.ts` | activate関数とTaskDetectorの初期化 |

**主要処理フロー**:
1. **332-335行目**: activate関数でTaskDetectorを作成・開始
2. **337-339行目**: deactivate関数でTaskDetectorを破棄

#### Step 3: タスク検出ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | main.ts | `extensions/jake/src/main.ts` | TaskDetectorクラス（204-329行目） |
| 3-2 | main.ts | `extensions/jake/src/main.ts` | FolderDetectorクラス（85-202行目） |

**主要処理フロー**:
- **267-283行目**: updateProvider()でタスクプロバイダーを登録
- **133-194行目**: computeTasks()でjake --tasksを実行してタスク取得
- **32-50行目**: isBuildTask/isTestTaskでタスクグループを判定

### プログラム呼び出し階層図

```
activate() [332行目]
    │
    └─ TaskDetector.start() [212-219行目]
           │
           ├─ updateWorkspaceFolders() [229-245行目]
           │      │
           │      └─ FolderDetector() [90-93行目]
           │             └─ findJakeCommand() [67-78行目]
           │
           ├─ onDidChangeWorkspaceFolders [217行目]
           │
           └─ updateProvider() [267-283行目]
                  │
                  └─ registerTaskProvider('jake', ...) [270-277行目]
                         │
                         ├─ provideTasks() -> getTasks() [285-287行目]
                         │      └─ FolderDetector.getTasks() [111-120行目]
                         │             └─ computeTasks() [133-194行目]
                         │
                         └─ resolveTask() -> getTask() [311-328行目]
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Jakefile           ──▶ FolderDetector.computeTasks()
                            │
                            ▼
jake --tasks       ──▶ 正規表現解析 (161-163行目)
                            │
                            ▼
タスク名           ──▶ isBuildTask/isTestTask判定  ──▶ vscode.Task[]
                            │
                            ▼
                   TaskGroup.Build/Test設定 (173-177行目)
                            │
                            ▼
                   TaskProvider登録              ──▶ VS Codeタスクシステム
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.ts | `extensions/jake/src/main.ts` | ソース | メインロジック（タスク検出、プロバイダー登録） |
| package.json | `extensions/jake/package.json` | 設定 | 拡張機能マニフェスト、タスク定義スキーマ |
| package.nls.json | `extensions/jake/package.nls.json` | リソース | ローカライズ文字列 |
| tsconfig.json | `extensions/jake/tsconfig.json` | 設定 | TypeScriptコンパイラ設定 |
