# 画面設計書: エディタペイン

## 1. 基本情報

| 項目 | 内容 |
|------|------|
| 画面ID | 10 |
| 画面名 | エディタペイン |
| 画面パス | `src/vs/workbench/browser/parts/editor/editorPane.ts` |
| 概要 | 実際のエディタコンテンツを表示するコンテナ。テキストエディタ、差分エディタ、カスタムエディタなど様々なエディタタイプを表示する |

## 2. 画面構成

### 2.1 レイアウト概要

```
+------------------------------------------------------------------+
|                                                                   |
|                                                                   |
|                      エディタコンテンツ                            |
|                   (Monaco Editor 等)                              |
|                                                                   |
|                                                                   |
+------------------------------------------------------------------+
```

### 2.2 構成要素

| 要素名 | 説明 | 必須 | 可視性 |
|--------|------|------|--------|
| editorContainer | エディタのルートコンテナ | はい | 常時 |
| editorWidget | 実際のエディタウィジェット | はい | 常時 |
| overlayWidgets | オーバーレイウィジェット | いいえ | 条件依存 |
| scrollbar | スクロールバー | はい | コンテンツ依存 |

## 3. 機能仕様

### 3.1 エディタタイプ

| タイプ | クラス | 説明 |
|--------|--------|------|
| テキストエディタ | `TextEditor` | Monaco Editor を使用したテキスト編集 |
| 差分エディタ | `DiffEditor` | 2ファイルの差分表示 |
| カスタムエディタ | `CustomEditor` | 拡張機能提供のエディタ |
| ウェルカムエディタ | `WelcomeEditor` | ウェルカムページ |
| 設定エディタ | `SettingsEditor` | 設定UI |
| Notebookエディタ | `NotebookEditor` | Jupyterノートブック等 |

### 3.2 ライフサイクル

```
createEditor() -> setEditorVisible() -> layout() -> setInput() -> focus() -> dispose()
```

| メソッド | 説明 |
|----------|------|
| `createEditor()` | エディタウィジェットの作成 |
| `setEditorVisible()` | 表示/非表示の切り替え |
| `layout()` | サイズの調整 |
| `setInput()` | エディタ入力の設定 |
| `focus()` | フォーカスの設定 |
| `clearInput()` | 入力のクリア |
| `dispose()` | リソースの解放 |

### 3.3 状態管理

- **EditorMemento**: エディタ状態の保存/復元
- **ViewState**: スクロール位置、カーソル位置等
- **Dirty状態**: 未保存変更の追跡

### 3.4 サイズ制約

| 制約 | デフォルト値 |
|------|--------------|
| minimumWidth | 220px |
| maximumWidth | 無制限 |
| minimumHeight | 0px |
| maximumHeight | 無制限 |

## 4. データフロー

```
[EditorInput（ファイル情報）]
         |
         v
[EditorPane.setInput()]
         |
         v
[エディタモデルの作成/取得]
         |
         v
[エディタウィジェットへの適用]
         |
         v
[描画/表示]
```

## 5. イベント

| イベント名 | トリガー | 処理内容 |
|------------|----------|----------|
| onDidChangeSizeConstraints | サイズ制約変更 | レイアウト再計算 |
| onDidChangeControl | エディタ制御変更 | UI更新 |

## 6. コードリーディングガイド

### 6.1 読解の流れ

1. **エントリーポイント**: `editorPane.ts` の `EditorPane` クラス
2. **基底クラス**: `Composite` を継承
3. **インターフェース**: `IEditorPane` を実装
4. **具象クラス**: 各種エディタ（TextEditor, DiffEditor等）

### 6.2 プログラム呼び出し階層図

```
EditorPane (abstract)
  |
  +-- Composite (extends)
  |
  +-- IEditorPane (implements)
  |
  +-- 具象クラス
        |
        +-- TextDiffEditor
        +-- TextResourceEditor
        +-- SideBySideEditor
        +-- NotebookEditor
        +-- CustomEditor
        +-- SettingsEditor
        +-- WelcomeEditor
```

### 6.3 データフロー図

```
[EditorGroupView.showEditor()]
         |
         v
[EditorPane.create(parent)]
         |
         v
[EditorPane.createEditor(parent)]
         |
         v
[EditorPane.setInput(input, options)]
         |
         v
[具象クラスの入力処理]
         |
         v
[エディタコンテンツの表示]
```

### 6.4 関連ファイル一覧

| ファイルパス | 説明 |
|--------------|------|
| `src/vs/workbench/browser/parts/editor/editorPane.ts` | エディタペイン基底クラス（行1-432） |
| `src/vs/workbench/browser/composite.ts` | Composite基底クラス |
| `src/vs/workbench/common/editor.ts` | エディタ共通インターフェース |
| `src/vs/workbench/browser/parts/editor/textDiffEditor.ts` | 差分エディタ |
| `src/vs/workbench/browser/parts/editor/textResourceEditor.ts` | テキストエディタ |
| `src/vs/workbench/contrib/notebook/browser/notebookEditor.ts` | ノートブックエディタ |

### 6.5 重要なコードスニペット

**EditorPaneクラスの構造（editorPane.ts 行49-88）**
```typescript
export abstract class EditorPane<MementoType extends object = object>
  extends Composite<MementoType>
  implements IEditorPane {

  readonly onDidChangeSizeConstraints = Event.None;

  protected readonly _onDidChangeControl = this._register(new Emitter<void>());
  readonly onDidChangeControl = this._onDidChangeControl.event;

  get minimumWidth() { return DEFAULT_EDITOR_MIN_DIMENSIONS.width; }
  get maximumWidth() { return DEFAULT_EDITOR_MAX_DIMENSIONS.width; }
  get minimumHeight() { return DEFAULT_EDITOR_MIN_DIMENSIONS.height; }
  get maximumHeight() { return DEFAULT_EDITOR_MAX_DIMENSIONS.height; }

  protected _input: EditorInput | undefined;
  get input(): EditorInput | undefined { return this._input; }

  protected _options: IEditorOptions | undefined;
  get options(): IEditorOptions | undefined { return this._options; }

  constructor(
    id: string,
    readonly group: IEditorGroup,
    telemetryService: ITelemetryService,
    themeService: IThemeService,
    storageService: IStorageService
  ) {
    super(id, telemetryService, themeService, storageService);
  }
}
```

**抽象メソッド createEditor（editorPane.ts 行97-101）**
```typescript
/**
 * Called to create the editor in the parent HTMLElement. Subclasses implement
 * this method to construct the editor widget.
 */
protected abstract createEditor(parent: HTMLElement): void;
```

**入力設定（editorPane.ts 行116-119）**
```typescript
async setInput(input: EditorInput, options: IEditorOptions | undefined, context: IEditorOpenContext, token: CancellationToken): Promise<void> {
  this._input = input;
  this._options = options;
}
```

**EditorMemento（editorPane.ts 行210-432）**
```typescript
export class EditorMemento<T> extends Disposable implements IEditorMemento<T> {
  private cache: LRUCache<string, MapGroupToMemento<T>> | undefined;

  saveEditorState(group: IEditorGroup, resource: URI, state: T): void {
    // 状態の保存
  }

  loadEditorState(group: IEditorGroup, resource: URI): T | undefined {
    // 状態の読み込み
  }

  clearEditorState(resource: URI, group?: IEditorGroup): void {
    // 状態のクリア
  }
}
```

## 7. 画面遷移

### 7.1 遷移元

| 遷移元 | 条件 | 説明 |
|--------|------|------|
| エディタグループ | エディタを開く | EditorPane作成/表示 |
| タブ | タブ切り替え | 表示エディタの切り替え |

### 7.2 遷移先

| 遷移先 | 条件 | 説明 |
|--------|------|------|
| 定義箇所 | Go to Definition | 他ファイル/位置へ遷移 |
| 参照箇所 | Find References | 参照一覧へ遷移 |

## 8. スタイリング

### 8.1 テーマカラー

| カラーID | 説明 |
|----------|------|
| `editor.background` | エディタ背景色 |
| `editor.foreground` | エディタ前景色 |
| `editorCursor.foreground` | カーソル色 |
| `editor.selectionBackground` | 選択範囲背景色 |
| `editor.lineHighlightBackground` | 現在行背景色 |

### 8.2 CSSクラス

| クラス名 | 説明 |
|----------|------|
| `.editor-instance` | エディタインスタンス |
| `.monaco-editor` | Monaco Editorコンテナ |
| `.overflow-guard` | オーバーフローガード |
| `.view-lines` | 行表示エリア |

## 9. 状態の保存と復元

### 9.1 EditorMemento

- **LRUCache**: 最近使用したエディタの状態をキャッシュ
- **グループ単位**: エディタグループごとに状態を管理
- **リソースキー**: URIをキーとして状態を保存

### 9.2 保存される状態

| 状態 | 説明 |
|------|------|
| スクロール位置 | 縦/横スクロール位置 |
| カーソル位置 | 行/列の位置 |
| 選択範囲 | 選択開始/終了位置 |
| 折りたたみ状態 | コード折りたたみ情報 |

## 10. アクセシビリティ

- **キーボードナビゲーション**: 全操作がキーボードで可能
- **スクリーンリーダー対応**: ARIA属性の適切な設定
- **ハイコントラストモード**: 視認性向上のテーマ対応
- **音声読み上げ**: 行内容の読み上げ

## 11. 拡張ポイント

### 11.1 カスタムエディタ

拡張機能から`vscode.window.registerCustomEditorProvider`でカスタムエディタを登録可能

```typescript
vscode.window.registerCustomEditorProvider(
  'myExtension.customEditor',
  new MyCustomEditorProvider(),
  {
    webviewOptions: { retainContextWhenHidden: true },
    supportsMultipleEditorsPerDocument: false
  }
);
```

### 11.2 エディタ装飾

`vscode.TextEditor.setDecorations`でエディタ内に装飾を追加可能
