# 画面設計書: タイトルバー

## 1. 基本情報

| 項目 | 内容 |
|------|------|
| 画面ID | 2 |
| 画面名 | タイトルバー |
| 画面パス | `src/vs/workbench/browser/parts/titlebar/titlebarPart.ts` |
| 概要 | ウィンドウの最上部に表示されるタイトルバー。ウィンドウタイトル、メニューバー、コマンドセンター、ウィンドウ制御ボタンを含む |

## 2. 画面構成

### 2.1 レイアウト概要

```
+------------------------------------------------------------------+
| [AppIcon] [Menubar] | [Command Center / Title] | [Actions] [WCB] |
+------------------------------------------------------------------+
  leftContent           centerContent              rightContent
```

- **AppIcon**: アプリケーションアイコン（Windows/Linux）
- **Menubar**: カスタムメニューバー（非ネイティブ時）
- **Command Center**: コマンドパレット起動ボタン/ウィンドウタイトル
- **Actions**: ツールバーアクション
- **WCB**: ウィンドウ制御ボタン（最小化/最大化/閉じる）

### 2.2 構成要素

| 要素名 | 説明 | 必須 | 可視性 |
|--------|------|------|--------|
| rootContainer | タイトルバーのルートコンテナ | はい | 常時 |
| dragRegion | ウィンドウドラッグ用の領域 | はい | 常時 |
| leftContent | 左側コンテンツエリア | はい | 常時 |
| centerContent | 中央コンテンツエリア | はい | 常時 |
| rightContent | 右側コンテンツエリア | はい | 常時 |
| appIcon | アプリケーションアイコン | いいえ | Windows/Linux |
| menubar | カスタムメニューバー | いいえ | 設定依存 |
| title | ウィンドウタイトル | はい | 設定依存 |
| commandCenter | コマンドセンター | いいえ | 設定依存 |
| actionToolBar | アクションツールバー | いいえ | 設定依存 |
| windowControlsContainer | ウィンドウ制御ボタン | いいえ | 設定依存 |

## 3. 機能仕様

### 3.1 ウィンドウタイトル表示

- **WindowTitle**: ウィンドウタイトルの管理クラス
- **タイトル変数**: カスタム変数を使用したタイトルテンプレート
- **動的更新**: ファイル変更、ダーティ状態に応じてタイトル更新

### 3.2 コマンドセンター

- **クイックアクセス**: コマンドパレットへのワンクリックアクセス
- **検索機能**: ファイル、コマンド、シンボルの統合検索
- **設定**: `workbench.commandCenter` で有効/無効切り替え

### 3.3 メニューバー

- **CustomMenubarControl**: カスタムメニューバーの実装
- **表示モード**: visible, toggle, compact, hidden
- **キーボードナビゲーション**: Alt/F10でメニューアクセス

### 3.4 ウィンドウ制御

- **最小化/最大化/閉じる**: 標準ウィンドウ操作
- **カスタムタイトルバー**: ネイティブタイトルバーの代替
- **WCO（Window Controls Overlay）**: PWA向けのオーバーレイ制御

### 3.5 アクティビティアクション

- **グローバルアクティビティ**: 設定、アカウント管理
- **エディタアクション**: エディタ固有の操作（タイトルバー配置時）
- **レイアウト制御**: レイアウト操作アクション

## 4. データフロー

```
[ファイル変更/状態変更]
         |
         v
[WindowTitle.updateTitle()]
         |
         v
[onDidChange イベント]
         |
         v
[タイトル要素の更新]
```

## 5. イベント

| イベント名 | トリガー | 処理内容 |
|------------|----------|----------|
| onMenubarVisibilityChange | メニューバー表示変更 | レイアウト再計算 |
| onDidChange | タイトル/サイズ変更 | UI更新 |
| onWillDispose | パーツ破棄前 | クリーンアップ |

## 6. 関連設定

| 設定キー | 説明 | デフォルト値 |
|----------|------|--------------|
| `window.titleBarStyle` | タイトルバースタイル | プラットフォーム依存 |
| `window.customTitleBarVisibility` | カスタムタイトルバー表示条件 | `auto` |
| `workbench.commandCenter` | コマンドセンター表示 | `true` |
| `window.menuBarVisibility` | メニューバー表示モード | `classic` |
| `workbench.layoutControl.enabled` | レイアウトコントロール | `true` |
| `workbench.editor.editorActionsLocation` | エディタアクション位置 | `default` |

## 7. コードリーディングガイド

### 7.1 読解の流れ

1. **エントリーポイント**: `titlebarPart.ts` の `BrowserTitlebarPart` クラス
2. **サービス登録**: `BrowserTitleService` でマルチウィンドウ対応
3. **タイトル管理**: `windowTitle.ts` の `WindowTitle` クラス
4. **メニューバー**: `menubarControl.ts` の `CustomMenubarControl`

### 7.2 プログラム呼び出し階層図

```
BrowserTitleService
  |
  +-- BrowserTitlebarPart (main window)
  |     |
  |     +-- createContentArea()
  |     |     +-- leftContent, centerContent, rightContent
  |     |     +-- installMenubar()
  |     |     +-- createTitle()
  |     |     +-- createActionToolBar()
  |     |
  |     +-- WindowTitle
  |     +-- CustomMenubarControl
  |     +-- CommandCenterControl
  |
  +-- AuxiliaryBrowserTitlebarPart (aux windows)
```

### 7.3 データフロー図

```
[User Action: File Open/Save]
         |
         v
[EditorService.onDidActiveEditorChange]
         |
         v
[WindowTitle.onActiveEditorChange]
         |
         v
[WindowTitle.updateTitle()]
         |
         v
[title.textContent = newTitle]
```

### 7.4 関連ファイル一覧

| ファイルパス | 説明 |
|--------------|------|
| `src/vs/workbench/browser/parts/titlebar/titlebarPart.ts` | メインのタイトルバーパーツ（行1-962） |
| `src/vs/workbench/browser/parts/titlebar/windowTitle.ts` | ウィンドウタイトル管理 |
| `src/vs/workbench/browser/parts/titlebar/menubarControl.ts` | カスタムメニューバー |
| `src/vs/workbench/browser/parts/titlebar/commandCenterControl.ts` | コマンドセンター |
| `src/vs/workbench/browser/parts/titlebar/titlebarActions.ts` | タイトルバーアクション |
| `src/vs/workbench/browser/parts/titlebar/media/titlebarpart.css` | スタイル定義 |

### 7.5 重要なコードスニペット

**タイトルバー構造の作成（titlebarPart.ts 行449-555）**
```typescript
protected override createContentArea(parent: HTMLElement): HTMLElement {
  this.element = parent;
  this.rootContainer = append(parent, $('.titlebar-container'));

  this.leftContent = append(this.rootContainer, $('.titlebar-left'));
  this.centerContent = append(this.rootContainer, $('.titlebar-center'));
  this.rightContent = append(this.rootContainer, $('.titlebar-right'));

  // App Icon (Windows, Linux)
  if ((isWindows || isLinux) && !hasNativeTitlebar(...)) {
    this.appIcon = prepend(this.leftContent, $('a.window-appicon'));
  }

  // Draggable region
  this.dragRegion = prepend(this.rootContainer, $('div.titlebar-drag-region'));

  // ...
}
```

**メニューバー表示制御（titlebarPart.ts 行407-429）**
```typescript
protected installMenubar(): void {
  if (this.menubar) {
    return; // If the menubar is already installed, skip
  }

  this.customMenubar.value = this.instantiationService.createInstance(CustomMenubarControl);
  this.menubar = append(this.leftContent, $('div.menubar'));
  this.menubar.setAttribute('role', 'menubar');

  this._register(this.customMenubar.value.onVisibilityChange(e => this.onMenubarVisibilityChanged(e)));
  this.customMenubar.value.create(this.menubar);
}
```

**高さ計算（titlebarPart.ts 行227-238）**
```typescript
get minimumHeight(): number {
  const wcoEnabled = isWeb && isWCOEnabled();
  let value = this.isCommandCenterVisible || wcoEnabled ? DEFAULT_CUSTOM_TITLEBAR_HEIGHT : 30;
  if (wcoEnabled) {
    value = Math.max(value, getWCOTitlebarAreaRect(getWindow(this.element))?.height ?? 0);
  }
  return value / (this.preventZoom ? getZoomFactor(getWindow(this.element)) : 1);
}
```

## 8. 画面遷移

### 8.1 遷移元

| 遷移元 | 条件 | 説明 |
|--------|------|------|
| ワークベンチ | 常時 | ワークベンチの一部として表示 |

### 8.2 遷移先

| 遷移先 | 条件 | 説明 |
|--------|------|------|
| コマンドパレット | コマンドセンタークリック | クイック入力を開く |
| メニュー | メニューバークリック | ドロップダウンメニュー表示 |

## 9. スタイリング

### 9.1 テーマカラー

| カラーID | 説明 |
|----------|------|
| `titleBar.activeBackground` | アクティブ時の背景色 |
| `titleBar.activeForeground` | アクティブ時の前景色 |
| `titleBar.inactiveBackground` | 非アクティブ時の背景色 |
| `titleBar.inactiveForeground` | 非アクティブ時の前景色 |
| `titleBar.border` | 下部ボーダー色 |

### 9.2 CSSクラス

| クラス名 | 説明 |
|----------|------|
| `.titlebar-container` | メインコンテナ |
| `.titlebar-left/center/right` | コンテンツエリア |
| `.titlebar-drag-region` | ドラッグ可能領域 |
| `.window-appicon` | アプリアイコン |
| `.menubar` | メニューバー |

## 10. プラットフォーム固有の動作

| プラットフォーム | 動作の違い |
|------------------|------------|
| macOS | ネイティブタイトルバー使用、トラフィックライト左配置 |
| Windows | カスタムタイトルバー、アプリアイコン表示 |
| Linux | カスタムタイトルバー、アプリアイコン表示 |
| Web | WCO対応、カスタムタイトルバー |

## 11. アクセシビリティ

- **キーボードナビゲーション**: Tab/矢印キーでメニュー操作
- **ARIAロール**: menubar, menuitem等の適切な設定
- **フォーカス管理**: `workbench.action.focusTitleBar`コマンド
