# 画面設計書: バナー

## 1. 基本情報

| 項目 | 内容 |
|------|------|
| 画面ID | 8 |
| 画面名 | バナー |
| 画面パス | `src/vs/workbench/browser/parts/banner/bannerPart.ts` |
| 概要 | ワークベンチのタイトルバー直下に配置される通知バナー。重要な情報やアクションを促すメッセージを表示する |

## 2. 画面構成

### 2.1 レイアウト概要

```
+------------------------------------------------------------------+
| [アイコン] [メッセージテキスト] [アクションリンク] [×閉じる]         |
+------------------------------------------------------------------+
```

### 2.2 構成要素

| 要素名 | 説明 | 必須 | 可視性 |
|--------|------|------|--------|
| icon | 通知アイコン | いいえ | 条件依存 |
| messageElement | メッセージテキスト | はい | 常時 |
| messageActionsContainer | アクションボタン/リンク | いいえ | 条件依存 |
| actionBar | クローズボタン等 | はい | 常時 |

## 3. 機能仕様

### 3.1 バナー表示

- **一時的表示**: 特定の条件で自動的に表示
- **永続表示**: 明示的に閉じるまで表示継続
- **優先度**: 複数バナーがある場合の表示順序

### 3.2 バナータイプ

| タイプ | 説明 | 用途例 |
|--------|------|--------|
| info | 情報通知 | 機能のお知らせ |
| warning | 警告 | 非推奨機能の案内 |
| error | エラー | 重大な問題の通知 |

### 3.3 アクション

- **リンクアクション**: 外部URL/内部コマンドへの誘導
- **ボタンアクション**: 直接実行可能な操作
- **クローズ**: バナーを閉じる

### 3.4 表示条件

- **条件付き表示**: 特定の状態でのみ表示
- **一度限り**: 一度閉じたら再表示しない
- **セッション**: セッションごとに表示

## 4. データフロー

```
[BannerService.show()]
         |
         v
[BannerPart.setVisibility(true)]
         |
         v
[バナーコンテンツの描画]
         |
         v
[ユーザー操作（クローズ/アクション）]
         |
         v
[BannerPart.setVisibility(false)]
```

## 5. イベント

| イベント名 | トリガー | 処理内容 |
|------------|----------|----------|
| onDidChangeVisibility | 表示/非表示変更 | レイアウト更新 |

## 6. 関連設定

バナー固有の設定はありません。バナーの表示はプログラム的に制御されます。

## 7. コードリーディングガイド

### 7.1 読解の流れ

1. **エントリーポイント**: `bannerPart.ts` の `BannerPart` クラス
2. **サービス**: `IBannerService` でバナー表示を制御
3. **インターフェース**: `IBannerItem` でバナー内容を定義

### 7.2 プログラム呼び出し階層図

```
BannerPart
  |
  +-- Part (extends)
  |
  +-- IBannerService (implements)
  |     |
  |     +-- show()
  |     +-- hide()
  |
  +-- ActionBar
        +-- クローズアクション
        +-- カスタムアクション
```

### 7.3 データフロー図

```
[拡張機能/コア機能]
         |
         v
[BannerService.show(item)]
         |
         v
[BannerPart.toDispose.clear()]
         |
         v
[アイコン/メッセージ/アクションの描画]
         |
         v
[レイアウト更新通知]
```

### 7.4 関連ファイル一覧

| ファイルパス | 説明 |
|--------------|------|
| `src/vs/workbench/browser/parts/banner/bannerPart.ts` | バナーパーツ本体 |
| `src/vs/workbench/services/banner/browser/bannerService.ts` | バナーサービス |
| `src/vs/platform/banner/common/banner.ts` | バナーインターフェース |
| `src/vs/workbench/browser/parts/banner/media/bannerpart.css` | スタイル定義 |

### 7.5 重要なコードスニペット

**BannerPartクラスの構造（bannerPart.ts）**
```typescript
export class BannerPart extends Part implements IBannerService {
  readonly minimumWidth: number = 0;
  readonly maximumWidth: number = Number.POSITIVE_INFINITY;

  get minimumHeight(): number {
    return this.visible ? 26 : 0;
  }
  get maximumHeight(): number {
    return this.visible ? 26 : 0;
  }

  private visible = false;
  private item: IBannerItem | undefined;

  constructor(
    @IThemeService themeService: IThemeService,
    @IStorageService storageService: IStorageService,
    // ...
  ) {
    super(Parts.BANNER_PART, { hasTitle: false }, themeService, storageService, layoutService);
  }
}
```

**バナー表示メソッド**
```typescript
show(item: IBannerItem): void {
  if (this.item && this.item.id === item.id) {
    return; // 同じバナーは再表示しない
  }

  this.item = item;
  this.visible = true;

  // コンテンツの描画
  this.renderContent();

  // レイアウト更新を通知
  this._onDidChangeSize.fire(undefined);
}
```

**IBannerItemインターフェース**
```typescript
interface IBannerItem {
  id: string;
  icon?: ThemeIcon;
  message: string;
  actions?: IAction[];
  ariaLabel?: string;
  onClose?: () => void;
}
```

## 8. 画面遷移

### 8.1 遷移元

| 遷移元 | 条件 | 説明 |
|--------|------|------|
| システム | 条件発生時 | 自動表示 |
| 拡張機能 | API呼び出し | プログラム的表示 |

### 8.2 遷移先

| 遷移先 | 条件 | 説明 |
|--------|------|------|
| 外部URL | リンククリック | ドキュメント等へ誘導 |
| コマンド実行 | アクションクリック | 設定変更等 |

## 9. スタイリング

### 9.1 テーマカラー

| カラーID | 説明 |
|----------|------|
| `banner.background` | 背景色 |
| `banner.foreground` | 前景色 |
| `banner.iconForeground` | アイコン色 |

### 9.2 CSSクラス

| クラス名 | 説明 |
|----------|------|
| `.part.banner` | バナーコンテナ |
| `.banner-message` | メッセージエリア |
| `.banner-actions` | アクションエリア |
| `.banner-icon` | アイコン |

## 10. バナーの高さ

| 状態 | 高さ |
|------|------|
| 表示時 | 26px |
| 非表示時 | 0px |

## 11. アクセシビリティ

- **ARIAロール**: alertまたはstatus
- **ARIAラベル**: バナー内容の説明
- **キーボード操作**: Tab/Enterでアクション実行
- **スクリーンリーダー**: メッセージとアクションの読み上げ

## 12. 使用例

### 12.1 制限モードバナー

```typescript
bannerService.show({
  id: 'workbench.banner.restrictedMode',
  icon: Codicon.warning,
  message: localize('restrictedMode', 'You are in Restricted Mode...'),
  actions: [
    {
      id: 'workbench.action.manageWorkspaceTrust',
      label: localize('manageWorkspaceTrust', 'Manage'),
      run: () => { /* ... */ }
    }
  ]
});
```

### 12.2 更新通知バナー

```typescript
bannerService.show({
  id: 'workbench.banner.updateAvailable',
  icon: Codicon.info,
  message: localize('updateAvailable', 'A new version is available...'),
  actions: [
    {
      id: 'workbench.action.update',
      label: localize('restartToUpdate', 'Restart to Update'),
      run: () => { /* ... */ }
    }
  ]
});
```
