# 画面設計書: エディタグループ

## 1. 基本情報

| 項目 | 内容 |
|------|------|
| 画面ID | 9 |
| 画面名 | エディタグループ |
| 画面パス | `src/vs/workbench/browser/parts/editor/editorGroupView.ts` |
| 概要 | 複数のエディタタブを管理するコンテナ。エディタの分割表示やタブ管理を担当する |

## 2. 画面構成

### 2.1 レイアウト概要

```
+------------------------------------------------------------------+
| [タブバー: file1.ts | file2.ts | file3.ts ...]  [アクション]      |
+------------------------------------------------------------------+
|                                                                   |
|                                                                   |
|                         エディタペイン                            |
|                    (実際のエディタコンテンツ)                      |
|                                                                   |
|                                                                   |
+------------------------------------------------------------------+
```

### 2.2 構成要素

| 要素名 | 説明 | 必須 | 可視性 |
|--------|------|------|--------|
| titleContainer | タイトル/タブエリア | はい | 設定依存 |
| tabsContainer | タブバー | いいえ | 設定依存 |
| breadcrumbsContainer | パンくずリスト | いいえ | 設定依存 |
| editorContainer | エディタコンテナ | はい | 常時 |
| editorPane | 現在のエディタ | はい | 常時 |

## 3. 機能仕様

### 3.1 エディタ管理

- **複数エディタ**: 1グループに複数のエディタをタブで管理
- **アクティブエディタ**: 現在表示中のエディタ
- **ピン留め**: エディタのピン留め（自動クローズ防止）
- **プレビューモード**: 一時的なエディタ表示

### 3.2 タブ表示モード

| モード | 設定値 | 説明 |
|--------|--------|------|
| 複数タブ | `multiple` | 全エディタをタブ表示 |
| 単一タブ | `single` | アクティブエディタのみ表示 |
| タブなし | `none` | タブバー非表示 |

### 3.3 グループ操作

- **分割**: 縦/横にグループを分割
- **移動**: エディタを他グループに移動
- **グループ閉じ**: グループ内の全エディタを閉じる
- **グループロック**: グループのロック

### 3.4 エディタ順序

| 順序 | 説明 |
|------|------|
| 開いた順 | 開いた順序で並べる |
| アルファベット順 | ファイル名順 |
| 最近使用順 | 最近アクセスした順 |

## 4. データフロー

```
[エディタを開く操作]
         |
         v
[EditorGroupView.openEditor()]
         |
         v
[EditorInput の作成/取得]
         |
         v
[EditorPane の作成/表示]
         |
         v
[タブの追加/更新]
```

## 5. イベント

| イベント名 | トリガー | 処理内容 |
|------------|----------|----------|
| onDidModelChange | エディタ追加/削除 | タブバー更新 |
| onDidActiveEditorChange | アクティブエディタ変更 | エディタペイン切り替え |
| onDidCloseEditor | エディタクローズ | リソース解放 |
| onDidGroupChange | グループ状態変更 | UI更新 |

## 6. 関連設定

| 設定キー | 説明 | デフォルト値 |
|----------|------|--------------|
| `workbench.editor.showTabs` | タブ表示モード | `multiple` |
| `workbench.editor.tabCloseButton` | 閉じるボタン位置 | `right` |
| `workbench.editor.tabSizing` | タブサイズ | `fit` |
| `workbench.editor.pinnedTabSizing` | ピン留めタブサイズ | `normal` |
| `workbench.editor.wrapTabs` | タブの折り返し | `false` |
| `workbench.editor.enablePreview` | プレビューモード | `true` |
| `workbench.editor.openPositioning` | 新規タブ位置 | `right` |

## 7. コードリーディングガイド

### 7.1 読解の流れ

1. **エントリーポイント**: `editorGroupView.ts` の `EditorGroupView` クラス
2. **タブ管理**: `EditorTabsControl` でタブバーを管理
3. **エディタペイン**: `EditorPane` で個別エディタを管理
4. **サービス**: `IEditorGroupsService` でグループ全体を管理

### 7.2 プログラム呼び出し階層図

```
EditorPart
  |
  +-- EditorGroupView (multiple)
  |     |
  |     +-- EditorTabsControl
  |     |     +-- タブバーの描画/操作
  |     |
  |     +-- BreadcrumbsControl
  |     |     +-- パンくずリスト
  |     |
  |     +-- EditorPane (active)
  |           +-- エディタコンテンツ
  |
  +-- EditorGroupsService
        +-- グループ管理
```

### 7.3 データフロー図

```
[ファイルダブルクリック / コマンド実行]
         |
         v
[EditorService.openEditor(input, options)]
         |
         v
[EditorGroupsService.activeGroup.openEditor()]
         |
         v
[EditorGroupView.doOpenEditor()]
         |
         v
[EditorInput解決 -> EditorPane作成 -> 表示]
```

### 7.4 関連ファイル一覧

| ファイルパス | 説明 |
|--------------|------|
| `src/vs/workbench/browser/parts/editor/editorGroupView.ts` | エディタグループビュー |
| `src/vs/workbench/browser/parts/editor/editorTabsControl.ts` | タブ制御 |
| `src/vs/workbench/browser/parts/editor/editorPane.ts` | エディタペイン基底 |
| `src/vs/workbench/browser/parts/editor/editorPart.ts` | エディタパート全体 |
| `src/vs/workbench/services/editor/common/editorGroupsService.ts` | グループサービス |
| `src/vs/workbench/common/editor.ts` | エディタ共通定義 |

### 7.5 重要なコードスニペット

**EditorGroupViewクラスの構造（editorGroupView.ts）**
```typescript
export class EditorGroupView extends Disposable implements IEditorGroupView {
  readonly id: GroupIdentifier;

  private readonly _group: EditorGroup;
  private readonly editorPane: EditorPane | undefined;
  private readonly tabsControl: EditorTabsControl;

  constructor(
    accessor: IEditorGroupsAccessor,
    group: EditorGroup | ISerializedEditorGroupModel,
    @IInstantiationService instantiationService: IInstantiationService,
    // ...
  ) {
    // グループの初期化
  }

  async openEditor(input: EditorInput, options?: IEditorOptions): Promise<IEditorPane | undefined> {
    // エディタを開く処理
  }

  closeEditor(input: EditorInput): Promise<void> {
    // エディタを閉じる処理
  }
}
```

**エディタを開く処理**
```typescript
async openEditor(input: EditorInput, options?: IEditorOptions): Promise<IEditorPane | undefined> {
  // 入力の検証
  if (!input) {
    return undefined;
  }

  // 既存エディタの検索
  const existingEditor = this.group.contains(input);

  // エディタの表示
  return this.doShowEditor(input, options, existingEditor);
}
```

## 8. 画面遷移

### 8.1 遷移元

| 遷移元 | 条件 | 説明 |
|--------|------|------|
| エクスプローラ | ファイル選択 | ファイルを開く |
| 検索結果 | 結果クリック | 該当ファイルを開く |
| クイックオープン | ファイル選択 | ファイルを開く |
| コマンド | 新規ファイル等 | 新しいエディタを開く |

### 8.2 遷移先

| 遷移先 | 条件 | 説明 |
|--------|------|------|
| 別グループ | ドラッグ&ドロップ | エディタの移動 |
| 定義箇所 | Go to Definition | 他ファイルへ遷移 |

## 9. スタイリング

### 9.1 テーマカラー

| カラーID | 説明 |
|----------|------|
| `editorGroupHeader.tabsBackground` | タブバー背景 |
| `editorGroupHeader.tabsBorder` | タブバーボーダー |
| `editorGroupHeader.noTabsBackground` | タブなし時背景 |
| `editorGroup.border` | グループ間ボーダー |
| `editorGroup.dropBackground` | ドロップ時背景 |
| `tab.activeBackground` | アクティブタブ背景 |
| `tab.activeForeground` | アクティブタブ前景 |
| `tab.inactiveBackground` | 非アクティブタブ背景 |
| `tab.inactiveForeground` | 非アクティブタブ前景 |
| `tab.border` | タブボーダー |

### 9.2 CSSクラス

| クラス名 | 説明 |
|----------|------|
| `.editor-group-container` | グループコンテナ |
| `.title` | タイトルエリア |
| `.tabs-container` | タブコンテナ |
| `.tab` | 個別タブ |
| `.editor-container` | エディタコンテナ |

## 10. グループ操作コマンド

| コマンドID | 説明 |
|------------|------|
| `workbench.action.splitEditor` | エディタを分割 |
| `workbench.action.splitEditorRight` | 右に分割 |
| `workbench.action.splitEditorDown` | 下に分割 |
| `workbench.action.moveEditorToNextGroup` | 次グループへ移動 |
| `workbench.action.closeActiveEditor` | アクティブエディタを閉じる |
| `workbench.action.closeAllEditors` | 全エディタを閉じる |

## 11. アクセシビリティ

- **キーボードナビゲーション**: Ctrl+Tab/Ctrl+PageUp/Downでタブ切り替え
- **フォーカス管理**: タブとエディタ間の適切なフォーカス移動
- **ARIAロール**: tablist, tab, tabpanelの適切な設定
- **スクリーンリーダー**: ファイル名とダーティ状態の読み上げ

## 12. グループレイアウト

### 12.1 分割パターン

```
+-------+-------+     +---------------+
|       |       |     |               |
|   1   |   2   |     |       1       |
|       |       |     +-------+-------+
+-------+-------+     |   2   |   3   |
                      +-------+-------+
  横2分割                縦横混合
```

### 12.2 グリッドレイアウト

EditorPartは`SerializableGrid`を使用してグループを配置
