# Zig Programming Language

## 概要

Zigは、堅牢性、最適化、再利用可能なソフトウェアを維持するための汎用プログラミング言語およびツールチェーンです。C/C++との相互運用性を重視し、LLVMをバックエンドとして使用することで、高性能なネイティブコードを生成します。自己ホスト型のコンパイラであり、Zig言語自身で書かれています。

## 主な機能

| 機能カテゴリ | 説明 |
| --- | --- |
| コンパイラ | 自己ホスト型のZigコンパイラ。LLVMバックエンド対応および非LLVMビルドをサポート |
| 標準ライブラリ | 包括的な標準ライブラリ（std）を提供。各種データ構造、I/O、暗号化等を含む |
| ビルドシステム | `build.zig`によるネイティブなZigビルドシステム。CMakeにも対応 |
| C/C++統合 | `@cImport`によるCヘッダファイルの直接インポート。translate-c機能でCコードをZigに変換 |
| クロスコンパイル | 多数のターゲットアーキテクチャ・OSへのクロスコンパイルをサポート |
| Autodoc | インタラクティブなドキュメント生成システム。WebAssemblyベースのブラウザアプリケーション |
| テストフレームワーク | 組み込みのテストフレームワーク。behavior、standalone、link等の各種テストを実行可能 |
| デバッグサポート | LLDB/GDB用のpretty printerスクリプトを提供 |

## 技術スタック

### コア

- 言語: Zig (v0.16.0-dev)
- バックエンド: LLVM 21.x（オプション）
- ビルドシステム: Zig Build System / CMake >= 3.15
- C/C++ツールチェーン: Clang, LLD

### 主要ライブラリ

- compiler_rt: コンパイラランタイムライブラリ
- libc: 各種Cランタイムライブラリのサポート（glibc、musl、mingw等）
- libcxx / libcxxabi: C++標準ライブラリ
- libunwind: スタックアンワインドライブラリ
- libtsan: Thread Sanitizer

## ディレクトリ構成

```
zig/
├── build.zig              # Zigビルドシステム設定
├── build.zig.zon          # パッケージマニフェスト
├── bootstrap.c            # LLVM無しブートストラップ用Cソース
├── CMakeLists.txt         # CMakeビルド設定
├── LICENSE                # MITライセンス
├── README.md              # 公式README
├── ci/                    # CI/CDスクリプト
├── cmake/                 # CMakeモジュール
├── doc/                   # ドキュメント
├── lib/                   # ライブラリファイル
│   ├── std/               # 標準ライブラリ
│   ├── compiler/          # コンパイラ内部モジュール
│   ├── compiler_rt/       # コンパイラランタイム
│   ├── docs/              # Autodocシステム
│   ├── libc/              # Cライブラリ
│   ├── libcxx/            # C++標準ライブラリ
│   ├── libcxxabi/         # C++ ABI
│   ├── libtsan/           # Thread Sanitizer
│   └── libunwind/         # スタックアンワインド
├── src/                   # コンパイラソースコード
│   ├── main.zig           # メインエントリポイント
│   ├── Compilation.zig    # コンパイル処理
│   ├── Sema.zig           # 意味解析
│   ├── codegen.zig        # コード生成
│   ├── link.zig           # リンカ
│   ├── InternPool.zig     # インターンプール
│   ├── Air.zig            # 中間表現(AIR)
│   ├── Type.zig           # 型システム
│   └── Value.zig          # 値表現
├── stage1/                # Stage1コンパイラ（レガシー）
├── test/                  # テストスイート
│   ├── behavior/          # 動作テスト
│   ├── standalone/        # スタンドアロンテスト
│   ├── link/              # リンカテスト
│   └── cases/             # テストケース
└── tools/                 # 開発ツール
    ├── docgen.zig         # ドキュメント生成
    ├── doctest.zig        # ドキュメントテスト
    ├── update_cpu_features.zig  # CPU機能更新
    └── lldb_pretty_printers.py  # LLDBデバッグサポート
```

## セットアップ

### 必要条件

- CMake >= 3.15
- システムC/C++ツールチェーン
- LLVM, Clang, LLD開発ライブラリ（バージョン 21.x）（オプション）

### インストール方法

#### 方法1: ビルド済みバイナリのダウンロード

[公式ダウンロードページ](https://ziglang.org/download/)からプラットフォームに応じたバイナリをダウンロードできます。

#### 方法2: ソースからのビルド（LLVM使用）

```bash
mkdir build
cd build
cmake ..
make install
```

`stage3/bin/zig` にコンパイラが生成されます。

#### 方法3: ソースからのビルド（LLVM無し）

```bash
cc -o bootstrap bootstrap.c
./bootstrap
```

カレントディレクトリに `zig2` 実行ファイルが生成されます。ただし、この方法ではリリースモード最適化など一部機能が制限されます。

#### 方法4: 既存のZigを使用したビルド

```bash
"$ZIG_PREFIX/zig" build \
  -p stage3 \
  --search-prefix "$LLVM_PREFIX" \
  --zig-lib-dir "lib" \
  -Dstatic-llvm
```

## 開発

### ビルドオプション

```bash
# Releaseビルド
zig build -Doptimize=ReleaseSafe

# LLVMバックエンド有効化
zig build -Denable-llvm

# インクリメンタルコンパイル（高速リビルド）
zig build --watch -fincremental
```

### テストの実行

```bash
# 全テスト実行
stage4/bin/zig build test

# 標準ライブラリテストのみ
stage4/bin/zig build test-std -Dskip-release

# 特定のテストフィルタ
zig test lib/std/std.zig --zig-lib-dir lib --test-filter "json."
```

### コード品質

- **フォーマット**: `zig fmt` による自動フォーマット
- **デバッグ**: LLDB/GDBのpretty printerスクリプトを`tools/`ディレクトリで提供
- **ドキュメント**: `zig std` コマンドで標準ライブラリのドキュメントを閲覧可能

### 推奨ビルド設定

開発時には以下のCMake設定を推奨:

```bash
cmake .. -DCMAKE_BUILD_TYPE=Release -GNinja -DZIG_NO_LIB=ON
```

- `-DCMAKE_BUILD_TYPE=Release`: 高速な再コンパイル
- `-GNinja`: Makeより高速なNinjaビルドシステム
- `-DZIG_NO_LIB=ON`: lib/ディレクトリのコピーをスキップし、ソースツリーから直接使用
