# アーキテクチャ設計書

## 概要

本ドキュメントは、Zig言語コンパイラ（ziglang/zig）のアーキテクチャ設計を記載したものである。Zigコンパイラは、セルフホスティングのプログラミング言語コンパイラであり、C/C++のコンパイル、リンキング、およびZigソースコードからネイティブコードへの変換を行う。バージョン0.16.0を対象としている。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| LLVM | C++ FFI / ライブラリリンク | コード最適化、コード生成、リンキング（オプション） |
| Clang | C++ FFI / ライブラリリンク | C/C++/Objective-Cコンパイル |
| LLD | C++ FFI / ライブラリリンク | リンカーバックエンド |
| システムC/C++コンパイラ | プロセス実行 | ブートストラップビルド |
| OS (ファイルシステム) | システムコール | ソースファイル読み込み、出力ファイル生成 |
| ビルドシステム (CMake/Zig Build) | CLI | ビルド設定、依存関係解決 |

## レイヤー構成

### アーキテクチャスタイル

**パイプライン・アーキテクチャ**（コンパイラフロントエンド→ミドルエンド→バックエンドの多段処理）と**モジュラー・アーキテクチャ**の組み合わせ。

コンパイラの各フェーズ（パース、セマンティック解析、コード生成、リンキング）が独立したモジュールとして実装され、中間表現（ZIR、AIR）を介して連携する。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| CLI Layer | コマンドライン引数処理、サブコマンドのディスパッチ | `main.zig` |
| Compilation Layer | コンパイル全体の管理、ジョブスケジューリング | `Compilation.zig`, `Zcu.zig` |
| Frontend Layer | ソースコードの構文解析、ZIR生成 | `std.zig.Ast`, `std.zig.AstGen`, `std.zig.Zir` |
| Semantic Analysis Layer | 型検査、コンパイル時評価、AIR生成 | `Sema.zig`, `Type.zig`, `Value.zig` |
| Codegen Layer | AIRからマシンコードへの変換 | `codegen.zig`, `codegen/llvm.zig`, `codegen/c.zig`, `codegen/x86_64/`, etc. |
| Linker Layer | オブジェクトファイルのリンキング、実行可能ファイル生成 | `link.zig`, `link/Elf.zig`, `link/MachO.zig`, `link/Coff.zig`, etc. |
| Standard Library Layer | ランタイムサポート、標準API | `lib/std/` |

### レイヤー間の依存関係ルール

```
CLI Layer
    ↓
Compilation Layer ← [設定・制御]
    ↓
Frontend Layer (AST → ZIR)
    ↓
Semantic Analysis Layer (ZIR → AIR)
    ↓
Codegen Layer (AIR → MIR/Machine Code)
    ↓
Linker Layer (Object Files → Executable)
```

**依存の方向性**:
- 上位レイヤーから下位レイヤーへの依存のみ許可
- `InternPool` は全レイヤーから参照される共有データ構造
- `Compilation` がコンパイル全体を統括し、各フェーズの実行を制御

**禁止される参照**:
- 下位レイヤーから上位レイヤーへの直接参照
- CodegenからFrontendへの直接参照（AIRを介する）

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| コンパイル制御 | コンパイル全体の管理、キャッシュ、並列化 | `Compilation.zig`, `Zcu.zig`, `Package.zig` |
| 構文解析 | Zigソースコードの構文解析 | `std.zig.Ast`, `std.zig.AstGen` |
| 中間表現 | ZIR/AIR/MIRの定義と操作 | `Air.zig`, `std.zig.Zir`, `InternPool.zig` |
| セマンティック解析 | 型検査、型推論、コンパイル時評価 | `Sema.zig`, `Type.zig`, `Value.zig` |
| コード生成 | ターゲット固有のマシンコード生成 | `codegen/*.zig` |
| リンキング | オブジェクトファイルの結合、シンボル解決 | `link/*.zig` |
| C言語サポート | C/C++コンパイル、translate-c | `zig_clang_*.cpp`, `libs/*.zig` |
| ビルドシステム | プロジェクトビルド、依存関係管理 | `std.Build`, `build.zig` |

### パッケージ構造

```
zig/
├── src/                          # コンパイラ本体
│   ├── main.zig                  # エントリポイント、CLIディスパッチ
│   ├── Compilation.zig           # コンパイル管理
│   ├── Zcu.zig                   # Zig Compilation Unit
│   ├── Sema.zig                  # セマンティック解析
│   ├── Air.zig                   # 解析済み中間表現
│   ├── InternPool.zig            # 値・型のインターン化プール
│   ├── Type.zig                  # 型システム
│   ├── Value.zig                 # 値表現
│   ├── Package.zig               # パッケージ管理
│   ├── codegen/                  # コード生成バックエンド
│   │   ├── llvm.zig              # LLVM バックエンド
│   │   ├── c.zig                 # C バックエンド
│   │   ├── x86_64/               # x86_64 ネイティブバックエンド
│   │   ├── aarch64/              # AArch64 ネイティブバックエンド
│   │   ├── riscv64/              # RISC-V 64 ネイティブバックエンド
│   │   ├── wasm/                 # WebAssembly バックエンド
│   │   └── spirv/                # SPIR-V バックエンド
│   ├── link/                     # リンカー実装
│   │   ├── Elf.zig               # ELF リンカー
│   │   ├── MachO.zig             # Mach-O リンカー
│   │   ├── Coff.zig              # COFF リンカー
│   │   ├── Wasm.zig              # WebAssembly リンカー
│   │   └── Dwarf.zig             # デバッグ情報生成
│   ├── libs/                     # C ライブラリサポート
│   │   ├── glibc.zig
│   │   ├── musl.zig
│   │   └── mingw.zig
│   └── *.cpp                     # LLVM/Clang インターフェース
├── lib/                          # 標準ライブラリ・ランタイム
│   ├── std/                      # 標準ライブラリ
│   │   ├── std.zig               # ルートモジュール
│   │   ├── zig/                  # Zig言語サポート
│   │   │   ├── Ast.zig           # 抽象構文木
│   │   │   ├── AstGen.zig        # ZIR生成
│   │   │   └── Zir.zig           # Zig IR定義
│   │   └── Build/                # ビルドシステム
│   ├── compiler_rt/              # コンパイラランタイム
│   └── libc/                     # libc 実装
├── build.zig                     # ビルド設定
└── test/                         # テストスイート
```

### コンポーネント依存関係

**主要コンポーネント間の依存関係**:

1. **main.zig** → Compilation.zig: コンパイルセッションの作成・管理
2. **Compilation.zig** → Zcu.zig: Zigソースコードのコンパイル管理
3. **Zcu.zig** → InternPool.zig: 型・値のインターン化
4. **Sema.zig** → Air.zig: 解析済み中間表現の生成
5. **codegen.zig** → link.zig: 生成コードのリンカーへの引き渡し
6. **全モジュール** → std: 標準ライブラリの利用

## ミドルウェア構成

### データベース

本プロジェクトは外部データベースを使用しない。代わりに、以下のキャッシュ機構を使用。

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| キャッシュ | ファイルシステムベースキャッシュ | N/A | ZIRキャッシュ、コンパイル成果物キャッシュ |
| インメモリストア | InternPool | N/A | 型・値のインターン化、依存関係追跡 |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| std.Build.Cache | N/A | コンパイル成果物キャッシュ（ハッシュベース） | 永続 |
| global_zir_cache | N/A | グローバルZIRキャッシュ | 永続 |
| local_zir_cache | N/A | ローカルZIRキャッシュ | 永続 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| work_queues (内部実装) | N/A | コンパイルジョブの並列実行管理 |
| c_object_work_queue | N/A | Cオブジェクトコンパイルジョブ管理 |
| link_queue | N/A | リンカータスク管理 |

### 検索エンジン

該当なし

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| LLVM | 21.x | コード最適化、コード生成（オプション） |
| Clang | 21.x | C/C++コンパイル |
| LLD | 21.x | リンカーバックエンド |
| Tracy | オプション | パフォーマンスプロファイリング |

## データフロー

### リクエスト処理の流れ

**Zigソースコードのコンパイル**:

1. **CLI引数解析** (`main.zig`): `zig build-exe foo.zig` などのコマンドをパース
2. **Compilation作成** (`Compilation.zig`): コンパイル設定の初期化、ターゲット解決
3. **ソースファイル読み込み** (`Zcu.zig`): ファイルをimportテーブルに追加
4. **構文解析** (`std.zig.Ast`): ソースコードからAST生成
5. **ZIR生成** (`std.zig.AstGen`): ASTからZig IR生成
6. **セマンティック解析** (`Sema.zig`): ZIRからAIR生成、型検査、コンパイル時評価
7. **コード生成** (`codegen.zig`): AIRからMIR/マシンコード生成
8. **リンキング** (`link.zig`): オブジェクトファイルを結合、実行可能ファイル生成
9. **出力** (`Compilation.zig`): 成果物をファイルシステムに書き出し

### 非同期処理の流れ

**並列コンパイル**:

1. **AstGenワーカー**: 複数ファイルの並列パース・ZIR生成
2. **Semaワーカー**: 依存関係を考慮した並列セマンティック解析
3. **Codegenワーカー**: 関数単位の並列コード生成
4. **リンカー**: バックグラウンドでの増分リンキング

処理はwork_queuesを通じてスケジューリングされ、スレッドプール上で実行される。

### データ永続化の流れ

1. **ZIRキャッシュ**: 構文解析済みZIRをファイルシステムにキャッシュ
2. **成果物キャッシュ**: コンパイル済みオブジェクト・実行可能ファイルをハッシュベースでキャッシュ
3. **増分コンパイル**: InternPoolで依存関係を追跡し、変更された部分のみ再コンパイル

## 横断的関心事

### 認証・認可

該当なし（コンパイラはローカルツールとして動作）

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| デバッグログ | `std.log` | 標準エラー出力 |
| Tracyプロファイリング | `-Dtracy` オプション | Tracy サーバー |
| verbose出力 | `--verbose-*` オプション | 標準出力/標準エラー |

### エラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| 構文エラー | AST.Error | ErrorBundle形式でユーザーに表示 |
| セマンティックエラー | Zcu.ErrorMsg | ErrorBundle形式でユーザーに表示 |
| リンカーエラー | link.Diags | ErrorBundle形式でユーザーに表示 |
| 内部エラー | panic/crash_report | クラッシュレポート生成 |

**エラー集約**: 全エラーは`std.zig.ErrorBundle`形式に集約され、ソースコード位置、メッセージ、参照トレースを含む形式で出力される。

### トランザクション管理

該当なし（ファイルシステムベースの処理のため）

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 単一責任の原則 | 全モジュール | `Sema.zig`はセマンティック解析のみ、`codegen.zig`はコード生成のみ |
| インターフェース分離 | リンカー・バックエンド | `link.File`インターフェース、複数バックエンドが同一APIを実装 |
| 依存性逆転 | コード生成 | 抽象的なAIRに依存、具体的なターゲットに依存しない |
| コンポジション | 型システム | `Type.zig`は`InternPool.Index`を内包 |
| データ指向設計 | InternPool | MultiArrayList、構造体のArrayを使用 |

### コーディング規約

- **命名規則**: snake_caseを基本とし、型名のみPascalCase
- **エラー処理**: Zig標準のerror union使用、`try`/`catch`による伝播
- **メモリ管理**: Arena allocatorによるスコープベースの管理、GPA(General Purpose Allocator)の使用
- **並列安全性**: Mutexによる排他制御、スレッドローカルストレージの活用
- **コメント**: ドキュメントコメント(`///`)による公開API文書化

## 備考

- **LLVMオプション**: `-fno-llvm`でLLVMなしのセルフホストバックエンドを使用可能
- **ブートストラップ**: `bootstrap.c`を使用したC言語からのブートストラップビルドをサポート
- **クロスコンパイル**: 任意のターゲットへのクロスコンパイルをネイティブサポート
- **増分コンパイル**: InternPoolの依存関係追跡により、変更箇所のみの再コンパイルを実現
