# Zig コンパイラ コードリーディングガイドライン

## はじめに

このガイドラインは、Zigコンパイラ（Zig言語の公式実装）のコードベースを効率的に理解するための手引きです。
Zig言語に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語（C/C++、Rust等）からの経験者
- コンパイラ実装に興味を持つ開発者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、Zig言語の基本構文と概念を解説します。

### 1.1 プログラム構造

Zigプログラムは、ファイル単位でモジュールとして構成されます。各ファイルは暗黙的に「ファイルスコープ」を持ち、`@import`で他のモジュールを参照できます。

```zig
// ファイル: src/main.zig:1-14
const builtin = @import("builtin");
const native_os = builtin.os.tag;

const std = @import("std");
const Io = std.Io;
const assert = std.debug.assert;
const fs = std.fs;
const mem = std.mem;
const process = std.process;
const Allocator = mem.Allocator;
const Ast = std.zig.Ast;
```

**ポイント:**
- `@import("std")` は標準ライブラリ全体をインポート
- `@import("builtin")` はコンパイル時情報（ターゲット、最適化レベル等）を取得
- ローカルファイルは相対パスでインポート（例: `@import("Compilation.zig")`）

### 1.2 データ型と変数

Zigは静的型付け言語で、明示的な型注釈が基本です。

```zig
// ファイル: src/Compilation.zig:50-57
/// General-purpose allocator. Used for both temporary and long-term storage.
gpa: Allocator,
/// Arena-allocated memory, mostly used during initialization.
arena: Allocator,
io: Io,
environ_map: *const std.process.Environ.Map,
thread_limit: usize,
```

**主要な型:**
| 型 | 説明 | 例 |
|-----|------|-----|
| `u8`, `u16`, `u32`, `u64` | 符号なし整数 | `u32 = 42` |
| `i8`, `i16`, `i32`, `i64` | 符号付き整数 | `i32 = -1` |
| `usize`, `isize` | ポインタサイズ整数 | `thread_limit: usize` |
| `bool` | 真偽値 | `skip_install: bool` |
| `[]const u8` | 不変スライス（文字列） | `"hello"` |
| `?T` | オプショナル型 | `?*Zcu` |
| `!T` | エラーユニオン型 | `!void` |

### 1.3 制御構造

```zig
// ファイル: src/Type.zig:40-58
pub fn isSelfComparable(ty: Type, zcu: *const Zcu, is_equality_cmp: bool) bool {
    return switch (ty.zigTypeTag(zcu)) {
        .int,
        .float,
        .comptime_float,
        .comptime_int,
        => true,

        .vector => ty.elemType2(zcu).isSelfComparable(zcu, is_equality_cmp),

        .bool,
        .type,
        .void,
        .error_set,
        .@"fn",
        .@"opaque",
        => is_equality_cmp,

        else => false,
    };
}
```

**ポイント:**
- `switch`は網羅的（全ケースをカバーするか`else`が必要）
- 複数の値に対して同じ処理を適用可能（カンマ区切り）
- `@"keyword"` 構文で予約語を識別子として使用可能

### 1.4 関数/メソッド定義

```zig
// ファイル: src/codegen.zig:67-80
pub fn legalizeFeatures(pt: Zcu.PerThread, nav_index: InternPool.Nav.Index) ?*const Air.Legalize.Features {
    const zcu = pt.zcu;
    const target = &zcu.navFileScope(nav_index).mod.?.resolved_target.result;
    switch (target_util.zigBackend(target, zcu.comp.config.use_llvm)) {
        else => unreachable,
        inline .stage2_llvm,
        .stage2_c,
        .stage2_wasm,
        => |backend| {
            // 処理
        },
    }
}
```

**ポイント:**
- `pub`で公開関数、省略でプライベート
- 第1引数が構造体自身の場合、メソッド呼び出しが可能
- `inline`キーワードでコンパイル時展開を強制

### 1.5 モジュール/インポート

```zig
// ファイル: src/Sema.zig:16-39
const Sema = @This();
const Value = @import("Value.zig");
const MutableValue = @import("mutable_value.zig").MutableValue;
const Type = @import("Type.zig");
const Air = @import("Air.zig");
const Zir = std.zig.Zir;
const Zcu = @import("Zcu.zig");
const trace = @import("tracy.zig").trace;
```

**インポートパターン:**
| パターン | 説明 |
|---------|------|
| `@import("std")` | 標準ライブラリ |
| `@import("builtin")` | コンパイル時組み込み情報 |
| `@import("Foo.zig")` | 同階層の.zigファイル |
| `@import("foo/bar.zig")` | サブディレクトリのファイル |
| `@This()` | 現在のファイル/構造体自身 |

---

## 2. プロジェクト固有の概念

> このセクションでは、Zigコンパイラ特有の概念を解説します。

### 2.1 コンパイルパイプライン

Zigコンパイラは以下の段階でソースコードを処理します：

```
ソースコード (.zig)
    ↓ [Parser]
AST (Abstract Syntax Tree)
    ↓ [AstGen]
ZIR (Zig Intermediate Representation)
    ↓ [Sema - 意味解析]
AIR (Analyzed Intermediate Representation)
    ↓ [CodeGen]
マシンコード / バイトコード
```

### 2.2 主要な中間表現

#### ZIR (Zig Intermediate Representation)
```zig
// ファイル: src/Sema.zig:1-6
//! Semantic analysis of ZIR instructions.
//! Shared to every Block. Stored on the stack.
//! State used for compiling a ZIR into AIR.
//! Transforms untyped ZIR instructions into semantically-analyzed AIR instructions.
//! Does type checking, comptime control flow, and safety-check generation.
//! This is the the heart of the Zig compiler.
```

#### AIR (Analyzed Intermediate Representation)
```zig
// ファイル: src/Air.zig:1-6
//! Analyzed Intermediate Representation.
//!
//! This data is produced by Sema and consumed by codegen.
//! Unlike ZIR where there is one instance for an entire source file, each function
//! gets its own `Air` instance.
```

### 2.3 InternPool

型と値の正規化されたストレージ：

```zig
// ファイル: src/InternPool.zig:1-3
//! All interned objects have both a value and a type.
//! This data structure is self-contained.
const InternPool = @This();
```

InternPoolは以下を管理：
- 型情報のインターン化（重複排除）
- 値のインターン化
- 依存関係追跡

### 2.4 Sema（意味解析）

```zig
// ファイル: src/Sema.zig:41-56
pt: Zcu.PerThread,
/// Alias to `zcu.gpa`.
gpa: Allocator,
/// Points to the temporary arena allocator of the Sema.
arena: Allocator,
code: Zir,
air_instructions: std.MultiArrayList(Air.Inst) = .{},
air_extra: std.ArrayList(u32) = .empty,
/// Maps ZIR to AIR.
inst_map: InstMap = .{},
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `PascalCase.zig` | 構造体/型を定義するファイル | `Compilation.zig`, `Sema.zig` |
| `snake_case.zig` | ユーティリティ/ヘルパーモジュール | `target.zig`, `tracy.zig` |
| `snake_case/` | 関連機能をグループ化 | `codegen/`, `link/` |
| `libs/` | 外部ライブラリ統合 | `libs/glibc.zig` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `PascalCase` | 型名（構造体、enum） | `Type`, `Value`, `InternPool` |
| `camelCase` | 関数名 | `zigTypeTag`, `isSelfComparable` |
| `snake_case` | ローカル変数、フィールド | `thread_limit`, `gpa` |
| `SCREAMING_SNAKE_CASE` | コンパイル時定数 | - |
| `_` プレフィックス | 未使用変数 | `const _ = unused;` |
| `Index` サフィックス | インデックス型 | `Nav.Index`, `TrackedInst.Index` |
| `Optional` サフィックス | オプショナルラッパー | `DepEntry.Index.Optional` |

### 3.3 特殊な命名パターン

| パターン | 意味 | 例 |
|---------|------|-----|
| `@This()` | 現在の型への参照 | `const Sema = @This();` |
| `@"keyword"` | 予約語を識別子化 | `.@"fn"`, `.@"enum"` |
| `pt` | PerThread（スレッドローカル状態） | `pt: Zcu.PerThread` |
| `zcu` | Zig Compilation Unit | `const zcu = pt.zcu;` |
| `gpa` | General Purpose Allocator | `gpa: Allocator` |
| `mod` | Module | `root_mod: *Package.Module` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
zig/
├── build.zig              # ビルドシステム定義
├── build.zig.zon          # パッケージ依存定義
├── bootstrap.c            # LLVMなしブートストラップ用C
├── CMakeLists.txt         # CMakeビルド定義
├── README.md              # プロジェクトドキュメント
├── LICENSE                # MITライセンス
│
├── src/                   # コンパイラ本体ソース
│   ├── main.zig           # エントリーポイント
│   ├── Compilation.zig    # コンパイル全体管理
│   ├── Sema.zig           # 意味解析（最大ファイル）
│   ├── Zcu.zig            # Zig Compilation Unit
│   ├── InternPool.zig     # 型/値のインターン化
│   ├── Air.zig            # 解析済み中間表現
│   ├── Type.zig           # 型システム
│   ├── Value.zig          # 値表現
│   ├── codegen/           # コード生成バックエンド
│   ├── link/              # リンカー実装
│   └── libs/              # libc等の統合
│
├── lib/                   # ランタイムライブラリ
│   ├── std/               # 標準ライブラリ
│   ├── c/                 # Cライブラリヘッダ
│   ├── compiler_rt/       # コンパイラランタイム
│   └── include/           # システムヘッダ
│
├── test/                  # テストスイート
│   ├── behavior/          # 動作テスト
│   ├── cases/             # コンパイラテストケース
│   └── standalone/        # スタンドアロンテスト
│
├── tools/                 # 開発ツール
│   ├── docgen.zig         # ドキュメント生成
│   └── update_*.zig       # 各種更新スクリプト
│
├── doc/                   # ドキュメントソース
├── ci/                    # CI設定
└── stage1/                # Stage1コンパイラ（WASM）
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | コンパイラコア実装 | `main.zig`, `Sema.zig`, `Compilation.zig` |
| `src/codegen/` | ターゲット別コード生成 | `llvm.zig`, `c.zig`, `x86_64/` |
| `src/link/` | リンカー実装 | `Elf.zig`, `MachO.zig`, `Coff.zig` |
| `lib/std/` | 標準ライブラリ | `std.zig`, `mem.zig`, `fs.zig` |
| `test/behavior/` | 言語機能テスト | `enum.zig`, `struct.zig` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Zigコンパイラは**パイプラインアーキテクチャ**を採用しています。

```mermaid
graph TB
    subgraph "フロントエンド"
        A[ソースファイル] --> B[Parser/AST]
        B --> C[AstGen/ZIR]
    end

    subgraph "ミドルエンド"
        C --> D[Sema]
        D --> E[AIR]
        D <--> F[InternPool]
    end

    subgraph "バックエンド"
        E --> G{CodeGen}
        G --> H[LLVM]
        G --> I[x86_64]
        G --> J[aarch64]
        G --> K[C]
        G --> L[WASM]
    end

    subgraph "リンク"
        H --> M[Link]
        I --> M
        J --> M
        M --> N[実行ファイル]
    end
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| CLI/Driver | コマンドライン処理 | `src/main.zig` |
| Compilation | ビルド全体管理 | `src/Compilation.zig` |
| 解析 | ソース→ZIR→AIR変換 | `src/Sema.zig` |
| 型システム | 型の表現と検証 | `src/Type.zig`, `src/Value.zig` |
| コード生成 | AIR→機械語変換 | `src/codegen/` |
| リンク | オブジェクト結合 | `src/link/` |
| 標準ライブラリ | ランタイムサポート | `lib/std/` |

### 5.3 データフロー

```
1. main.zig: CLIパース → Compilationインスタンス生成
2. Compilation.zig: 依存解決 → Zcuインスタンス作成
3. Zcu.zig: ファイル管理 → ZIR生成
4. Sema.zig: ZIR解析 → AIR生成 (InternPool参照)
5. codegen/: AIR → ターゲット固有コード
6. link/: オブジェクト結合 → 最終バイナリ
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

```zig
// ファイル: src/main.zig:77-119
const normal_usage =
    \\Usage: zig [command] [options]
    \\
    \\Commands:
    \\
    \\  build            Build project from build.zig
    \\  fetch            Copy a package into global cache and print its hash
    \\  init             Initialize a Zig package in the current directory
    \\
    \\  build-exe        Create executable from source or object files
    \\  build-lib        Create library from source or object files
    ...
;
```

主なコマンド:
- `build`: `build.zig`からプロジェクトをビルド
- `build-exe`: 実行可能ファイルを生成
- `test`: ユニットテストを実行
- `cc`: Cコンパイラとして動作

### 6.2 Compilation（コンパイル管理）

```zig
// ファイル: src/Compilation.zig:1-48
const Compilation = @This();
// ... imports ...

pub const Config = @import("Compilation/Config.zig");

/// General-purpose allocator. Used for both temporary and long-term storage.
gpa: Allocator,
/// Arena-allocated memory
arena: Allocator,
/// Not every Compilation compiles .zig code!
zcu: ?*Zcu,
/// All compilations have a root module
root_mod: *Package.Module,
/// User-specified settings
config: Config,
/// The main output file.
bin_file: ?*link.File,
```

### 6.3 Sema（意味解析 - コンパイラの心臓部）

```zig
// ファイル: src/Sema.zig:1-7
//! Semantic analysis of ZIR instructions.
//! Shared to every Block. Stored on the stack.
//! State used for compiling a ZIR into AIR.
//! Transforms untyped ZIR instructions into semantically-analyzed AIR instructions.
//! Does type checking, comptime control flow, and safety-check generation.
//! This is the the heart of the Zig compiler.
```

Semaの責務:
- 型チェック
- コンパイル時評価
- 安全性チェック生成
- ZIR → AIR変換

### 6.4 InternPool（型/値のインターン化）

```zig
// ファイル: src/InternPool.zig:20-34
/// One item per thread, indexed by `tid`
locals: []Local,
/// Represents the number of simultaneous writers
shards: []Shard,
/// Key is the error name, index is the error tag value
global_error_set: GlobalErrorSet,
```

### 6.5 CodeGen（コード生成）

```zig
// ファイル: src/codegen.zig:50-64
fn importBackend(comptime backend: std.builtin.CompilerBackend) type {
    return switch (backend) {
        .stage2_aarch64 => aarch64,
        .stage2_c => @import("codegen/c.zig"),
        .stage2_llvm => @import("codegen/llvm.zig"),
        .stage2_riscv64 => @import("codegen/riscv64/CodeGen.zig"),
        .stage2_sparc64 => @import("codegen/sparc64/CodeGen.zig"),
        .stage2_spirv => @import("codegen/spirv/CodeGen.zig"),
        .stage2_wasm => @import("codegen/wasm/CodeGen.zig"),
        .stage2_x86, .stage2_x86_64 => @import("codegen/x86_64/CodeGen.zig"),
        _ => unreachable,
    };
}
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Self-referential struct | `@This()`による自己参照 | 高 | 全ての主要ファイル |
| Error union handling | `try`/`catch`によるエラー処理 | 高 | 全ファイル |
| Tagged union | `union(enum)`による型安全な共用体 | 高 | `Air.zig`, `Type.zig` |
| Comptime dispatch | `inline switch`によるコンパイル時分岐 | 中 | `codegen.zig` |
| Arena allocation | アリーナアロケータの活用 | 高 | `Compilation.zig`, `Sema.zig` |

### パターン1: Self-referential struct

**目的:** ファイルスコープの構造体を自身の型として参照

```zig
// ファイル: src/Sema.zig:15
const Sema = @This();

// 使用例
pub fn analyze(self: *Sema) !void {
    // ...
}
```

**解説:** `@This()`は現在のファイルが暗黙的に持つ構造体型への参照を返します。これにより、ファイル全体を1つの構造体として扱えます。

### パターン2: Tagged Union

**目的:** 型安全な可変データ表現

```zig
// ファイル: src/Air.zig:38-70
pub const Tag = enum(u8) {
    arg,
    add,
    add_safe,
    add_optimized,
    add_wrap,
    add_sat,
    sub,
    // ...
};
```

**解説:** enumをタグとして使用することで、実行時にどのバリアントが有効かを安全に判別できます。

### パターン3: Inline Switch

**目的:** コンパイル時にバックエンドを選択

```zig
// ファイル: src/codegen.zig:71-80
switch (target_util.zigBackend(target, zcu.comp.config.use_llvm)) {
    else => unreachable,
    inline .stage2_llvm,
    .stage2_c,
    .stage2_wasm,
    => |backend| {
        return importBackend(backend).legalizeFeatures(target);
    },
}
```

**解説:** `inline`キーワードにより、各ケースがコンパイル時に展開され、不要なケースは除去されます。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（`src/main.zig`）
2. コマンドハンドラを特定
3. 主要処理クラスへの委譲を追跡
4. 中間表現の変換を確認
5. 最終出力を確認

### 8.2 フロー追跡の実例

#### 例1: `zig build-exe`コマンドの処理フロー

**概要:** ソースファイルから実行可能ファイルを生成するフロー

**処理フロー:**
```
main.zig → Compilation.zig → Zcu.zig → Sema.zig → codegen/ → link/
```

**詳細な追跡:**

1. **CLIパース** (`src/main.zig:77-119`)
   ```zig
   const normal_usage =
       \\Usage: zig [command] [options]
       \\
       \\Commands:
       \\  build-exe        Create executable from source or object files
   ```

2. **Compilationインスタンス生成** (`src/Compilation.zig:50-70`)
   ```zig
   gpa: Allocator,
   arena: Allocator,
   zcu: ?*Zcu,
   root_mod: *Package.Module,
   config: Config,
   bin_file: ?*link.File,
   ```

3. **意味解析実行** (`src/Sema.zig:41-50`)
   ```zig
   pt: Zcu.PerThread,
   gpa: Allocator,
   arena: Allocator,
   code: Zir,
   air_instructions: std.MultiArrayList(Air.Inst) = .{},
   ```

4. **コード生成** (`src/codegen.zig:50-64`)
   ```zig
   fn importBackend(comptime backend: std.builtin.CompilerBackend) type {
       return switch (backend) {
           .stage2_llvm => @import("codegen/llvm.zig"),
           // ...
       };
   }
   ```

5. **リンク処理** (`src/link.zig:33-62`)
   ```zig
   pub const Diags = struct {
       gpa: Allocator,
       io: Io,
       mutex: Io.Mutex,
       msgs: std.ArrayList(Msg),
       // ...
   };
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] 中間表現（ZIR/AIR）の変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要とビルド方法
2. `src/main.zig` - エントリーポイントとコマンド一覧
3. `src/Compilation.zig` - コンパイル全体の流れ

#### コンパイラ内部を理解したい場合
1. `src/Sema.zig` - 意味解析（コンパイラの心臓部）
2. `src/InternPool.zig` - 型/値のインターン化
3. `src/Air.zig` - 解析済み中間表現
4. `src/Type.zig`, `src/Value.zig` - 型システム

#### 特定バックエンドを理解したい場合
1. `src/codegen.zig` - バックエンド選択ロジック
2. `src/codegen/llvm.zig` - LLVMバックエンド
3. `src/codegen/x86_64/` - x86_64ネイティブバックエンド

#### 標準ライブラリを理解したい場合
1. `lib/std/std.zig` - 標準ライブラリエントリーポイント
2. `lib/std/mem.zig` - メモリ操作
3. `lib/std/fs.zig` - ファイルシステム

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 最初 |
| `doc/langref.html` | 言語リファレンス | 言語仕様確認時 |
| `build.zig` | ビルド設定 | ビルドカスタマイズ時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `@This()`とは何ですか？
A: 現在のファイルが持つ暗黙の構造体型への参照です。ファイルスコープの関数やフィールドを持つ構造体として扱えます。参照: `src/Sema.zig:15`

#### Q: `?T`と`!T`の違いは？
A: `?T`はオプショナル型（nullまたはT）、`!T`はエラーユニオン型（エラーまたはT）です。

#### Q: `comptime`キーワードの役割は？
A: コンパイル時に評価されることを示します。ジェネリクスやメタプログラミングに使用されます。

#### Q: 大きなファイル（Sema.zigなど）を読む際のコツは？
A:
1. まずファイル冒頭のドキュメントコメントを読む
2. 構造体のフィールド定義を確認
3. `pub fn`の一覧を把握
4. 特定の処理を追う場合は関数名で検索

### デバッグのヒント

1. **`std.log.scoped`の活用**
   ```zig
   const log = std.log.scoped(.sema);
   log.debug("analyzing: {}", .{instruction});
   ```

2. **Tracy統合**
   ```zig
   const tracy = @import("tracy.zig");
   const trace = tracy.trace;
   ```

3. **アサーションの活用**
   ```zig
   const assert = std.debug.assert;
   assert(condition);  // 条件が偽なら即座にパニック
   ```

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| ZIR | Zig Intermediate Representation。パース後の中間表現 |
| AIR | Analyzed Intermediate Representation。意味解析後の中間表現 |
| Sema | Semantic Analysis。意味解析 |
| InternPool | 型と値をインターン化（正規化）するデータ構造 |
| Zcu | Zig Compilation Unit。コンパイル単位 |
| Nav | Named Value。名前付き値 |
| comptime | コンパイル時評価 |
| Stage1 | 旧C++実装のコンパイラ |
| Stage2 | 現在のZig自己ホストコンパイラ |
| Stage3 | Stage2でビルドされたコンパイラ |

### B. ファイル一覧（主要なもの）

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/main.zig` | エントリーポイント | CLIパース、コマンドディスパッチ |
| `src/Compilation.zig` | コンパイル管理 | ビルドプロセス全体の調整 |
| `src/Sema.zig` | 意味解析 | 型チェック、コンパイル時評価 |
| `src/InternPool.zig` | インターン化 | 型/値の正規化ストレージ |
| `src/Air.zig` | 解析済みIR | コード生成への入力 |
| `src/Type.zig` | 型表現 | 型システムの実装 |
| `src/Value.zig` | 値表現 | 値の内部表現 |
| `src/codegen/` | コード生成 | ターゲット別バックエンド |
| `src/link/` | リンカー | オブジェクトファイル結合 |
| `lib/std/` | 標準ライブラリ | ランタイムサポート |

### C. 参考資料

- [Zig公式サイト](https://ziglang.org/)
- [Zig言語リファレンス](https://ziglang.org/documentation/master/)
- [Zig標準ライブラリドキュメント](https://ziglang.org/documentation/master/std/)
- [Zig GitHub/Codeberg](https://codeberg.org/ziglang/zig)
