# セキュリティ設計書

## 概要

本ドキュメントは、Zig言語コンパイラおよび標準ライブラリにおけるセキュリティ設計を記載する。Zigはシステムプログラミング言語であり、メモリ安全性、暗号化機能、通信セキュリティなどの多岐にわたるセキュリティ機能を提供する。

## 認証設計

### 認証方式

Zigコンパイラおよび標準ライブラリは、ユーザー認証を直接提供するアプリケーションではないが、TLS通信における証明書ベースの認証を標準ライブラリで提供している。

**TLS証明書認証（`lib/std/crypto/Certificate.zig`）:**
- X.509証明書のパース・検証機能
- RSA、ECDSA、Ed25519署名アルゴリズムのサポート
- 証明書チェーンの検証
- ホスト名検証（RFC 6125準拠、ワイルドカード証明書対応）

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| TLSセッション管理 | TLS 1.3準拠 | `lib/std/crypto/tls.zig` で実装 |
| Pre-Shared Key (PSK) | サポート | `psk_ke`, `psk_dhe_ke` モード対応 |
| セッション再開 | TLS 1.3チケット方式 | 0-RTT対応 |

## 認可設計

### 権限体系

Zigはコンパイラ/ランタイムライブラリであり、アプリケーションレベルの権限体系は持たないが、コンパイル時のセキュリティオプションを提供する。

| オプション | 機能 | 説明 |
| --- | --- | --- |
| `sanitize_c` | Cコードのサニタイズ | 未定義動作の検出 |
| `sanitize_thread` | スレッドサニタイザ | データ競合の検出 |
| `stack_protection` | スタック保護 | スタックバッファオーバーフロー検出 |

### アクセス制御

**メモリアクセス制御:**
- ポインタ演算の安全性チェック（ランタイムセーフティ）
- 境界チェック（配列・スライスアクセス）
- null ポインタ参照の検出
- アラインメント検証

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | TLS 1.2/1.3クライアント実装（`lib/std/crypto/tls/Client.zig`） |
| プロトコルバージョン | TLS 1.0〜1.3対応、TLS 1.3推奨 |
| 暗号スイート | AES-GCM, ChaCha20-Poly1305, AEGIS対応 |
| 鍵交換 | X25519, secp256r1, secp384r1, ML-KEM（ポスト量子） |
| 証明書検証 | X.509証明書チェーン検証、ホスト名検証 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 対称暗号 | AES-128/256-GCM, AES-128/256-GCM-SIV, AES-CCM, AES-OCB |
| ストリーム暗号 | ChaCha20/12/8, XChaCha20, Salsa20, XSalsa20 |
| AEAD | ChaCha20-Poly1305, XSalsa20-Poly1305, AEGIS-128L/256, ISAP |
| 鍵導出 | HKDF, PBKDF2, Argon2, bcrypt, scrypt |
| ハッシュ関数 | SHA-1, SHA-2 (224/256/384/512), SHA-3, BLAKE2, BLAKE3, MD5 |
| 署名 | Ed25519, ECDSA (P-256/P-384/secp256k1), ML-DSA |
| 鍵交換 | X25519, ECDH (P-256/P-384), ML-KEM（Kyber） |

### 機密情報管理

**セキュアメモリ管理（`lib/std/crypto.zig`）:**
- `secureZero()`: メモリの安全なゼロ化（コンパイラによる最適化を防止）
- `timing_safe.eql()`: 定数時間比較（タイミング攻撃対策）
- `timing_safe.compare()`: 定数時間整数比較
- `classify()`/`declassify()`: Valgrind連携による秘密データのマーキング

**乱数生成（`lib/std/Random.zig`）:**
- `DefaultCsprng`: ChaChaベースの暗号論的安全な乱数生成器
- `DefaultPrng`: Xoshiro256（非暗号用途）

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| バッファオーバーフロー | スタックプロテクター（`lib/compiler_rt/ssp.zig`）、境界チェック |
| 整数オーバーフロー | ランタイム安全性チェック、`@addWithOverflow`等のビルトイン |
| フォーマット文字列攻撃 | 型安全なフォーマット（`std.fmt`） |
| メモリ解放後使用 | アロケータによる検証、デバッグアロケータ |
| タイミング攻撃 | `timing_safe`モジュールによる定数時間演算 |

### スタックスマッシング保護（`lib/compiler_rt/ssp.zig`）

```
- __stack_chk_fail: スタックカナリ検証失敗時のハンドラ
- __stack_chk_guard: スタックカナリ値
- __strcpy_chk, __strncpy_chk: バッファオーバーフロー検出付き文字列コピー
- __strcat_chk, __strncat_chk: バッファオーバーフロー検出付き文字列連結
- __memcpy_chk, __memmove_chk, __memset_chk: バッファオーバーフロー検出付きメモリ操作
```

### サイドチャネル対策（`lib/std/crypto.zig`）

```zig
pub const SideChannelsMitigations = enum {
    none,    // 対策なし（最速）
    basic,   // 基本的な対策
    medium,  // 中程度の対策（デフォルト）
    full,    // 最高レベルの対策
};
```

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| コンパイルエラー | エラーメッセージ、位置情報 | コンパイル時のみ |
| ランタイムパニック | スタックトレース、エラー原因 | プロセス終了まで |
| TLSアラート | TLSプロトコルエラー | 接続終了まで |

**TLSアラート定義（`lib/std/crypto/tls.zig`）:**
- `bad_record_mac`: MACエラー
- `certificate_revoked`: 証明書失効
- `certificate_expired`: 証明書期限切れ
- `handshake_failure`: ハンドシェイク失敗
- `insufficient_security`: セキュリティ不足

## 備考

### ポスト量子暗号サポート

Zig標準ライブラリは、ポスト量子暗号への移行を視野に入れた実装を提供している:
- ML-KEM (Kyber): 鍵カプセル化機構
- ML-DSA: デジタル署名
- ハイブリッド鍵交換: secp256r1_ml_kem256, x25519_ml_kem768

### コンパイル時セキュリティオプション

ビルドシステム（`build.zig`）を通じて、以下のセキュリティオプションを設定可能:
- PIE (Position Independent Executable)
- スタックプロテクター
- サニタイザ（AddressSanitizer, ThreadSanitizer等）
- LTO (Link Time Optimization)

### 参照ファイル

- `lib/std/crypto.zig`: 暗号プリミティブの統合エントリポイント
- `lib/std/crypto/tls.zig`: TLSプロトコル実装
- `lib/std/crypto/Certificate.zig`: X.509証明書処理
- `lib/std/crypto/timing_safe.zig`: 定数時間演算
- `lib/std/Random.zig`: 乱数生成
- `lib/compiler_rt/ssp.zig`: スタック保護
- `src/Compilation/Config.zig`: コンパイルオプション設定
