# データベース設計書

## 概要

本ドキュメントは、Zigコンパイラプロジェクトにおけるデータ構造の設計について記述します。

**重要**: Zigコンパイラはリレーショナルデータベース（SQL/NoSQL）を使用していません。代わりに、コンパイル時のデータを効率的に管理するためのインメモリデータ構造を使用しています。本設計書では、これらのインメモリデータ構造を「論理的なデータストア」として記述します。

### プロジェクト概要

- **プロジェクト名**: Zig Programming Language Compiler
- **プロジェクトタイプ**: システムプログラミング言語コンパイラ
- **データ永続化方式**: インメモリデータ構造（コンパイルセッション中のみ保持）

## テーブル一覧

以下は、Zigコンパイラ内部で使用される主要なデータ構造（論理エンティティ）の一覧です。

| テーブル名（データ構造） | 対応エンティティ | 説明 |
| --- | --- | --- |
| InternPool | 型・値のインターン化プール | すべての型と値を格納するメインデータ構造 |
| TrackedInst | ZIR命令追跡 | ZIR命令へのインクリメンタル更新対応参照 |
| Nav | 名前付きアドレス値 | グローバル値の名前とアドレスを管理 |
| AnalUnit | 解析ユニット | セマンティック解析の対象単位 |
| Namespace | 名前空間 | ZIR名前空間のデータを管理 |
| DepEntry | 依存関係エントリ | インクリメンタルコンパイルの依存関係追跡 |
| File | ソースファイル | Zigソースファイルの情報 |
| EmbedFile | 埋め込みファイル | @embedFileで読み込まれたファイル |
| Export | エクスポート | 外部公開シンボルの情報 |
| Type | 型情報 | 型システムの表現 |
| Value | 値情報 | コンパイル時値の表現 |

## 各テーブル定義

### 1. InternPool（インターンプール）

コンパイラ内で使用されるすべての型と値を格納するメインデータ構造です。マルチスレッド対応のシャード化されたハッシュマップを使用しています。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| locals | []Local | NO | スレッドローカルなデータストレージ（スレッドIDでインデックス） |
| shards | []Shard | NO | シャード化されたデータ構造（並行書き込み対応） |
| global_error_set | GlobalErrorSet | NO | グローバルエラーセット（エラー名からタグ値へのマッピング） |
| tid_width | u0/Log2Int(u32) | NO | スレッドIDのビット幅（シングル/マルチスレッドで異なる） |
| tid_shift_30 | u0/Log2Int(u32) | NO | 30ビット値用のスレッドIDシフト量 |
| tid_shift_31 | u0/Log2Int(u32) | NO | 31ビット値用のスレッドIDシフト量 |
| tid_shift_32 | u0/Log2Int(u32) | NO | 32ビット値用のスレッドIDシフト量 |
| src_hash_deps | AutoArrayHashMap | NO | ZIR命令のソースコードハッシュ依存関係 |
| nav_val_deps | AutoArrayHashMap | NO | Navの値への依存関係 |
| nav_ty_deps | AutoArrayHashMap | NO | Navの型への依存関係 |
| interned_deps | AutoArrayHashMap | NO | インターン化された値への依存関係 |
| zon_file_deps | AutoArrayHashMap | NO | ZONファイルへの依存関係 |
| embed_file_deps | AutoArrayHashMap | NO | 埋め込みファイルへの依存関係 |
| namespace_deps | AutoArrayHashMap | NO | 名前空間への依存関係 |
| namespace_name_deps | AutoArrayHashMap | NO | 名前空間内の名前への依存関係 |
| first_dependency | AutoArrayHashMap | NO | 各AnalUnitの最初の依存関係エントリへのポインタ |
| dep_entries | ArrayList(DepEntry) | NO | 依存関係エントリのリスト |
| free_dep_entries | ArrayList(DepEntry.Index) | NO | 再利用可能な空きエントリのインデックスリスト |

### 2. TrackedInst（追跡命令）

ZIR命令をインクリメンタル更新を通じて追跡するためのデータ構造です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| file | FileIndex | NO | 命令が存在するファイルのインデックス |
| inst | Zir.Inst.Index | NO | ZIR命令のインデックス |

**Index型の補足**:
- `TrackedInst.Index`: 32ビット整数で、スレッドIDと実際のインデックスがパックされている
- `TrackedInst.MaybeLost`: 更新により「失われた」可能性のある命令を表現

### 3. Nav（名前付きアドレス値）

グローバル値の名前とアドレスを管理するデータ構造です。26バイトのコンパクト表現を使用しています。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | NullTerminatedString | NO | 非修飾名（名前空間ルックアップ・エラーメッセージ用） |
| fqn | NullTerminatedString | NO | 完全修飾名 |
| analysis.namespace | NamespaceIndex | YES | 解析対象の名前空間（セマンティック解析時のみ） |
| analysis.zir_index | TrackedInst.Index | YES | ZIR命令インデックス（セマンティック解析時のみ） |
| status | union(enum) | NO | 解析状態（unresolved/type_resolved/fully_resolved） |

**status.type_resolved サブフィールド**:
| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| type | InternPool.Index | NO | 解決された型 |
| is_const | bool | NO | 定数かどうか |
| alignment | Alignment | NO | アライメント |
| linksection | OptionalNullTerminatedString | YES | リンカセクション |
| addrspace | std.builtin.AddressSpace | NO | アドレス空間 |
| is_threadlocal | bool | NO | スレッドローカルかどうか |
| is_extern_decl | bool | NO | extern宣言かどうか |

**status.fully_resolved サブフィールド**:
| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| val | InternPool.Index | NO | 解決された値 |
| is_const | bool | NO | 定数かどうか |
| alignment | Alignment | NO | アライメント |
| linksection | OptionalNullTerminatedString | YES | リンカセクション |
| addrspace | std.builtin.AddressSpace | NO | アドレス空間 |

### 4. AnalUnit（解析ユニット）

セマンティック解析を受ける単一のエンティティを表します。インクリメンタル依存関係の「ソース」です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| kind | Kind (enum u32) | NO | 解析ユニットの種類 |
| id | u32 | NO | 種類に応じた識別子 |

**Kind列挙型**:
| 値 | 説明 |
| --- | --- |
| comptime | comptime宣言のボディを解析 |
| nav_val | Navの値を解決 |
| nav_ty | Navの型を解決 |
| type | struct/union/enum型を解決 |
| func | ランタイム関数のボディを解析 |
| memoized_state | Zcu上のメモ化状態を解決 |

### 5. ComptimeUnit（コンパイル時ユニット）

comptime宣言のコンテキストを表すデータ構造です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| zir_index | TrackedInst.Index | NO | ZIR命令インデックス |
| namespace | NamespaceIndex | NO | 名前空間インデックス |

### 6. DepEntry（依存関係エントリ）

インクリメンタルコンパイルにおける依存関係を追跡するための連結リストエントリです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| depender | AnalUnit | NO | 依存する側のAnalUnit |
| next_dependee | DepEntry.Index | YES | 同じdependeeに依存する次のエントリ |
| next_depender | DepEntry.Index | YES | 同じdependerの次の依存関係 |

### 7. Zcu（Zigコンパイルユニット）

Zigソースコード全体のコンパイルを管理するメインデータ構造です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| gpa | Allocator | NO | 汎用アロケータ |
| comp | *Compilation | NO | Compilationへのポインタ |
| llvm_object | ?LlvmObject.Ptr | YES | LLVMバックエンド使用時のオブジェクト |
| root_mod | *Package.Module | NO | ルートモジュール |
| main_mod | *Package.Module | NO | メインモジュール（通常はroot_modと同じ） |
| std_mod | *Package.Module | NO | 標準ライブラリモジュール |
| global_zir_cache | Cache.Directory | NO | グローバルZIRキャッシュディレクトリ |
| local_zir_cache | Cache.Directory | NO | ローカルZIRキャッシュディレクトリ |
| all_exports | ArrayList(Export) | NO | すべてのエクスポート値 |
| free_exports | ArrayList(Export.Index) | NO | 再利用可能なエクスポートインデックス |
| single_exports | AutoArrayHashMap | NO | 単一エクスポートを行うAnalUnitのマップ |
| multi_exports | AutoArrayHashMap | NO | 複数エクスポートを行うAnalUnitのマップ |
| builtin_modules | AutoArrayHashMap | NO | ビルトインモジュールのマップ |
| module_roots | AutoArrayHashMap | NO | モジュールルートファイルのマップ |
| import_table | ArrayHashMap | NO | インポートテーブル |
| alive_files | AutoArrayHashMap | NO | 現在の更新で「生きている」ファイル |
| embed_table | ArrayHashMap | NO | 埋め込みファイルテーブル |
| intern_pool | InternPool | NO | 型・値のインターンプール |
| analysis_in_progress | AutoArrayHashMap | NO | 解析中のユニット |
| failed_analysis | AutoArrayHashMap | NO | 解析失敗したユニットとエラーメッセージ |
| transitive_failed_analysis | AutoArrayHashMap | NO | 推移的に失敗したユニット |
| failed_codegen | AutoArrayHashMap | NO | コード生成に失敗したNav |
| failed_types | AutoArrayHashMap | NO | 解決に失敗した型 |
| compile_logs | AutoArrayHashMap | NO | @compileLogの呼び出し記録 |

### 8. Type（型）

Zigの型システムにおける型を表現するラッパー構造体です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| ip_index | InternPool.Index | NO | InternPool内のインデックス |

### 9. Value（値）

Zigのコンパイル時値を表現するラッパー構造体です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| ip_index | InternPool.Index | NO | InternPool内のインデックス |

### 10. Export（エクスポート）

外部に公開されるシンボルの情報を表すデータ構造です。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | String | NO | エクスポート名 |
| owner | AnalUnit | NO | エクスポートを所有するAnalUnit |
| nav | Nav.Index | NO | エクスポートされるNav |

## 備考

### データ構造の特徴

1. **インメモリ設計**: すべてのデータはコンパイルプロセス中のみメモリ上に存在し、外部データベースへの永続化は行われません。

2. **マルチスレッド対応**: InternPoolは複数スレッドからの同時アクセスに対応するため、シャード化されたハッシュマップとミューテックスを使用しています。

3. **インクリメンタルコンパイル対応**: 依存関係追跡（DepEntry）により、変更されたファイルのみを再コンパイルするインクリメンタルコンパイルをサポートしています。

4. **コンパクト表現**: Nav等のデータ構造は、メモリ効率を最大化するためにビットパッキングを使用しています（例：Navは26バイト）。

5. **キャッシュ機構**: ZIRキャッシュディレクトリを使用して、AstGenの結果をファイルシステムにキャッシュします。

### インデックス型の設計

多くのインデックス型（`TrackedInst.Index`、`Nav.Index`等）は、スレッドIDとローカルインデックスを32ビット整数にパックしています。これにより：
- スレッド間でのユニークな識別が可能
- メモリ効率の向上
- 高速なルックアップが実現

### 依存関係管理

インクリメンタルコンパイルのため、以下の依存関係が追跡されます：
- ソースコードハッシュ依存（`src_hash_deps`）
- Nav値/型依存（`nav_val_deps`、`nav_ty_deps`）
- インターン化値依存（`interned_deps`）
- ファイル依存（`zon_file_deps`、`embed_file_deps`）
- 名前空間依存（`namespace_deps`、`namespace_name_deps`）
