# バッチ設計書 21-generate_c_size_and_align_checks

## 概要

本ドキュメントは、Cの基本型のサイズとアライメントを検証するための`_Static_assert`文を生成するバッチ処理「generate_c_size_and_align_checks」の設計書です。

### 本バッチの処理概要

このバッチは、指定されたターゲットトリプル（例: x86_64-linux-gnu）に対して、Cの基本組み込み型（char, short, int, long, float, double等）のサイズとアライメントを検証するC言語の`_Static_assert`文を自動生成します。生成された出力は、対象ターゲットのCコンパイラに通すことで、ZigコンパイラのABI実装がCコンパイラと同じ値を使用していることを検証できます。

**業務上の目的・背景**：ZigコンパイラはCとのABI互換性を重視しており、クロスコンパイル時に各ターゲットプラットフォームでCの基本型のサイズとアライメントが正確に一致している必要があります。このバッチは、Zigが内部で保持するC型情報と、実際のCコンパイラが使用する値との整合性を自動検証する手段を提供します。これにより、ABI互換性のバグを早期に発見し、クロスプラットフォーム開発の信頼性を向上させます。

**バッチの実行タイミング**：手動実行。新しいターゲットプラットフォームのサポート追加時、またはABI関連の変更を行った際に実行します。CI/CDパイプラインでの回帰テストとしても使用可能です。

**主要な処理内容**：
1. コマンドライン引数からターゲットトリプル（例: x86_64-linux-gnu）を解析
2. ターゲット情報を基にZigのTarget.Queryを生成
3. 全てのC基本型（char, short, int, long, float, double等12種類）について繰り返し処理
4. 各型のsizeof値を取得し`_Static_assert`文を生成
5. 各型の`_Alignof`値を取得し`_Static_assert`文を生成
6. 各型の`__alignof`（preferred alignment）値を取得し`_Static_assert`文を生成
7. 生成した全ての`_Static_assert`文を標準出力に出力

**前後の処理との関連**：前提となるバッチはありません。スタンドアロンの検証ツールとして動作します。生成された出力は、対象プラットフォームのCコンパイラ（gcc, clang等）でコンパイルすることで検証が完了します。

**影響範囲**：このバッチ自体はシステムやデータを変更しません。標準出力にC言語コードを生成するのみです。生成されたコードを使用した検証結果は、ZigのABI実装の正確性確認に使用されます。

## バッチ種別

コード生成 / テスト・検証

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時 |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 / CI/CD |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Zigランタイム環境 | Zigコンパイラがインストールされ、`zig run`コマンドが実行可能であること |
| ターゲットサポート | 指定するターゲットトリプルがZigでサポートされていること |

### 実行可否判定

コマンドライン引数としてターゲットトリプルが指定されている場合に実行します。引数が不足している場合は使用方法を表示して終了します（exit code 1）。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| target_triple | string | Yes | なし | ターゲットトリプル（例: x86_64-linux-gnu, aarch64-macos） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| std.Target.Query | Zig内部データ | ターゲット情報の解析に使用 |
| std.Target.CType | Zig内部enum | C基本型の定義（char, short, int等12種類） |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| 標準出力 | C言語ソースコード | `_Static_assert`文の集合 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | 標準出力（リダイレクトで任意のファイル名に保存可能） |
| 出力先 | 標準出力 |
| 文字コード | UTF-8 |
| 区切り文字 | 改行 |

### 出力例

```c
_Static_assert(sizeof(char) == 1, "sizeof(char) == 1");
_Static_assert(_Alignof(char) == 1, "_Alignof(char) == 1");
_Static_assert(__alignof(char) == 1, "__alignof(char) == 1");

_Static_assert(sizeof(short) == 2, "sizeof(short) == 2");
_Static_assert(_Alignof(short) == 2, "_Alignof(short) == 2");
_Static_assert(__alignof(short) == 2, "__alignof(short) == 2");
...
```

## 処理フロー

### 処理シーケンス

```
1. コマンドライン引数の解析
   └─ args.len != 2 の場合、使用方法を表示してexit(1)
2. ターゲット情報の解析
   └─ Target.Query.parseでターゲットトリプルを解析
3. ターゲット解決
   └─ zig.system.resolveTargetQueryでターゲット情報を取得
4. C型のイテレーション
   └─ std.Target.CTypeの全enumフィールド（12種類）を処理
5. 各型について3種類のassert文を生成
   └─ sizeof, _Alignof, __alignof の値を取得して出力
6. 出力のフラッシュ
   └─ バッファリングされた出力を確定
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{引数チェック}
    B -->|引数不足| C[使用方法表示]
    C --> D[exit 1]
    B -->|引数あり| E[ターゲット解析]
    E --> F[C型enumをイテレート]
    F --> G[sizeof assert生成]
    G --> H[_Alignof assert生成]
    H --> I[__alignof assert生成]
    I --> J{次の型あり?}
    J -->|Yes| F
    J -->|No| K[出力フラッシュ]
    K --> L[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行いません。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| exit(1) | 引数エラー | コマンドライン引数が不足 | 正しい引数を指定して再実行 |
| ParseError | ターゲット解析エラー | 無効なターゲットトリプル指定 | サポートされているターゲットを確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

引数エラーの場合は、表示される使用方法に従って正しいターゲットトリプルを指定して再実行します。ターゲット解析エラーの場合は、Zigがサポートするターゲット一覧を確認してください。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 該当なし |
| コミットタイミング | 該当なし |
| ロールバック条件 | 該当なし |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 12種類のC型 x 3種類のassert = 36行 |
| 目標処理時間 | 1秒以内 |
| メモリ使用量上限 | 2KB（バッファサイズ） |

## 排他制御

同時実行可能。標準出力への書き込みのみのため、排他制御は不要です。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| エラーログ | 引数不足時 | 使用方法の表示 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 該当なし | - | - |

## 備考

- 使用例: `zig run tools/generate_c_size_and_align_checks.zig -- x86_64-linux-gnu`
- 出力されたCコードは、対象プラットフォームのCコンパイラでコンパイルすることで検証が完了します
- 全てのassertが通過すれば、ZigのABI実装がCコンパイラと整合していることが確認できます
- 対応するC基本型: char, short, unsigned short, int, unsigned int, long, unsigned long, long long, unsigned long long, float, double, long double
