# バッチ設計書 22-generate_JSONTestSuite

## 概要

本ドキュメントは、JSONTestSuiteのテストケースからZigのテストコードを自動生成するバッチ処理「generate_JSONTestSuite」の設計書です。

### 本バッチの処理概要

このバッチは、JSONTestSuite（https://github.com/nst/JSONTestSuite）のテストケースファイルを読み込み、ZigのJSONパーサーをテストするためのZigテストコードを自動生成します。テストケースファイルの命名規則（y_, n_, i_プレフィックス）に基づいて、それぞれ適切なテスト関数（ok, err, any）を呼び出すテストコードを生成します。

**業務上の目的・背景**：ZigのJSONパーサーの正確性を検証するために、業界標準のJSONTestSuiteを利用した網羅的なテストが必要です。JSONTestSuiteには、有効なJSON（y_）、無効なJSON（n_）、実装依存のJSON（i_）の3カテゴリのテストケースが含まれており、これらを自動的にZigテストコードに変換することで、JSONパーサーの品質保証を効率化します。手動でテストケースを記述する手間を省き、テストケースの更新にも迅速に対応できます。

**バッチの実行タイミング**：手動実行。JSONTestSuiteのテストデータが更新された場合、またはZigのJSONパーサー実装を変更した際に実行します。

**主要な処理内容**：
1. カレントディレクトリ（JSONTestSuiteのtest_parsing/ディレクトリ）のファイル一覧を取得
2. ファイル名をアルファベット順にソート
3. 各テストファイルの内容を読み込み
4. ファイル名の先頭文字（y, n, i）に基づいてテスト種別を判定
5. ファイル内容を適切にエスケープしてZig文字列リテラルに変換
6. 特大ファイル（200文字超）はZig式で圧縮表現
7. 生成したZigテストコードを標準出力に出力

**前後の処理との関連**：前提として、JSONTestSuiteリポジトリのtest_parsing/ディレクトリ内でこのバッチを実行する必要があります。生成されたテストコードは、ZigのJSONパーサーテストスイートに組み込まれます。

**影響範囲**：このバッチ自体はファイルシステムを変更しません。標準出力にZigテストコードを生成するのみです。生成されたコードはlib/std/hash/のテストファイルとして使用されます。

## バッチ種別

コード生成 / テスト・検証

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時 |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Zigランタイム環境 | Zigコンパイラがインストールされ、`zig run`コマンドが実行可能であること |
| 作業ディレクトリ | JSONTestSuiteリポジトリのtest_parsing/ディレクトリ内で実行すること |
| テストファイル存在 | test_parsing/ディレクトリにy_, n_, i_プレフィックスのテストファイルが存在すること |

### 実行可否判定

カレントディレクトリにテストファイルが存在する場合に実行します。ファイルが存在しない場合は空の出力となります。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| なし | - | - | - | コマンドライン引数は不要 |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| カレントディレクトリ | ファイル群 | JSONTestSuiteのtest_parsing/内のテストファイル |
| y_*.json | JSONファイル | 有効なJSONを含むテストケース |
| n_*.json | JSONファイル | 無効なJSONを含むテストケース |
| i_*.json | JSONファイル | 実装依存のJSONを含むテストケース |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| 標準出力 | Zigソースコード | JSONパーサーテストコード |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | 標準出力（リダイレクトで任意のファイル名に保存可能） |
| 出力先 | 標準出力 |
| 文字コード | UTF-8 |
| 区切り文字 | 改行 |

### 出力例

```zig
// This file was generated by _generate_JSONTestSuite.zig
// These test cases are sourced from: https://github.com/nst/JSONTestSuite
const ok = @import("./test.zig").ok;
const err = @import("./test.zig").err;
const any = @import("./test.zig").any;

test "y_array_empty.json" {
    try ok("[]");
}
test "n_array_extra_comma.json" {
    try err("[1,]");
}
test "i_structure_500_nested_arrays.json" {
    try any("[" ** 500 ++ "]" ** 500);
}
```

## 処理フロー

### 処理シーケンス

```
1. ヘッダーコメントとimport文の出力
   └─ 生成ツール情報とテスト関数のimportを出力
2. カレントディレクトリのファイル一覧取得
   └─ Io.Dir.cwd().openDir()でディレクトリを開く
3. ファイル名の収集
   └─ iterateでファイル名をArrayListに追加
4. ファイル名のソート
   └─ std.mem.sortでアルファベット順にソート
5. 各ファイルの処理
   └─ ファイル内容を読み込み、テストコードを生成
6. テスト種別の判定
   └─ ファイル名先頭の文字でok/err/anyを決定
7. 文字列のエスケープ処理
   └─ 制御文字、引用符、バックスラッシュを適切にエスケープ
8. 特大文字列の圧縮
   └─ 200文字超のパターンマッチする文字列をZig式で表現
9. 出力のフラッシュ
   └─ バッファリングされた出力を確定
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[ヘッダー出力]
    B --> C[ディレクトリを開く]
    C --> D[ファイル名を収集]
    D --> E[ファイル名をソート]
    E --> F[次のファイルを取得]
    F --> G{ファイルあり?}
    G -->|No| H[出力フラッシュ]
    H --> I[バッチ終了]
    G -->|Yes| J[ファイル内容読み込み]
    J --> K{ファイル名先頭}
    K -->|y| L[ok関数呼び出し]
    K -->|n| M[err関数呼び出し]
    K -->|i| N[any関数呼び出し]
    L --> O[文字列エスケープ]
    M --> O
    N --> O
    O --> P{200文字超?}
    P -->|Yes| Q[圧縮表現生成]
    P -->|No| R[通常文字列出力]
    Q --> S[テストコード出力]
    R --> S
    S --> F
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行いません。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| IOError | ファイル読み込みエラー | ファイルが存在しない/読み取り権限なし | ファイルの存在と権限を確認 |
| unreachable | 未対応ファイル名 | ファイル名がy,n,i以外で始まる | テストファイル命名規則を確認 |
| unreachable | 未対応の大容量ファイル | 200文字超で既知パターンに一致しない | ソースコードにパターン追加が必要 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

JSONTestSuiteのtest_parsing/ディレクトリ内で実行していることを確認してください。ファイル読み込みエラーの場合は、ファイルの存在と権限を確認します。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 該当なし |
| コミットタイミング | 該当なし |
| ロールバック条件 | 該当なし |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 約300件のテストファイル |
| 目標処理時間 | 5秒以内 |
| メモリ使用量上限 | 250KB（最大ファイルサイズ） + 2KB（出力バッファ） |

## 排他制御

同時実行可能。読み取り専用の処理のため、排他制御は不要です。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| なし | - | ログ出力機能なし |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 該当なし | - | - |

## 備考

- 使用例: JSONTestSuiteのtest_parsing/ディレクトリ内で`zig run path/to/tools/generate_JSONTestSuite.zig`を実行
- 出力をファイルにリダイレクトして使用: `zig run ... > generated_tests.zig`
- 特大ファイルの圧縮対応パターン:
  - `i_structure_500_nested_arrays`: 500個のネストした配列
  - `n_structure_100000_opening_arrays`: 100000個の開き括弧
  - `n_structure_open_array_object`: 50000個のネストしたオブジェクト
- JSONTestSuiteは業界標準のJSONパーサーテストスイートで、RFC 8259準拠の検証に使用されます
