# 機能設計書 10-ast-check

## 概要

本ドキュメントは、Zigコンパイラの`ast-check`コマンドの機能設計について記述する。このコマンドは任意のファイルセットで単純なコンパイルエラーを検査する、構文チェック機能である。

### 本機能の処理概要

`zig ast-check`コマンドは、ZigソースファイルまたはZONファイルの構文解析とAstGen（ZIR生成）を実行し、コンパイルエラーがないかをチェックする。完全なコンパイルは行わず、構文レベルのエラー検出に特化している。これにより、高速なフィードバックを得ることができる。

**業務上の目的・背景**：開発中にコードの構文エラーを素早く検出したい場合、完全なコンパイルを行うのは時間がかかる。`ast-check`は構文解析とZIR生成のみを行い、構文レベルのエラーを高速に検出する。エディタやIDEの構文チェック機能、CI/CDパイプラインでの早期エラー検出などに活用できる。

**機能の利用シーン**：
- エディタ/IDE統合での構文チェック
- CI/CDパイプラインでの早期エラー検出
- コードレビュー前の構文確認
- build.zig.zon（ZONファイル）の検証

**主要な処理内容**：
1. コマンドライン引数の解析
2. ソースファイルの読み込み
3. AST（抽象構文木）の構築
4. ZIRまたはZOIRの生成
5. エラーの検出と報告

**関連システム・外部連携**：
- 標準入力：ファイル指定なしの場合は標準入力から読み込み

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | AST検査画面 | 主機能 | ファイルの構文エラーをチェックする処理 |

## 機能種別

コード品質 / 構文チェック

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| file | []const u8 | No | チェック対象のファイルパス | 有効なファイルパス |
| -t | bool | No | ZIRのテキスト出力（デバッグ用） | - |
| --zon | bool | No | ZONファイルとして強制解析 | - |
| --color | Color | No | カラー出力の制御 | auto/on/off |

### 入力データソース

- コマンドライン引数
- Zigソースファイル（.zig）
- ZONファイル（.zon）
- 標準入力（ファイル未指定時）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| エラーメッセージ | テキスト | 構文エラーの詳細 |
| ZIRテキスト | テキスト | -t指定時のZIR出力 |
| 終了コード | 整数 | 0: エラーなし、1: エラーあり |

### 出力先

- 標準出力（ZIRテキスト、-t指定時）
- 標準エラー出力（エラーメッセージ）

## 処理フロー

### 処理シーケンス

```
1. コマンドライン引数の解析
   └─ cmdAstCheck関数で処理
2. ファイル種別判定
   └─ .zonまたは--zonでZONモード
3. ソース読み込み
   └─ ファイルまたは標準入力から読み込み
4. AST解析
   └─ std.zig.Ast.parseでAST構築
5. ZIR/ZOIR生成
   └─ AstGen.generateまたはZonGen.generate
6. エラーチェック
   └─ hasCompileErrors()で判定
7. 結果出力
   └─ エラーがあればエラーバンドルを表示
```

### フローチャート

```mermaid
flowchart TD
    A[zig ast-check 開始] --> B[コマンドライン引数解析]
    B --> C{ファイル指定?}
    C -->|Yes| D[ファイルを開く]
    C -->|No| E[標準入力を使用]
    D --> F{.zonファイル or --zon?}
    E --> F
    F -->|Yes| G[ZONモード]
    F -->|No| H[Zigモード]
    G --> I[ZonGen.generate]
    H --> J[Ast.parse]
    J --> K[AstGen.generate]
    I --> L{エラーあり?}
    K --> L
    L -->|Yes| M[エラーバンドル表示]
    L -->|No| N{-t指定?}
    M --> O[終了コード1で終了]
    N -->|Yes| P[ZIRテキスト出力]
    N -->|No| Q[終了コード0で終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ファイル種別自動判定 | .zonファイルは自動的にZONモードで解析 | ファイル拡張子による |
| BR-02 | 標準入力対応 | ファイル未指定時は標準入力から読み込み | ファイル未指定時 |
| BR-03 | エラー時終了コード | エラー検出時は終了コード1を返す | エラーあり時 |
| BR-04 | デバッグ出力 | -tオプションでZIRのテキスト表現を出力 | -t指定時 |

### 計算ロジック

特になし

## データベース操作仕様

データベースは使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | SyntaxError | 構文エラー | コードを修正 |
| - | FileNotFound | ファイルが存在しない | パスを確認 |
| - | InvalidZon | 無効なZON形式 | ZON構文を確認 |

### リトライ仕様

自動リトライなし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 高速なフィードバック（完全コンパイルより大幅に高速）
- 単一ファイル処理に最適化

## セキュリティ考慮事項

- 入力ファイルの読み取りのみ実行
- コード実行は行わない

## 備考

- 完全なセマンティック解析は行わない（型チェック等は限定的）
- エディタ統合での利用を想定した高速な構文チェック機能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | main.zig | `src/main.zig` | ast-checkコマンドの分岐処理 |

**主要処理フロー**:
1. **379-380行目**: `mem.eql(u8, cmd, "ast-check")`で判定
2. **cmdAstCheck関数**が呼び出される

#### Step 2: cmdAstCheck関数の理解

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | main.zig | `src/main.zig` | cmdAstCheck関数の実装 |

**主要処理フロー**:
- **6184行目**: `fn cmdAstCheck`の開始
- **6189-6192行目**: オプション変数の初期化（color, want_output_text, force_zon）
- **6194-6222行目**: コマンドライン引数の解析
- **6224-6236行目**: ソースファイルの読み込み
- **6238-6246行目**: モード判定（zig/zon）
- **6248行目**: `Ast.parse`でAST構築
- **6254行目**: `AstGen.generate`でZIR生成（Zigモード）
- **6326行目**: `ZonGen.generate`でZOIR生成（ZONモード）

#### Step 3: エラー処理の理解

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | main.zig | `src/main.zig` | エラーバンドルの構築と表示 |

**主要処理フロー**:
- **6256-6265行目**: hasCompileErrors時のエラー処理
- **6257-6260行目**: ErrorBundle.Wipの構築

### プログラム呼び出し階層図

```
main (src/main.zig)
    │
    └─ cmdAstCheck
            │
            ├─ オプション解析
            │
            ├─ ソース読み込み
            │      └─ ファイルまたは標準入力
            │
            ├─ モード判定 (zig/zon)
            │
            ├─ Ast.parse
            │      └─ AST構築
            │
            └─ ZIR/ZOIR生成
                   ├─ AstGen.generate (Zigモード)
                   └─ ZonGen.generate (ZONモード)
                          │
                          └─ hasCompileErrors()
                                 └─ エラーバンドル表示
```

### データフロー図

```
[入力]                    [処理]                         [出力]

.zig/.zon ファイル ──▶ 読み込み ──▶ Ast.parse ──┐
     │                                          │
標準入力 ─────────────▶ 読み込み ────────────────┤
                                                │
                                                ▼
                                        AstGen/ZonGen
                                                │
                                                ▼
                                        hasCompileErrors?
                                                │
                              ┌─────────────────┴─────────────────┐
                              ▼                                   ▼
                         エラーあり                           エラーなし
                              │                                   │
                              ▼                                   ▼
                      エラーバンドル ──▶ stderr            成功 ──▶ exit(0)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.zig | `src/main.zig` | ソース | cmdAstCheck関数 |
| Ast.zig | `lib/std/zig/Ast.zig` | ライブラリ | AST解析 |
| AstGen.zig | `lib/std/zig/AstGen.zig` | ライブラリ | ZIR生成 |
| ZonGen.zig | `lib/std/zig/ZonGen.zig` | ライブラリ | ZOIR生成 |
| ErrorBundle.zig | `lib/std/zig/ErrorBundle.zig` | ライブラリ | エラー表示 |
