# 機能設計書 13-translate-c

## 概要

本ドキュメントは、Zigコンパイラに組み込まれたC言語変換機能「translate-c」の設計仕様を記載する。`zig translate-c`コマンドは、CソースコードをZigソースコードに自動変換するトランスレーターである。

### 本機能の処理概要

**業務上の目的・背景**：ZigはC言語との高い相互運用性を設計目標としている。既存のCコードベースをZigに移行する際、手動での書き換えは時間がかかり、エラーが発生しやすい。本機能は、Cソースコードを自動的にZigコードに変換することで、移行作業を効率化し、Zigエコシステムへの参入障壁を下げる。また、Cヘッダーファイルを変換することで、ZigからCライブラリを利用する際のバインディング生成にも活用される。

**機能の利用シーン**：
- 既存のCプロジェクトをZigに移植する際の初期変換
- Cライブラリのヘッダーファイルからバインディングを生成する場合
- `@cImport`ビルトインによるCヘッダーの自動インクルード時（内部的に使用）
- Cコードの動作をZigで理解・分析したい場合
- レガシーCコードのリファクタリング支援

**主要な処理内容**：
1. Cソースコードのプリプロセス（マクロ展開、インクルード処理）
2. Cソースコードの構文解析（Aroライブラリによるパース）
3. C AST（抽象構文木）からZig ASTへの変換
4. Zigソースコードの生成と出力

**関連システム・外部連携**：
- Aroライブラリ（`lib/compiler/aro/`）: Cプリプロセッサ・パーサー
- Clang（オプション）: 代替のCフロントエンドとして使用可能
- ファイルシステムI/O
- `@cImport`ビルトイン経由での内部呼び出し

**権限による制御**：特に権限制御は実装されていない。ファイルシステムのアクセス権限に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | C言語変換画面 | 主機能 | CコードをZigコードに変換する処理 |

## 機能種別

言語変換処理 / コード生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| file | []const u8 | Yes | 変換対象のCソースファイル | 有効なCソースファイルであること |
| -target | []const u8 | No | ターゲットプラットフォーム指定 | 有効なターゲットトリプル |
| -I | [][]const u8 | No | インクルードパスの追加 | - |
| -D | [][]const u8 | No | プリプロセッサマクロの定義 | - |
| -isystem | [][]const u8 | No | システムインクルードパスの追加 | - |
| --libc | []const u8 | No | libcパスファイルの指定 | - |
| -x | enum | No | 入力言語の明示的指定（c/c-header等） | - |
| --color | enum | No | カラー出力の制御（auto/off/on） | - |

### 入力データソース

- ファイルシステム上のCソースファイル（`.c`, `.h`）
- Cプリプロセッサディレクティブで参照されるヘッダーファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| zig_source | []const u8 | 変換されたZigソースコード |
| error_messages | ErrorBundle | 変換エラー・警告メッセージ |

### 出力先

- 標準出力（変換されたZigソースコード）
- 標準エラー出力（エラー・警告メッセージ）
- ファイル（`-femit-bin`オプション指定時）

## 処理フロー

### 処理シーケンス

```
1. コマンドライン引数の解析
   └─ ターゲット、インクルードパス、マクロ定義等の取得
2. Cコンパイラ環境の構築
   └─ Aroコンパイレーションの初期化
   └─ ターゲット固有の設定
3. プリプロセス処理
   └─ マクロ展開
   └─ ヘッダーファイルのインクルード解決
4. C構文解析
   └─ Aroパーサーによるトークン化
   └─ C AST構築
5. Zig変換処理
   └─ C型からZig型への変換
   └─ C構文からZig構文への変換
   └─ 未サポート構文のフォールバック処理
6. Zigソースコード生成
   └─ Zig ASTの正規化
   └─ テキスト出力
7. 結果出力
   └─ 標準出力またはファイルへ出力
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数解析]
    B --> C[Aro環境構築]
    C --> D[プリプロセス]
    D --> E{構文エラー?}
    E -->|Yes| F[エラー出力]
    E -->|No| G[C AST構築]
    G --> H[Zig変換]
    H --> I{変換エラー?}
    I -->|Yes| J[警告付きで続行]
    I -->|No| K[Zigソース生成]
    J --> K
    K --> L[出力]
    F --> M[終了]
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 型マッピング | Cの基本型をZigの対応する型に変換 | 全ての型宣言 |
| BR-02 | ポインタ変換 | Cポインタは`[*c]T`または`?*T`に変換 | ポインタ型 |
| BR-03 | 配列変換 | C配列は`[N]T`に変換 | 配列型 |
| BR-04 | 関数変換 | C関数はZig関数宣言に変換 | 関数定義・宣言 |
| BR-05 | マクロ変換 | 単純なマクロはZig定数に変換 | プリプロセッサマクロ |
| BR-06 | 未サポート構文 | 変換不可能な構文はコメントまたはopaque型として出力 | 複雑なマクロ等 |

### 型マッピング表

| C型 | Zig型 |
|-----|-------|
| char | c_char |
| short | c_short |
| int | c_int |
| long | c_long |
| long long | c_longlong |
| unsigned char | u8 |
| unsigned short | c_ushort |
| unsigned int | c_uint |
| unsigned long | c_ulong |
| unsigned long long | c_ulonglong |
| float | f32 |
| double | f64 |
| void* | ?*anyopaque |
| T* | [*c]T または ?*T |

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 構文エラー | Cソースに構文エラーがある場合 | Aroの診断メッセージを出力 |
| - | 変換エラー | 変換不可能なC構文がある場合 | 警告を出力し、opaque型またはコメントで代替 |
| - | ファイルエラー | 入力ファイルが見つからない場合 | エラーメッセージを出力して終了 |
| - | インクルードエラー | ヘッダーファイルが見つからない場合 | エラーメッセージを出力 |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能はトランザクション処理を必要としない。

## パフォーマンス要件

- 大規模なCヘッダーファイル（システムヘッダー等）を効率的に処理
- `@cImport`での使用時はキャッシュ機構を活用

## セキュリティ考慮事項

- 入力Cファイルの内容は信頼されたソースからのものであることが前提
- 生成されたZigコードは手動レビューを推奨

## 備考

- AroはZig言語で書かれた自己完結型のCコンパイラフロントエンド
- Clangを使用する代替モードも存在（`-fclang`オプション）
- 完全な変換は保証されず、手動での調整が必要な場合がある
- `@cImport`ビルトインは内部的にtranslate-c機能を使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、translate-c処理で使用される主要なデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Compilation.zig | `src/Compilation.zig` | CImportResult構造体: 変換結果を保持 |
| 1-2 | Tree.zig | `lib/compiler/aro/aro/Tree.zig` | Aroの構文木構造: C言語のAST表現 |

**読解のコツ**: translate-cはAro（C言語パーサー）とZigのAST生成の橋渡しを行う。Aroの`Tree`がCのAST、`std.zig.Ast`がZigのASTである。

#### Step 2: エントリーポイントを理解する

処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | main.zig | `src/main.zig` | **306-308行目**: `zig translate-c`コマンドのディスパッチ。buildOutputTypeに`.translate_c`として渡される |
| 2-2 | Compilation.zig | `src/Compilation.zig` | **5750-5883行目**: cImportByPath関数。外部translate-c処理のエントリーポイント |
| 2-3 | Compilation.zig | `src/Compilation.zig` | **5887-5994行目**: cImport関数。`@cImport`ビルトイン用のエントリーポイント |

**主要処理フロー**:
1. **5753行目**: dev.check(.translate_c_command)で機能チェック
2. **5757-5762行目**: 一時ファイル名の生成
3. **5764-5850行目**: 子プロセスでのtranslate-c実行
4. **5852-5868行目**: 結果の読み取りと解析

#### Step 3: Aroライブラリを理解する

Cプリプロセッサ・パーサーの処理を読み解く。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | aro.zig | `lib/compiler/aro/aro.zig` | Aroモジュールの公開インターフェース |
| 3-2 | Compilation.zig | `lib/compiler/aro/aro/Compilation.zig` | Aroのコンパイレーション管理 |
| 3-3 | Preprocessor.zig | `lib/compiler/aro/aro/Preprocessor.zig` | Cプリプロセッサの実装 |
| 3-4 | Parser.zig | `lib/compiler/aro/aro/Parser.zig` | Cパーサーの実装 |

**主要処理フロー**:
- Preprocessor: マクロ展開、条件コンパイル、インクルード処理
- Parser: トークン列からAST構築

#### Step 4: 変換処理を理解する

C ASTからZigコードへの変換処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Tree.zig | `lib/compiler/aro/aro/Tree.zig` | C AST構造とノード種別 |
| 4-2 | TypeStore.zig | `lib/compiler/aro/aro/TypeStore.zig` | C型システムの表現 |

### プログラム呼び出し階層図

```
main.zig::mainArgs
    │
    └─ buildOutputType (output_mode = .translate_c)
           │
           └─ Compilation処理
                  │
                  ├─ cImportByPath (ファイルからの変換)
                  │      │
                  │      ├─ 子プロセス生成
                  │      │      └─ translate-c実行
                  │      │
                  │      └─ 結果読み取り
                  │
                  └─ cImport (@cImport経由)
                         │
                         ├─ キャッシュ確認
                         │
                         └─ Aro処理
                                │
                                ├─ Preprocessor.zig
                                │      └─ マクロ展開・インクルード
                                │
                                ├─ Parser.zig
                                │      └─ C AST構築
                                │
                                └─ Zig変換・出力
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Cソースファイル ───▶ Preprocessor ───▶ 展開済みソース
                           │
                           ▼
                        Parser
                           │
                           ▼
                        C AST
                           │
                           ▼
                    Zig変換処理
                           │
                           ▼
                   Zigソースコード ───▶ 標準出力/ファイル
                           │
                           ▼
                   エラー/警告 ───▶ 標準エラー出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.zig | `src/main.zig` | ソース | CLIエントリーポイント |
| Compilation.zig | `src/Compilation.zig` | ソース | コンパイレーション管理、cImport処理 |
| aro.zig | `lib/compiler/aro/aro.zig` | ソース | Aroモジュール公開インターフェース |
| Compilation.zig | `lib/compiler/aro/aro/Compilation.zig` | ソース | Aroコンパイレーション管理 |
| Preprocessor.zig | `lib/compiler/aro/aro/Preprocessor.zig` | ソース | Cプリプロセッサ |
| Parser.zig | `lib/compiler/aro/aro/Parser.zig` | ソース | Cパーサー |
| Tree.zig | `lib/compiler/aro/aro/Tree.zig` | ソース | C AST定義 |
| TypeStore.zig | `lib/compiler/aro/aro/TypeStore.zig` | ソース | C型システム |
| Driver.zig | `lib/compiler/aro/aro/Driver.zig` | ソース | Aroドライバー（オプション処理） |
