# 機能設計書 15-c++

## 概要

本ドキュメントは、Zigコンパイラに組み込まれたドロップインC++コンパイラ機能「c++」の設計仕様を記載する。`zig c++`コマンドは、ZigをG++/Clang++互換のC++コンパイラとして使用するためのインターフェースを提供する。

### 本機能の処理概要

**業務上の目的・背景**：多くの既存プロジェクトはC++コンパイラ（g++, clang++等）を前提としたビルドシステムを使用している。Zigは高品質なクロスコンパイル機能と、バンドルされたC++標準ライブラリ（libc++）を持つため、環境構築の煩雑さなく任意のターゲット向けにC++コードをコンパイルできる。本機能は、既存のビルドシステム（Make, CMake, Autotools等）を変更することなく、環境変数`CXX=zig c++`を設定するだけでZigのクロスコンパイル機能を活用できるようにする。

**機能の利用シーン**：
- クロスコンパイル環境を簡単にセットアップしたい場合
- プロジェクトのC++コンパイラをZig c++に置き換えてビルドを高速化したい場合
- 特定のC++標準バージョン（C++11, C++14, C++17, C++20等）でビルドしたい場合
- 静的リンク環境でC++プロジェクトをビルドしたい場合
- CI/CD環境で一貫したビルド環境を構築したい場合

**主要な処理内容**：
1. G++/Clang++互換のコマンドライン引数を解析
2. ターゲットプラットフォームの決定
3. 必要に応じてバンドルされたlibc++を使用
4. Clangバックエンドを介したC++コンパイル
5. リンク処理（C++標準ライブラリの自動リンク含む）

**関連システム・外部連携**：
- Clangバックエンド: 実際のC++コンパイル処理
- バンドルされたlibc++（C++標準ライブラリ）
- libunwind（スタック巻き戻し）
- LLDリンカ

**権限による制御**：特に権限制御は実装されていない。ファイルシステムのアクセス権限に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 15 | C++コンパイラ画面 | 主機能 | Zigをドロップイン C++コンパイラとして使用する処理 |

## 機能種別

コンパイル処理 / ドロップイン互換ツール

## 入力仕様

### 入力パラメータ

G++/Clang++互換の引数をサポート。主要なオプションを以下に示す。

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| source files | []const u8 | Yes | コンパイル対象のC++ソースファイル | .cpp, .cxx, .cc, .C, .c++, .hpp, .hxx等 |
| -o | []const u8 | No | 出力ファイル名 | - |
| -c | フラグ | No | コンパイルのみ（リンクしない） | - |
| -target | []const u8 | No | ターゲットトリプル | 有効なターゲット指定 |
| -I | [][]const u8 | No | インクルードパス | - |
| -L | [][]const u8 | No | ライブラリ検索パス | - |
| -l | [][]const u8 | No | リンクするライブラリ | - |
| -D | [][]const u8 | No | プリプロセッサマクロ定義 | - |
| -O | enum | No | 最適化レベル（0/1/2/3/s/z） | - |
| -g | フラグ | No | デバッグ情報生成 | - |
| -std | []const u8 | No | C++言語標準バージョン | c++11, c++14, c++17, c++20, c++23等 |
| -stdlib | []const u8 | No | 使用するC++標準ライブラリ | libc++, libstdc++ |
| -fexceptions | フラグ | No | 例外サポート有効化 | - |
| -frtti | フラグ | No | RTTI有効化 | - |
| -shared | フラグ | No | 共有ライブラリ生成 | - |
| -static | フラグ | No | 静的リンク | - |

### 入力データソース

- C++ソースファイル（.cpp, .cxx, .cc, .C, .c++）
- C++ヘッダーファイル（.hpp, .hxx, .h）
- Cソースファイル（.c）
- アセンブリファイル（.s, .S）
- オブジェクトファイル（.o）
- 静的ライブラリ（.a）
- 動的ライブラリ（.so, .dylib, .dll）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| executable | バイナリ | 実行可能ファイル |
| object_file | バイナリ | オブジェクトファイル（-c指定時） |
| shared_library | バイナリ | 共有ライブラリ（-shared指定時） |
| static_library | バイナリ | 静的ライブラリ |

### 出力先

- ファイルシステム（-oで指定されたパス）

## 処理フロー

### 処理シーケンス

```
1. コマンドライン引数の解析
   └─ G++/Clang++互換オプションの解釈
2. ターゲット決定
   └─ -target指定または自動検出
3. C++標準ライブラリ選択
   └─ libc++（バンドル）または libstdc++
4. プリプロセス
   └─ マクロ展開、テンプレートインスタンス化準備
5. コンパイル
   └─ Clangバックエンドによるコード生成
6. リンク（-c未指定時）
   └─ C++ランタイムのリンク
   └─ LLDまたはシステムリンカによるリンク
7. 出力
   └─ 指定された形式で出力
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数解析]
    B --> C[ターゲット決定]
    C --> D[C++標準ライブラリ選択]
    D --> E{-E指定?}
    E -->|Yes| F[プリプロセスのみ]
    E -->|No| G[プリプロセス]
    G --> H{-S指定?}
    H -->|Yes| I[アセンブリ出力]
    H -->|No| J[コンパイル]
    J --> K{-c指定?}
    K -->|Yes| L[オブジェクト出力]
    K -->|No| M[リンク]
    M --> N[実行可能ファイル出力]
    F --> O[終了]
    I --> O
    L --> O
    N --> O
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | G++互換性 | G++と同等のコマンドラインオプションをサポート | 全実行 |
| BR-02 | クロスコンパイル | -targetでクロスコンパイルターゲットを指定可能 | -target指定時 |
| BR-03 | libc++バンドル | デフォルトでバンドルされたlibc++を使用 | C++標準ライブラリ使用時 |
| BR-04 | 例外サポート | デフォルトで例外サポート有効 | -fno-exceptions未指定時 |
| BR-05 | RTTIサポート | デフォルトでRTTI有効 | -fno-rtti未指定時 |
| BR-06 | libunwind統合 | スタック巻き戻しにlibunwindを使用 | 例外使用時 |

### サポートされるC++標準

| 標準 | フラグ |
|------|--------|
| C++11 | -std=c++11 |
| C++14 | -std=c++14 |
| C++17 | -std=c++17 |
| C++20 | -std=c++20 |
| C++23 | -std=c++23 |

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | コンパイルエラー | C++ソースに構文/意味エラー | Clangのエラーメッセージを出力 |
| - | テンプレートエラー | テンプレートインスタンス化エラー | 詳細なエラートレースを出力 |
| - | リンクエラー | シンボル未解決等 | リンカのエラーメッセージを出力 |
| - | ABI互換エラー | 異なるABI間でのリンク | 警告メッセージを出力 |

### リトライ仕様

リトライ機能は実装されていない。

## トランザクション仕様

本機能はトランザクション処理を必要としない。

## パフォーマンス要件

- 大規模C++プロジェクトのビルドに対応
- テンプレートの効率的なインスタンス化

## セキュリティ考慮事項

- 入力ソースファイルは信頼されたソースからのものであることが前提
- C++の複雑な機能（テンプレート、例外等）は注意して使用

## 備考

- `zig c++`は`zig cc`のC++版で、内部的にClangを使用
- libc++とlibunwindがバンドルされており、追加インストール不要
- 環境変数`CXX=zig c++`でMake等から利用可能
- CMakeでは`-DCMAKE_CXX_COMPILER=zig c++`で設定可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、c++処理で使用される主要なデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Compilation.zig | `src/Compilation.zig` | Config構造体: コンパイル設定を保持 |
| 1-2 | main.zig | `src/main.zig` | ArgMode: コマンドモードの列挙（.cpp） |

**読解のコツ**: `zig c++`は内部的に`buildOutputType`関数に`.cpp`モードとして渡される。

#### Step 2: エントリーポイントを理解する

処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | main.zig | `src/main.zig` | **303-305行目**: `zig c++`コマンドのディスパッチ。buildOutputTypeに`.cpp`として渡される |

**主要処理フロー**:
1. **303-305行目**: dev.check(.cc_command)で機能チェック（ccと共通）
2. **305行目**: buildOutputType(gpa, arena, io, args, .cpp, environ_map)を呼び出し

#### Step 3: C++ランタイムを理解する

バンドルされたC++ランタイムライブラリを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | libcxx/ | `lib/libcxx/` | バンドルされたlibc++実装 |
| 3-2 | libunwind/ | `lib/libunwind/` | スタック巻き戻しライブラリ |

### プログラム呼び出し階層図

```
main.zig::mainArgs
    │
    └─ buildOutputType (.cpp モード)
           │
           ├─ G++/Clang++互換引数解析
           │
           ├─ ターゲット決定
           │
           ├─ libc++/libunwind選択
           │
           └─ Clangバックエンド
                  │
                  ├─ プリプロセス
                  ├─ コンパイル
                  └─ リンク (LLD + libc++ + libunwind)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

C++ソースファイル ───▶ 引数解析 ───▶ ターゲット決定
                            │
                            ▼
                     libc++/libunwind選択
                            │
                            ▼
                     Clangプリプロセス
                            │
                            ▼
                     Clangコンパイル
                            │
                            ▼
                     LLDリンク ───▶ 実行可能ファイル
                            │
                            └───▶ オブジェクトファイル（-c時）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.zig | `src/main.zig` | ソース | CLIエントリーポイント、引数解析 |
| Compilation.zig | `src/Compilation.zig` | ソース | コンパイレーション管理 |
| dev.zig | `src/dev.zig` | ソース | 機能フラグ管理（cc_command） |
| libcxx/ | `lib/libcxx/` | ディレクトリ | C++標準ライブラリ（libc++） |
| libunwind/ | `lib/libunwind/` | ディレクトリ | スタック巻き戻しライブラリ |
