# 機能設計書 29-LLVMバックエンド

## 概要

本ドキュメントは、ZigコンパイラのLLVMバックエンド機能について記述する。この機能は、ZigのAIR（Abstract Intermediate Representation）をLLVM IRに変換し、LLVMの最適化パイプラインとコード生成機能を利用してネイティブマシンコードを生成する。

### 本機能の処理概要

**業務上の目的・背景**：
Zigは独自のコンパイラインフラストラクチャを持つが、高品質なネイティブコード生成のためにLLVMをバックエンドとして利用できる。LLVMは成熟した最適化パイプラインと幅広いターゲットアーキテクチャのサポートを提供し、Zigコンパイラに産業グレードのコード生成能力を与える。LLVMバックエンドは、パフォーマンスが重要なリリースビルドや、Zigネイティブバックエンドがまだサポートしていないターゲットへのコンパイルに使用される。

**機能の利用シーン**：
- リリースビルドでの最適化されたコード生成
- LLVM特有の最適化（LTO: Link Time Optimization）の利用
- Zigネイティブバックエンドがサポートしていないターゲット向けビルド
- LLVM IRやビットコードの出力
- Cコードとの相互運用（Clang互換のABI）

**主要な処理内容**：
1. ターゲットトリプルの生成（LLVMフォーマット）
2. データレイアウトの決定
3. AIRからLLVM IRへの変換
4. LLVM最適化パイプラインの実行
5. マシンコード生成
6. デバッグ情報（DWARF）の生成

**関連システム・外部連携**：
- LLVM C API（llvm/bindings.zig経由）
- リンカ（ELF、Mach-O、COFF形式のオブジェクト出力）

**権限による制御**：
本機能は特定の権限制御を持たない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 実行ファイルビルド画面 | 遷移先機能 | LLVMを使用したネイティブコード生成処理 |
| 4 | ライブラリビルド画面 | 遷移先機能 | LLVMを使用したライブラリコード生成処理 |
| 5 | オブジェクトビルド画面 | 遷移先機能 | LLVMを使用したオブジェクトコード生成処理 |

## 機能種別

コード生成処理 / 最適化処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| -fllvm | フラグ | No | LLVMバックエンドを強制使用 | - |
| -fno-llvm | フラグ | No | LLVMバックエンドを使用しない | - |
| -flto | フラグ | No | Link Time Optimizationを有効化 | - |
| -femit-llvm-ir | フラグ | No | LLVM IRを.llファイルに出力 | - |
| -femit-llvm-bc | フラグ | No | LLVMビットコードを.bcファイルに出力 | - |
| -O | 最適化レベル | No | Debug/ReleaseFast/ReleaseSafe/ReleaseSmall | - |

### 入力データソース

- AIR（Zig Abstract Intermediate Representation）
- ターゲット情報（アーキテクチャ、OS、ABI）
- 最適化設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| オブジェクトファイル | バイナリ | ネイティブマシンコード（.o/.obj） |
| LLVM IR | テキスト | .llファイル（-femit-llvm-ir指定時） |
| LLVMビットコード | バイナリ | .bcファイル（-femit-llvm-bc指定時） |

### 出力先

- 指定された出力ディレクトリまたはキャッシュディレクトリ

## 処理フロー

### 処理シーケンス

```
1. ターゲット情報の準備
   └─ targetTriple(): LLVMターゲットトリプルの生成
   └─ dataLayout(): データレイアウトの決定

2. LLVMモジュールの初期化
   └─ LLVMコンテキストの作成
   └─ モジュールの作成

3. AIRからLLVM IRへの変換
   └─ 関数ごとにLLVM関数を生成
   └─ 基本ブロックの構築
   └─ 命令の変換

4. 最適化パイプラインの実行
   └─ パスマネージャの設定
   └─ 最適化パスの実行

5. コード生成
   └─ マシンコード生成
   └─ オブジェクトファイル出力

6. デバッグ情報の生成（デバッグビルド時）
   └─ DWARF情報の生成
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[ターゲット情報準備]
    B --> C[LLVMモジュール初期化]
    C --> D[AIR取得]
    D --> E[LLVM IR生成]
    E --> F{最適化レベル?}
    F -->|Debug| G[最小最適化]
    F -->|Release| H[フル最適化]
    G --> I[コード生成]
    H --> I
    I --> J{出力形式?}
    J -->|オブジェクト| K[.o/.obj出力]
    J -->|LLVM IR| L[.ll出力]
    J -->|ビットコード| M[.bc出力]
    K --> N[終了]
    L --> N
    M --> N
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | LLVMサポート確認 | ターゲットがLLVMでサポートされているか確認 | 常時 |
| BR-02 | データレイアウト一貫性 | Clangと同じデータレイアウトを使用 | 常時 |
| BR-03 | テールコール制限 | 一部アーキテクチャではテールコール最適化不可 | MIPS、PowerPC等 |
| BR-04 | ABI互換性 | C言語とのABI互換性を維持 | 常時 |

### 計算ロジック

**テールコールサポート判定**:
```zig
pub fn supportsTailCall(target: *const std.Target) bool {
    return switch (target.cpu.arch) {
        .wasm32, .wasm64 => target.cpu.has(.wasm, .tail_call),
        .mips, .mipsel, .mips64, .mips64el => false,
        .powerpc, .powerpcle, .powerpc64, .powerpc64le => false,
        else => true,
    };
}
```

## データベース操作仕様

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| CodegenFail | コード生成エラー | LLVM IR生成失敗 | エラーメッセージを確認 |
| OutOfMemory | メモリエラー | メモリ不足 | 利用可能メモリを増やす |
| - | ターゲットエラー | LLVMがターゲットをサポートしない | 別のバックエンドを使用 |

### リトライ仕様

本機能にリトライ機構は存在しない。

## トランザクション仕様

本機能はトランザクションを使用しない。

## パフォーマンス要件

- ReleaseFastビルドでは最大限の最適化を実行
- コンパイル時間はLLVM最適化レベルに依存
- LTOは追加のリンク時間が必要

## セキュリティ考慮事項

- スタック保護オプションのサポート
- アドレス空間レイアウトランダム化（ASLR）対応コード生成

## 備考

- LLVMバックエンドはbuild_options.have_llvmがtrueの場合のみ利用可能
- 一部のターゲットはLLVMでのみサポート
- LLVM APIはllvm/bindings.zigを通じてアクセス

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

LLVMバックエンドの主要なデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | llvm.zig | `src/codegen/llvm.zig` | Builder型、Error型の定義 |

**読解のコツ**:
- std.zig.llvm.Builderが主要なIR構築インターフェース
- CodegenFailとOutOfMemoryが主なエラー型

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | llvm.zig | `src/codegen/llvm.zig` | targetTriple関数（58-324行目） |
| 2-2 | llvm.zig | `src/codegen/llvm.zig` | dataLayout関数（336-493行目） |

**主要処理フロー**:
1. **58-127行目**: アーキテクチャ名のLLVMフォーマットへの変換
2. **131-183行目**: サブアーキテクチャの追加（ARM、SPIRVなど）
3. **210-252行目**: OS名のLLVMフォーマットへの変換
4. **272-305行目**: ABI名のLLVMフォーマットへの変換
5. **336-493行目**: ターゲットごとのデータレイアウト文字列

#### Step 3: 機能サポート判定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | llvm.zig | `src/codegen/llvm.zig` | supportsTailCall関数（326-334行目） |
| 3-2 | llvm.zig | `src/codegen/llvm.zig` | legalizeFeatures関数（43-48行目） |

**主要処理フロー**:
- **326-334行目**: テールコールサポートの判定（MIPS、PowerPCは非サポート）
- **43-48行目**: AIRの合法化フィーチャー定義

### プログラム呼び出し階層図

```
Compilation
    │
    ├─ codegen選択
    │      │
    │      └─ LLVMバックエンド（-fllvm指定時またはデフォルト）
    │             │
    │             ├─ targetTriple() [ターゲットトリプル生成]
    │             │      ├─ アーキテクチャ変換
    │             │      ├─ OS変換
    │             │      └─ ABI変換
    │             │
    │             ├─ dataLayout() [データレイアウト取得]
    │             │
    │             ├─ LLVM IR生成
    │             │      ├─ 関数生成
    │             │      ├─ 基本ブロック生成
    │             │      └─ 命令変換
    │             │
    │             ├─ 最適化パイプライン
    │             │
    │             └─ マシンコード生成
```

### データフロー図

```
[入力]                    [処理]                         [出力]

AIR ────────────────────▶ LLVM IR生成
                              │
                              ▼
ターゲット情報 ──────────▶ targetTriple()
  arch, os, abi               │
                              ▼
                         dataLayout() ───────────────▶ データレイアウト文字列
                              │
                              ▼
最適化設定 ──────────────▶ 最適化パイプライン
  -O Debug/Release            │
                              ▼
                         コード生成 ─────────────────▶ オブジェクトファイル
                              │                        .o / .obj
                              │
                              ├─────────────────────▶ LLVM IR
                              │                        .ll
                              │
                              └─────────────────────▶ ビットコード
                                                       .bc
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| llvm.zig | `src/codegen/llvm.zig` | ソース | LLVMバックエンドのメイン実装 |
| bindings.zig | `src/codegen/llvm/bindings.zig` | ソース | LLVM C APIのZigバインディング |
| Air.zig | `src/Air.zig` | ソース | Abstract IR定義 |
| codegen.zig | `src/codegen.zig` | ソース | コード生成の共通インターフェース |
| Compilation.zig | `src/Compilation.zig` | ソース | コンパイル全体の管理 |
| target.zig | `src/target.zig` | ソース | ターゲット関連ユーティリティ |

### サポートアーキテクチャ

LLVMバックエンドは以下のアーキテクチャをサポート:
- x86, x86_64
- aarch64, aarch64_be, arm, armeb, thumb, thumbeb
- mips, mipsel, mips64, mips64el
- powerpc, powerpcle, powerpc64, powerpc64le
- riscv32, riscv64
- sparc, sparc64
- wasm32, wasm64
- その他多数

**注意**: 以下のアーキテクチャはLLVMでサポートされない:
- alpha, arceb, hppa, hppa64, kalimba, kvx, microblaze, microblazeel, or1k, propeller, sh, sheb, x86_16, xtensaeb
