# 機能設計書 40-WASMリンカ

## 概要

本ドキュメントは、ZigコンパイラにおけるWebAssemblyモジュールのリンカ機能の設計を記述する。

### 本機能の処理概要

WASMリンカは、コンパイラが生成したWebAssemblyオブジェクトファイルを結合し、最終的なWebAssemblyモジュール（.wasm）を生成する機能を提供する。WebAssemblyは、Webブラウザやサーバーサイド環境（WASI）で実行されるポータブルなバイナリ形式である。

**業務上の目的・背景**：WebAssemblyは、Webプラットフォームの標準技術として広く採用されており、高性能なWebアプリケーションやサーバーレス環境でのコード実行に使用される。Zigがネイティブでリンカを実装することで、(1) LLVMのwasm-ldへの依存を排除し、(2) Zig固有の最適化を適用し、(3) インクリメンタルコンパイルをサポートできる。

**機能の利用シーン**：
- Webブラウザ向けWebAssemblyモジュールの生成
- WASI環境向けモジュールの生成
- Cloudflare Workers等のエッジコンピューティング向けモジュール
- ブラウザ拡張機能向けモジュール

**主要な処理内容**：
1. 入力ファイル（WebAssemblyオブジェクト、アーカイブ）の解析
2. シンボル解決（関数、グローバル、テーブル、メモリ）
3. 関数インポート/エクスポートの解決
4. データセグメントの配置
5. 関数型のインターン化
6. 間接呼び出しテーブルの構築
7. 初期化関数（__wasm_call_ctors）の生成
8. カスタムセクションの処理

**関連システム・外部連携**：wasmバックエンドから生成されたMIRを受け取り、最終的なWebAssemblyバイナリを出力する。

**権限による制御**：特になし（コンパイル時の処理）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 実行ファイルビルド画面 | 遷移先機能 | WebAssemblyモジュールのリンク処理 |

## 機能種別

リンク処理 / バイナリ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| comp | *Compilation | Yes | コンパイル単位 | 有効なCompilationであること |
| emit | Path | Yes | 出力パス | 有効なパスであること |
| options | - | Yes | リンクオプション | - |

### 入力データソース

- Zigコンパイラが生成したWASMオブジェクト
- 外部WebAssemblyオブジェクトファイル（.o）
- アーカイブ（.a）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| string_bytes | ArrayList(u8) | 文字列バイト列 |
| func_types | AutoArrayHashMap | 関数型マップ |
| object_functions | ArrayList | オブジェクト関数リスト |
| object_globals | ArrayList | オブジェクトグローバルリスト |
| object_data_segments | ArrayList | データセグメントリスト |

### 出力先

WebAssemblyモジュール（.wasm）

## 処理フロー

### 処理シーケンス

```
1. 初期化フェーズ
   └─ 文字列テーブル、関数型マップの初期化
2. prelink（事前リンク）
   └─ オブジェクトファイルの解析、シンボル収集
3. シンボル解決
   └─ 関数、グローバル、テーブル、メモリのインポート/エクスポート解決
4. 関数型インターン化
   └─ 同一関数型の統合
5. 間接呼び出しテーブル構築
   └─ call_indirect用のテーブルエントリ生成
6. データセグメント配置
   └─ メモリ上へのデータ配置
7. flush（書き出し）
   └─ WebAssemblyバイナリの生成
```

### フローチャート

```mermaid
flowchart TD
    A[開始: 入力ファイル] --> B[初期化]
    B --> C[prelink処理]
    C --> D[シンボル解決]
    D --> E{未定義シンボル?}
    E -->|Yes| F[インポート/エラー]
    E -->|No| G[関数型インターン化]
    G --> H[間接呼び出しテーブル構築]
    H --> I[データセグメント配置]
    I --> J[flush処理]
    J --> K[WASMファイル出力]
    K --> L[終了]
    F --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-1 | 関数型共有 | 同一シグネチャの関数型は共有される | 全関数 |
| BR-40-2 | メモリ初期化 | 初期メモリサイズはglobal_baseとstack_sizeから計算 | メモリセクション生成時 |
| BR-40-3 | 初期化関数呼び出し | __wasm_call_ctorsで初期化関数を順次呼び出し | 起動時 |

### 計算ロジック

- 初期メモリ: global_base + データサイズ + スタックサイズ
- テーブルインデックス: 間接呼び出し可能関数に連番割り当て

## データベース操作仕様

該当なし（データベース操作は行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| undefined symbol | リンクエラー | 未定義シンボル | インポートまたはエラー |
| duplicate symbol | リンクエラー | 重複シンボル | エラー報告 |

### リトライ仕様

リトライ機構は実装されていない

## トランザクション仕様

該当なし

## パフォーマンス要件

- LEB128エンコーディングの最適化
- 関数型インターン化によるサイズ削減

## セキュリティ考慮事項

- WebAssemblyのサンドボックス特性を活用
- メモリ境界チェックはランタイムで実施

## 備考

Flush構造体でリンク処理の実際の出力を管理。prelinkで収集した情報をflushで出力に変換。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、WASMリンカの主要データ構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Wasm.zig | `src/link/Wasm.zig` | 主要フィールド定義 |

**読解のコツ**: WASMはセクションベースの構造を持つ。Type/Function/Table/Memory/Global/Export/Element/Code/Dataセクションの役割を理解する。

#### Step 2: エントリーポイントを理解する

リンカの主要構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Wasm.zig | `src/link/Wasm.zig` | 主要フィールド（47-300行目） |

**主要処理フロー**:
1. **47-80行目**: 基本フィールド（string_bytes、entry_name、import_symbols等）
2. **83-100行目**: 関数型とインポート管理
3. **107-135行目**: リロケーションとfixup管理
4. **177-210行目**: Zcu関連データ（navs_obj/exe、uavs_obj/exe）
5. **280-300行目**: MIR関連データ（mir_instructions、mir_extra、mir_locals）

#### Step 3: オブジェクト管理を理解する

入力オブジェクトファイルの管理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Wasm/Object.zig | `src/link/Wasm/Object.zig` | Object構造体 |

#### Step 4: Flush処理を理解する

最終出力を生成するFlush処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Wasm/Flush.zig | `src/link/Wasm/Flush.zig` | Flush構造体 |

### プログラム呼び出し階層図

```
link.zig (リンカエントリ)
    │
    ├─ Wasm.zig
    │      ├─ prelink() - 事前リンク
    │      ├─ flush() - 書き出し
    │      │
    │      ├─ string_bytes/string_table
    │      │      └─ 文字列管理
    │      │
    │      ├─ func_types
    │      │      └─ 関数型インターン化
    │      │
    │      ├─ object_*
    │      │      ├─ object_functions
    │      │      ├─ object_globals
    │      │      ├─ object_tables
    │      │      ├─ object_memories
    │      │      └─ object_data_segments
    │      │
    │      ├─ *_imports
    │      │      ├─ function_imports
    │      │      ├─ global_imports
    │      │      └─ table_imports
    │      │
    │      ├─ zcu_*
    │      │      ├─ navs_obj/exe
    │      │      ├─ uavs_obj/exe
    │      │      └─ zcu_funcs
    │      │
    │      └─ mir_*
    │             ├─ mir_instructions
    │             ├─ mir_extra
    │             └─ mir_locals
    │
    ├─ Wasm/Object.zig
    │      └─ Object構造体
    │
    ├─ Wasm/Archive.zig
    │      └─ アーカイブ処理
    │
    └─ Wasm/Flush.zig
           └─ 出力処理
```

### データフロー図

```
[入力]               [処理]                    [出力]

ZigObject ───┐
             │
Object   ────┼──▶ prelink処理 ───▶ シンボル解決 ───▶ WASMファイル
             │         │                   │
Archive  ────┘         ▼                   ▼
                  関数型収集          データ配置
                       │                   │
                       ▼                   ▼
                  インターン化 ───▶ flush処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Wasm.zig | `src/link/Wasm.zig` | ソース | メインWASMリンカロジック |
| Wasm/Object.zig | `src/link/Wasm/Object.zig` | ソース | オブジェクトファイル処理 |
| Wasm/Archive.zig | `src/link/Wasm/Archive.zig` | ソース | アーカイブ処理 |
| Wasm/Flush.zig | `src/link/Wasm/Flush.zig` | ソース | 出力処理 |
