# 機能設計書 44-型システム

## 概要

本ドキュメントは、Zigコンパイラの型システム（Type.zig）について記述する。型と値は共に32ビット整数でInternPoolのインデックスとして表現されるが、Type構造体は型特有の操作を提供するための抽象化レイヤーを提供する。

### 本機能の処理概要

型システムは、Zigの型情報を管理・検証する機能である。すべての型はInternPoolに格納され、Type構造体はそのインデックスをラップして型固有のメソッドを提供する。

**業務上の目的・背景**：Zigは静的型付け言語であり、コンパイル時に完全な型チェックを行う。型システムはこの型チェックの基盤となり、型推論、型比較、型変換の可否判定などを担当する。また、Zigは低レベルシステムプログラミングを目的としているため、正確なメモリレイアウト計算（abiSize、abiAlignment）も重要な機能である。

**機能の利用シーン**：
- 意味解析（Sema）での型チェック
- コード生成でのメモリレイアウト計算
- ランタイム型情報（@TypeOf）の提供
- ジェネリック関数の型パラメータ処理

**主要な処理内容**：
1. 型の等価性判定（eql）
2. 型のフォーマット出力（print）
3. ポインタ型情報の取得（ptrInfo）
4. 配列型情報の取得（arrayInfo）
5. ABIサイズ・アライメント計算
6. 型カテゴリ判定（zigTypeTag）

**関連システム・外部連携**：
- InternPool：型データの実体格納
- Value：型と値の相互変換
- Sema：型チェック処理

**権限による制御**：特になし（コンパイラ内部処理）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 型システムは直接的な画面を持たない内部機能 |

## 機能種別

型管理 / コンパイラ内部処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ip_index | InternPool.Index | Yes | InternPool内の型インデックス | 有効なインデックス |

### 入力データソース

- InternPoolに登録された型データ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| zigTypeTag | TypeId | 型カテゴリ（int、float、pointer等） |
| ptrInfo | PtrType | ポインタ型の詳細情報 |
| arrayInfo | ArrayInfo | 配列型の詳細情報 |
| abiSize | u64 | ABIサイズ（バイト） |
| abiAlignment | Alignment | ABIアライメント |

### 出力先

Sema、codegen等のコンパイラコンポーネント

## 処理フロー

### 処理シーケンス

```
1. Type構造体の作成
   └─ InternPool.IndexをType.ip_indexに格納
2. 型情報の取得
   └─ InternPool.indexToKey()で実データ取得
3. 型カテゴリの判定
   └─ zigTypeTag()でTypeIdを取得
4. 詳細情報の取得
   └─ 型カテゴリに応じたメソッド呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[InternPool.Index] --> B[Type.fromInterned]
    B --> C[Type構造体]
    C --> D{操作種別}
    D -->|カテゴリ判定| E[zigTypeTag]
    D -->|比較| F[eql]
    D -->|サイズ取得| G[abiSize]
    D -->|アライメント| H[abiAlignment]
    D -->|ポインタ情報| I[ptrInfo]
    D -->|配列情報| J[arrayInfo]
    E --> K[TypeId]
    F --> L[bool]
    G --> M[u64]
    H --> N[Alignment]
    I --> O[PtrType]
    J --> P[ArrayInfo]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-44-01 | インターンベース等価性 | 型の等価性はInternPool.Indexの一致で判定 | 全型比較時 |
| BR-44-02 | ポインタサイズ | ポインタサイズはターゲットアーキテクチャ依存 | ポインタ型 |
| BR-44-03 | 配列サイズ | 配列サイズ = 要素サイズ * 要素数 | 配列型 |
| BR-44-04 | アライメント継承 | 構造体アライメントは最大フィールドアライメント以上 | 構造体型 |

### 計算ロジック

**ABIサイズ計算**:
- 基本型: 固定サイズ（i32=4、i64=8等）
- 配列: elem_size * len
- 構造体: フィールドオフセット + パディング計算

**ABIアライメント計算**:
- 基本型: サイズと同じ（最大8バイト）
- 配列: 要素型のアライメント
- 構造体: 最大フィールドアライメント

## データベース操作仕様

該当なし（メモリ上での処理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | 型エラー | 不正な型アクセス | 正しい型メソッド使用 |
| LayoutError | レイアウトエラー | 不完全な型のサイズ取得 | 型解決後にアクセス |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 型比較はO(1)（インデックス比較のみ）
- 型情報取得はO(1)（ハッシュテーブル参照）

## セキュリティ考慮事項

特になし（コンパイラ内部処理）

## 備考

Type.zigは約165KBのソースコードであり、Zigの全型に対する操作を定義している。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 基本構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Type.zig | `src/Type.zig` | Type構造体の定義（1-22行目） |

**読解のコツ**:
- `ip_index: InternPool.Index` - 型のユニークな識別子
- TypeはValueと同じくInternPool.Indexをラップする薄いラッパー
- 型固有のメソッドを提供することで型安全性を確保

**主要処理フロー**:
1. **1-5行目**: ドキュメントコメント
2. **22行目**: `ip_index`フィールド定義
3. **24-26行目**: `zigTypeTag`関数 - 型カテゴリ取得

#### Step 2: 型比較・判定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Type.zig | `src/Type.zig` | eql関数（116-122行目） |
| 2-2 | Type.zig | `src/Type.zig` | isSelfComparable関数（39-76行目） |

**読解のコツ**:
- `eql`はインデックス比較のみで高速
- `isSelfComparable`は型が自己比較可能かを判定

**主要処理フロー**:
- **116-122行目**: InternPool.Indexの一致で等価性判定

#### Step 3: 型情報取得を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Type.zig | `src/Type.zig` | ptrInfo関数（105-114行目） |
| 3-2 | Type.zig | `src/Type.zig` | arrayInfo関数（97-103行目） |

**読解のコツ**:
- InternPool.indexToKey()で実データを取得
- 戻り値は専用の構造体（PtrType、ArrayInfo等）

**主要処理フロー**:
- **91-95行目**: ArrayInfo構造体定義
- **97-103行目**: arrayInfo関数実装
- **105-114行目**: ptrInfo関数実装

#### Step 4: 型フォーマットを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Type.zig | `src/Type.zig` | print関数（160行目以降） |
| 4-2 | Type.zig | `src/Type.zig` | Formatter構造体（130-146行目） |

**読解のコツ**:
- `@typeName`相当の文字列生成
- Formatterはstd.fmtとの統合用

### プログラム呼び出し階層図

```
InternPool (型データ格納)
    │
    ▼
Type.fromInterned(ip_index)
    │
    ├─ zigTypeTag() - TypeId取得
    ├─ eql() - 型等価判定
    ├─ ptrInfo() - ポインタ情報
    ├─ arrayInfo() - 配列情報
    ├─ abiSize() - サイズ計算
    ├─ abiAlignment() - アライメント計算
    └─ print() - 文字列化
          │
          └─ @typeName相当の出力
```

### データフロー図

```
[入力]                      [処理]                        [出力]

InternPool.Index  ──▶  Type.fromInterned()  ──▶  Type構造体
       │                      │                        │
       │                      ▼                        │
       │              InternPool.indexToKey()          │
       │                      │                        │
       │                      ▼                        │
       │              型データ(Key)取得                │
       │                      │                        │
       └──────────────────────┼────────────────────────┘
                              │
                              ▼
                    型固有メソッド実行
                    (ptrInfo, arrayInfo等)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Type.zig | `src/Type.zig` | ソース | 型システムの主実装（約165KB） |
| Value.zig | `src/Value.zig` | ソース | 値システム（型と密接に連携） |
| InternPool.zig | `src/InternPool.zig` | ソース | 型・値データの実体格納 |
| Sema.zig | `src/Sema.zig` | ソース | 型チェックの主要クライアント |
| target.zig | `src/target.zig` | ソース | ターゲット依存の型サイズ情報 |
