# 機能設計書 45-値システム

## 概要

本ドキュメントは、Zigコンパイラの値システム（Value.zig）について記述する。値はInternPoolのインデックスとして表現され、Value構造体は値固有の操作を提供するための抽象化レイヤーである。

### 本機能の処理概要

値システムは、コンパイル時の値（整数、浮動小数点、文字列、関数ポインタ等）を管理する機能である。型システム（Type）と密接に連携し、コンパイル時評価（comptime）の基盤を提供する。

**業務上の目的・背景**：Zigはコンパイル時評価を言語の中核機能として持つ。値システムはこのcomptime機能を実現するために、コンパイル時に完全に解決可能な値を表現・操作する。定数畳み込み、ジェネリクス、コンパイル時関数実行など、多くの機能がこのシステムに依存している。

**機能の利用シーン**：
- コンパイル時定数の評価
- const変数の値追跡
- コンパイル時関数実行結果の保持
- ジェネリック関数の型引数処理

**主要な処理内容**：
1. 値の取得（fromInterned）
2. 整数値への変換（toUnsignedInt、toBigInt）
3. 配列要素の取得（elemValue）
4. 文字列変換（toIpString、toAllocatedBytes）
5. 関数・変数情報の取得（getFunction、getVariable）
6. フォーマット出力（fmtValue）

**関連システム・外部連携**：
- InternPool：値データの実体格納
- Type：型情報との連携
- Sema：コンパイル時評価処理

**権限による制御**：特になし（コンパイラ内部処理）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 値システムは直接的な画面を持たない内部機能 |

## 機能種別

値管理 / コンパイラ内部処理 / コンパイル時評価

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ip_index | InternPool.Index | Yes | InternPool内の値インデックス | 有効なインデックス |

### 入力データソース

- InternPoolに登録された値データ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| unsignedInt | ?u64 | 符号なし整数値 |
| bigInt | BigIntConst | 任意精度整数 |
| function | ?Func | 関数情報 |
| variable | ?Variable | 変数情報 |
| type | Type | 値を型として解釈 |

### 出力先

Sema、codegen等のコンパイラコンポーネント

## 処理フロー

### 処理シーケンス

```
1. Value構造体の作成
   └─ InternPool.IndexをValue.ip_indexに格納
2. 値データの取得
   └─ InternPool.indexToKey()で実データ取得
3. 値種別の判定
   └─ Key構造体のタグで判別
4. 具体値への変換
   └─ 各種変換関数の呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[InternPool.Index] --> B[Value.fromInterned]
    B --> C[Value構造体]
    C --> D{操作種別}
    D -->|整数取得| E[getUnsignedInt]
    D -->|BigInt変換| F[toBigInt]
    D -->|関数情報| G[getFunction]
    D -->|型変換| H[toType]
    D -->|文字列化| I[fmtValue]
    D -->|配列要素| J[elemValue]
    E --> K[?u64]
    F --> L[BigIntConst]
    G --> M[?Func]
    H --> N[Type]
    I --> O[フォーマット出力]
    J --> P[Value]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-45-01 | インターンベース | 値はInternPoolで一意に管理 | 全値操作時 |
| BR-45-02 | 遅延サイズ評価 | lazy_size/lazy_alignは必要時に解決 | サイズ取得時 |
| BR-45-03 | 型としての値 | type型の値はTypeとして解釈可能 | type値の場合 |
| BR-45-04 | undefined値 | undefined値へのアクセスは未定義動作 | undefined検出時 |

### 計算ロジック

**整数値取得（getUnsignedInt）**:
- u64に収まる場合: 直接返却
- 収まらない場合: null返却
- lazy値: 解決後に返却

**BigInt変換（toBigInt）**:
- 任意精度整数への変換
- sema戦略でレイアウト解決が必要な場合あり

## データベース操作仕様

該当なし（メモリ上での処理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| UndefinedAccess | 値エラー | undefined値へのアクセス | 初期化を確認 |
| TypeMismatch | 型エラー | 不正な値アクセス | 正しいメソッド使用 |
| OutOfRange | 範囲エラー | u64に収まらない整数 | toBigInt使用 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 値取得はO(1)（ハッシュテーブル参照）
- BigInt変換は値サイズに比例

## セキュリティ考慮事項

特になし（コンパイラ内部処理）

## 備考

Value.zigは約135KBのソースコードであり、コンパイル時定数の全操作を定義している。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 基本構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Value.zig | `src/Value.zig` | Value構造体とip_indexフィールド（17-19行目） |

**読解のコツ**:
- `ip_index: InternPool.Index` - 値のユニークな識別子
- Type構造体と同じパターン（インデックスラッパー）
- 直接フォーマットは禁止（compileError）

**主要処理フロー**:
1. **19行目**: `ip_index`フィールド定義
2. **21-25行目**: format関数（直接使用禁止）
3. **134-142行目**: fromInterned/toIntern関数

#### Step 2: 整数値操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Value.zig | `src/Value.zig` | getUnsignedInt関数（245-252行目） |
| 2-2 | Value.zig | `src/Value.zig` | toBigInt関数（180-186行目） |

**読解のコツ**:
- `getUnsignedInt`はu64に収まる場合のみ値を返す
- `toBigInt`は任意精度整数に変換
- lazy_size/lazy_alignは遅延評価される

**主要処理フロー**:
- **245-252行目**: getUnsignedInt - u64範囲チェック付き取得
- **180-186行目**: toBigInt - BigIntConstへの変換
- **260-299行目**: getUnsignedIntInner - 実装詳細

#### Step 3: 関数・変数情報取得を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Value.zig | `src/Value.zig` | getFunction関数（229-234行目） |
| 3-2 | Value.zig | `src/Value.zig` | getVariable関数（236-241行目） |

**読解のコツ**:
- InternPool.Key.Func/Variableとして取得
- 該当しない場合はnullを返す

#### Step 4: フォーマット出力を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Value.zig | `src/Value.zig` | fmtValue関数（37-44行目） |
| 4-2 | print_value.zig | `src/print_value.zig` | 実際のフォーマット処理 |

**読解のコツ**:
- fmtValueはprint_value.formatへの委譲
- デバッグ出力用のfmtDebugも提供

### プログラム呼び出し階層図

```
InternPool (値データ格納)
    │
    ▼
Value.fromInterned(ip_index)
    │
    ├─ toIntern() - インデックス取得
    ├─ toType() - 型としての解釈
    ├─ getUnsignedInt() - u64取得
    ├─ toBigInt() - BigInt変換
    ├─ getFunction() - 関数情報
    ├─ getVariable() - 変数情報
    ├─ elemValue() - 配列要素
    └─ fmtValue() - フォーマット出力
          │
          └─ print_value.format()
```

### データフロー図

```
[入力]                      [処理]                        [出力]

InternPool.Index  ──▶  Value.fromInterned()  ──▶  Value構造体
       │                      │                        │
       │                      ▼                        │
       │              InternPool.indexToKey()          │
       │                      │                        │
       │                      ▼                        │
       │              値データ(Key)取得                │
       │                      │                        │
       └──────────────────────┼────────────────────────┘
                              │
                              ▼
                    値固有メソッド実行
                    (getUnsignedInt, toBigInt等)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Value.zig | `src/Value.zig` | ソース | 値システムの主実装（約135KB） |
| Type.zig | `src/Type.zig` | ソース | 型システム（密接に連携） |
| InternPool.zig | `src/InternPool.zig` | ソース | 型・値データの実体格納 |
| print_value.zig | `src/print_value.zig` | ソース | 値のフォーマット出力 |
| mutable_value.zig | `src/mutable_value.zig` | ソース | 可変値の管理 |
| Sema.zig | `src/Sema.zig` | ソース | コンパイル時評価の主要クライアント |
