# 機能設計書 49-std.fs

## 概要

本ドキュメントは、Zig標準ライブラリのファイルシステム操作モジュール（std.fs）について記述する。std.fsはファイルやディレクトリの読み書き、パス操作、ディレクトリ走査など、クロスプラットフォームなファイルシステム操作を提供する。

### 本機能の処理概要

std.fsは、ファイルシステムに対するクロスプラットフォームなインターフェースを提供するモジュールである。現在のZigでは、std.fsはdeprecatedとなり、主要機能はstd.Ioモジュール（特にIo.Dir、Io.File）に移行している。std.Ioはファイルシステムだけでなく、ネットワーク、プロセス、時間、並行処理、メモリマップドファイルなど、すべてのI/O操作を統合したクロスプラットフォームインターフェースである。

**業務上の目的・背景**：アプリケーション開発において、ファイルの読み書きやディレクトリ操作は基本的な機能である。std.Io（旧std.fs）は、POSIX、Windows、WASIなどの異なるOSのファイルシステムAPIを抽象化し、一貫したインターフェースを提供する。これにより、クロスプラットフォームなアプリケーション開発が容易になる。パス操作はPOSIX（/区切り）とWindows（\区切り）の両方に対応し、WTF-8エンコーディングによりWindowsのファイルパスも正しく処理できる。

**機能の利用シーン**：
- ファイルの読み書き（テキスト、バイナリ）
- ディレクトリの作成・削除・走査
- ファイルのメタデータ取得（サイズ、権限、タイムスタンプ）
- シンボリックリンクの作成・解決
- アトミックなファイル更新
- ファイルロック（共有/排他）
- パス操作（結合、正規化、相対/絶対判定）

**主要な処理内容**：
1. ファイル操作（open、create、read、write、close、sync）
2. ディレクトリ操作（openDir、createDir、iterate、walk）
3. パス操作（join、isAbsolute、dirname、basename）
4. メタデータ操作（stat、setPermissions、setTimestamps）
5. ファイルコピー・移動（updateFile、copyFile、rename）
6. 削除操作（deleteFile、deleteDir、deletePath）

**関連システム・外部連携**：
- std.posix：POSIX APIラッパー
- std.os.windows：Windows APIラッパー
- std.Io：新しいI/O統合インターフェース
- std.mem：メモリ操作

**権限による制御**：ファイル権限（Permissions）によりread/write/execute権限を制御

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | std.fsは直接的な画面を持たない標準ライブラリ機能 |

## 機能種別

標準ライブラリ / ファイルシステム操作 / I/O

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | []const u8 | Yes | ファイル/ディレクトリパス | プラットフォーム固有のエンコーディング |
| io | Io | Yes | I/Oインスタンス | 有効なIoオブジェクト |
| flags | OpenFlags/CreateFlags | No | オープンフラグ | 有効なフラグ組み合わせ |
| permissions | Permissions | No | ファイル権限 | 有効な権限値 |

### 入力データソース

- ファイルシステム上のファイル/ディレクトリ
- プログラムから渡されるパス文字列

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| File | Io.File | オープンされたファイルハンドル |
| Dir | Io.Dir | オープンされたディレクトリハンドル |
| stat | File.Stat | ファイルメタデータ |
| data | []u8 | 読み取ったファイル内容 |

### 出力先

ファイルシステム、メモリバッファ

## 処理フロー

### 処理シーケンス

```
1. パス解析
   └─ パス文字列をプラットフォーム形式に変換
2. システムコール発行
   └─ POSIXまたはWindows APIを呼び出し
3. 結果処理
   └─ ハンドル取得またはデータ読み取り
4. エラーハンドリング
   └─ OS固有エラーをZigエラーに変換
```

### フローチャート

```mermaid
flowchart TD
    A[パス入力] --> B{プラットフォーム?}
    B -->|POSIX| C[POSIX API呼び出し]
    B -->|Windows| D[Windows API呼び出し]
    B -->|WASI| E[WASI API呼び出し]
    C --> F{成功?}
    D --> F
    E --> F
    F -->|Yes| G[ハンドル/データ返却]
    F -->|No| H[エラー返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-49-01 | パスエンコーディング | Windows: WTF-8、WASI: UTF-8、他: opaque bytes | パス解析時 |
| BR-49-02 | 最大パス長 | max_path_bytesを超えるパスは使用不可 | パス操作時 |
| BR-49-03 | ファイルロック | 共有/排他ロックによる同時アクセス制御 | ファイルオープン時 |
| BR-49-04 | アトミック更新 | 一時ファイル経由でアトミックに更新 | updateFile時 |

### 計算ロジック

**パス長制限**:
- POSIX: PATH_MAX（通常4096バイト）
- Windows: PATH_MAX_WIDE * 3 + 1（WTF-8展開後）
- max_name_bytes: 各プラットフォームのNAME_MAX

## データベース操作仕様

該当なし（ファイルシステム操作）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| FileNotFound | ファイルエラー | ファイルが存在しない | パス確認 |
| AccessDenied | 権限エラー | 権限不足 | 権限確認/変更 |
| IsDir | 種別エラー | ディレクトリにファイル操作 | 操作種別確認 |
| NotDir | 種別エラー | ファイルにディレクトリ操作 | 操作種別確認 |
| NoSpaceLeft | 容量エラー | ディスク容量不足 | 容量確保 |
| SymLinkLoop | パスエラー | シンボリックリンクループ | パス確認 |

### リトライ仕様

WouldBlock発生時にリトライ可能（ノンブロッキングロック）

## トランザクション仕様

該当なし（アトミックファイル更新はFile.Atomicで提供）

## パフォーマンス要件

- ディレクトリ走査はバッファリングにより最適化
- sendfileによるゼロコピー転送（対応プラットフォーム）
- メモリマップドファイル（MemoryMap）による高速アクセス

## セキュリティ考慮事項

- ファイル権限（Permissions）による適切なアクセス制御
- シンボリックリンク追跡の制御（follow_symlinksオプション）
- TOCTOU（Time-of-check-time-of-use）競合への注意喚起

## 備考

std.fsは現在deprecatedとなり、主要機能はstd.Ioモジュールに移行している。新規開発ではstd.Io.Dir、std.Io.Fileを使用すること。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: fs.zigの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | fs.zig | `lib/std/fs.zig` | ファイル全体（1-22行目） |

**読解のコツ**:
- `//! File System.` - モジュール説明コメント（1行目）
- `pub const path = @import("fs/path.zig");` - パス操作サブモジュール（6行目）
- Deprecated属性：std.Ioへの移行案内

**主要処理フロー**:
1. **6行目**: pathモジュールのインポート（deprecated）
2. **8-12行目**: base64関連のdeprecated定義
3. **14-16行目**: max_path_bytes, max_name_bytesのdeprecated定義

#### Step 2: std.Ioの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Io.zig | `lib/std/Io.zig` | ファイル先頭コメント（1-14行目） |
| 2-2 | Io.zig | `lib/std/Io.zig` | サブモジュールのインポート（478-498行目） |

**読解のコツ**:
- クロスプラットフォームI/O抽象化の説明
- Poller構造体によるマルチプレクシング
- VTableによる多態的I/O実装

**主要処理フロー**:
- **1-14行目**: Ioモジュールの役割説明
- **478-491行目**: Evented I/O実装（IoUring, Kqueue）
- **493-498行目**: Reader, Writer, net, Dir, File, Terminal

#### Step 3: Dir構造体を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Dir.zig | `lib/std/Io/Dir.zig` | Handle, max_path_bytes（13-44行目） |
| 3-2 | Dir.zig | `lib/std/Io/Dir.zig` | cwd関数（88-95行目） |
| 3-3 | Dir.zig | `lib/std/Io/Dir.zig` | Iterator, Walker（180-399行目） |

**読解のコツ**:
- `Handle = std.posix.fd_t` - ファイルディスクリプタ型
- `cwd()` - カレントワーキングディレクトリ取得
- `Iterator` - シンプルなディレクトリ走査
- `Walker` - 再帰的ディレクトリ走査

**主要処理フロー**:
- **72-76行目**: Entry構造体（名前、種別、inode）
- **88-95行目**: cwd関数の実装
- **180-209行目**: Iterator構造体
- **342-383行目**: Walker構造体

#### Step 4: File構造体を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | File.zig | `lib/std/Io/File.zig` | Kind, Stat構造体（28-74行目） |
| 4-2 | File.zig | `lib/std/Io/File.zig` | OpenFlags, CreateFlags（124-249行目） |
| 4-3 | File.zig | `lib/std/Io/File.zig` | Permissions構造体（422-490行目） |

**読解のコツ**:
- `Kind` - ファイル種別（file, directory, sym_link等）
- `Stat` - inode, size, permissions, mtime等
- `OpenFlags` - 読み取り/書き込みモード、ロック
- `Permissions` - POSIX mode / Windows attributes

**主要処理フロー**:
- **28-40行目**: Kind列挙型
- **42-74行目**: Stat構造体
- **76-107行目**: stdout, stderr, stdin関数
- **422-490行目**: Permissions型定義

#### Step 5: パス操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | path.zig | `lib/std/fs/path.zig` | パス区切り文字定義（26-42行目） |
| 5-2 | path.zig | `lib/std/fs/path.zig` | join関数（69-137行目） |

**読解のコツ**:
- `sep`, `sep_str` - プラットフォーム固有区切り文字
- `delimiter` - パス区切り（POSIX: ':', Windows: ';'）
- `join` - 複数パス結合

### プログラム呼び出し階層図

```
アプリケーションコード
    │
    ▼
std.Io ★新しいインターフェース
    │
    ├─ Dir (lib/std/Io/Dir.zig)
    │      │
    │      ├─ cwd() - カレントディレクトリ
    │      ├─ openDir() - ディレクトリオープン
    │      ├─ createDir() - ディレクトリ作成
    │      ├─ iterate() / walk() - 走査
    │      └─ openFile() / createFile() - ファイル操作
    │
    ├─ File (lib/std/Io/File.zig)
    │      │
    │      ├─ read() / write() - 読み書き
    │      ├─ stat() - メタデータ取得
    │      ├─ sync() - 同期
    │      └─ close() - クローズ
    │
    └─ path (lib/std/fs/path.zig)
           │
           ├─ join() - パス結合
           ├─ isAbsolute() - 絶対パス判定
           └─ dirname() / basename()

std.fs (deprecated)
    │
    └─ std.Io へリダイレクト
```

### データフロー図

```
[入力]                      [処理]                        [出力]

パス文字列  ─────────▶  Dir/Fileモジュール  ─────────▶  Handle
                              │
                              ├─ パス解析
                              │    └─ WTF-8/UTF-8変換
                              │
                              ├─ OS API呼び出し
                              │    ├─ POSIX (open, read, write...)
                              │    ├─ Windows (CreateFile...)
                              │    └─ WASI
                              │
                              └─ エラー変換

読み取り  ─────────────▶  reader.read()  ──────────▶  []u8 バッファ

書き込み  ─────────────▶  writer.write()  ─────────▶  ファイルシステム

ディレクトリ走査  ──────▶  Iterator.next()  ─────────▶  Entry (name, kind)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fs.zig | `lib/std/fs.zig` | ソース | ファイルシステムモジュール（deprecated） |
| Io.zig | `lib/std/Io.zig` | ソース | 統合I/Oインターフェース |
| Dir.zig | `lib/std/Io/Dir.zig` | ソース | ディレクトリ操作 |
| File.zig | `lib/std/Io/File.zig` | ソース | ファイル操作 |
| Reader.zig | `lib/std/Io/File/Reader.zig` | ソース | ファイル読み取り |
| Writer.zig | `lib/std/Io/File/Writer.zig` | ソース | ファイル書き込み |
| Atomic.zig | `lib/std/Io/File/Atomic.zig` | ソース | アトミックファイル操作 |
| MemoryMap.zig | `lib/std/Io/File/MemoryMap.zig` | ソース | メモリマップドファイル |
| path.zig | `lib/std/fs/path.zig` | ソース | パス操作ユーティリティ |
| posix.zig | `lib/std/posix.zig` | ソース | POSIXシステムコールラッパー |
