# 機能設計書 50-std.http

## 概要

本ドキュメントは、Zig標準ライブラリのHTTP通信モジュール（std.http）について記述する。std.httpはHTTP/1.0およびHTTP/1.1プロトコルに対応したクライアント・サーバー実装を提供し、TLS暗号化通信やコンテンツ圧縮（gzip、zstd、deflate）もサポートする。

### 本機能の処理概要

std.httpは、HTTP通信のためのクライアントとサーバーの実装を提供するモジュールである。HTTPクライアント（Client.zig）はスレッドセーフな接続プール管理、TLSサポート、プロキシ対応を備える。HTTPサーバー（Server.zig）は単一接続のライフサイクルを管理し、リクエストのパースとレスポンス送信を処理する。

**業務上の目的・背景**：現代のアプリケーション開発において、HTTP通信は不可欠な機能である。std.httpは、外部依存なしにHTTP/1.xプロトコルを実装し、Zigアプリケーションがネイティブにウェブサービスと通信できるようにする。TLSサポートにより安全な通信（HTTPS）も可能であり、コンテンツ圧縮対応によりデータ転送効率も向上する。接続プーリングにより、同一ホストへの複数リクエストでオーバーヘッドを削減できる。

**機能の利用シーン**：
- REST APIへのリクエスト送信
- ウェブサービスからのデータ取得
- ファイルダウンロード
- HTTPサーバーの構築
- プロキシ経由の通信
- WebSocketアップグレード（ws/wss）

**主要な処理内容**：
1. HTTPリクエストの構築と送信（Client）
2. HTTPレスポンスの受信とパース（Client）
3. HTTPリクエストの受信とパース（Server）
4. HTTPレスポンスの構築と送信（Server）
5. TLS暗号化通信（HTTPS）
6. コンテンツ圧縮・展開（gzip、zstd、deflate）
7. チャンク転送エンコーディング
8. 接続プーリング

**関連システム・外部連携**：
- std.Io：I/O抽象化レイヤー
- std.crypto.tls：TLS実装
- std.compress：圧縮・展開
- std.Uri：URI解析

**権限による制御**：特になし（ネットワーク通信機能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | std.httpは直接的な画面を持たない標準ライブラリ機能 |

## 機能種別

標準ライブラリ / ネットワーク通信 / HTTP

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| uri | Uri | Yes | リクエスト先URI | 有効なURI形式 |
| method | Method | Yes | HTTPメソッド | GET, POST, PUT等 |
| headers | []Header | No | HTTPヘッダー | 有効なヘッダー形式 |
| body | []const u8 | No | リクエストボディ | メソッドに応じて |

### 入力データソース

- アプリケーションからのリクエストデータ
- ネットワークからの受信データ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| status | Status | HTTPステータスコード |
| headers | HeaderIterator | レスポンスヘッダー |
| body | Reader | レスポンスボディストリーム |

### 出力先

アプリケーションメモリ、ネットワーク

## 処理フロー

### 処理シーケンス

```
1. URI解析
   └─ スキーム、ホスト、ポート、パスを抽出
2. 接続確立
   └─ 接続プールから取得または新規作成
3. TLSハンドシェイク（HTTPS）
   └─ 証明書検証、暗号化確立
4. リクエスト送信
   └─ ヘッダー、ボディを送信
5. レスポンス受信
   └─ ヘッダーパース、ボディ読み取り
6. 接続プールへ返却
   └─ Keep-Aliveの場合
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト開始] --> B{接続プール確認}
    B -->|既存接続| C[接続取得]
    B -->|新規| D[TCP接続]
    D --> E{HTTPS?}
    E -->|Yes| F[TLSハンドシェイク]
    E -->|No| G[HTTPリクエスト送信]
    F --> G
    C --> G
    G --> H[ヘッダー送信]
    H --> I{ボディあり?}
    I -->|Yes| J[ボディ送信]
    I -->|No| K[レスポンス待機]
    J --> K
    K --> L[ヘッダー受信・パース]
    L --> M[ボディ受信]
    M --> N{Keep-Alive?}
    N -->|Yes| O[接続プールへ返却]
    N -->|No| P[接続クローズ]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-50-01 | HTTPバージョン | HTTP/1.0およびHTTP/1.1をサポート | 全通信 |
| BR-50-02 | Keep-Alive | HTTP/1.1はデフォルトでKeep-Alive | HTTP/1.1接続 |
| BR-50-03 | 圧縮サポート | gzip、zstd、deflate、compress | Accept-Encoding |
| BR-50-04 | チャンク転送 | Transfer-Encoding: chunkedサポート | ストリーミング時 |
| BR-50-05 | 接続プール | 最大32接続をプール（デフォルト） | 接続管理時 |

### 計算ロジック

**ステータスコード分類**:
- 100-199: Informational
- 200-299: Success
- 300-399: Redirect
- 400-499: Client Error
- 500-599: Server Error

## データベース操作仕様

該当なし（ネットワーク通信機能）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HttpHeadersOversize | プロトコルエラー | ヘッダーサイズ超過 | バッファサイズ増加 |
| HttpHeadersInvalid | プロトコルエラー | 不正なヘッダー形式 | ヘッダー確認 |
| HttpChunkInvalid | プロトコルエラー | 不正なチャンク形式 | サーバー確認 |
| ConnectionRefused | 接続エラー | 接続拒否 | ホスト・ポート確認 |
| TlsError | 暗号化エラー | TLSハンドシェイク失敗 | 証明書確認 |

### リトライ仕様

接続エラー時にアプリケーション側でリトライ可能

## トランザクション仕様

該当なし

## パフォーマンス要件

- 接続プールによるオーバーヘッド削減
- バッファサイズ設定（read_buffer_size: 8192、write_buffer_size: 1024）
- コンテンツ圧縮によるデータ転送量削減

## セキュリティ考慮事項

- TLS対応によるHTTPS通信
- 証明書バンドルによるサーバー認証
- SSL Key Logのオプショナルサポート（デバッグ用）
- プロキシ認証対応

## 備考

std.httpはHTTP/1.xをサポート。HTTP/2およびHTTP/3は現在未実装。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: http.zigの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | http.zig | `lib/std/http.zig` | サブモジュールのインポート（7-11行目） |
| 1-2 | http.zig | `lib/std/http.zig` | Version列挙型（13-16行目） |
| 1-3 | http.zig | `lib/std/http.zig` | Method列挙型（23-90行目） |
| 1-4 | http.zig | `lib/std/http.zig` | Status列挙型（93-268行目） |

**読解のコツ**:
- `Client` - HTTPクライアント実装
- `Server` - HTTPサーバー実装
- `HeadParser` - ヘッダーパーサー
- `ChunkParser` - チャンクパーサー
- `HeaderIterator` - ヘッダーイテレーター

**主要処理フロー**:
1. **7-11行目**: サブモジュールのpub const
2. **13-16行目**: HTTP/1.0, HTTP/1.1バージョン
3. **23-90行目**: GET, POST等のメソッドと特性
4. **93-268行目**: ステータスコードとフレーズ

#### Step 2: HTTPメソッドの特性を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | http.zig | `lib/std/http.zig` | requestHasBody（36-41行目） |
| 2-2 | http.zig | `lib/std/http.zig` | safe, idempotent, cacheable（52-89行目） |

**読解のコツ**:
- `requestHasBody` - POST, PUT, PATCHはtrue
- `safe` - GET, HEAD, OPTIONS, TRACEはサーバー状態を変更しない
- `idempotent` - 同じリクエストを複数回送っても結果が同じ
- `cacheable` - GET, HEADはキャッシュ可能

#### Step 3: Client構造体を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Client.zig | `lib/std/http/Client.zig` | ファイル先頭コメント（1-5行目） |
| 3-2 | Client.zig | `lib/std/http/Client.zig` | Clientフィールド（24-61行目） |
| 3-3 | Client.zig | `lib/std/http/Client.zig` | ConnectionPool構造体（64-203行目） |
| 3-4 | Client.zig | `lib/std/http/Client.zig` | Protocol列挙型（205-229行目） |

**読解のコツ**:
- `allocator` - クライアント用アロケータ
- `io` - Ioインスタンス（TCP接続）
- `ca_bundle` - TLS証明書バンドル
- `connection_pool` - 接続プール
- `http_proxy`, `https_proxy` - プロキシ設定

**主要処理フロー**:
- **24-27行目**: allocator, io
- **29-36行目**: TLS関連設定
- **45-52行目**: connection_pool, バッファサイズ
- **57-61行目**: プロキシ設定

#### Step 4: Server構造体を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Server.zig | `lib/std/http/Server.zig` | ファイル先頭コメント（1行目） |
| 4-2 | Server.zig | `lib/std/http/Server.zig` | Serverフィールド（14-16行目） |
| 4-3 | Server.zig | `lib/std/http/Server.zig` | init関数（25-36行目） |
| 4-4 | Server.zig | `lib/std/http/Server.zig` | Request.Head（70-235行目） |

**読解のコツ**:
- `out` - レスポンス出力Writer
- `reader` - リクエスト入力Reader
- `receiveHead` - ヘッダー受信
- `Head.parse` - ヘッダーパース

**主要処理フロー**:
- **14-16行目**: out, reader
- **46-55行目**: receiveHead
- **92-199行目**: Head.parse実装

#### Step 5: コンテンツエンコーディングを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | http.zig | `lib/std/http.zig` | TransferEncoding（271-274行目） |
| 5-2 | http.zig | `lib/std/http.zig` | ContentEncoding（276-303行目） |

**読解のコツ**:
- `TransferEncoding` - chunked, none
- `ContentEncoding` - zstd, gzip, deflate, compress, identity
- `fromString` - 文字列からエンコーディングへ変換
- `minBufferCapacity` - 必要バッファサイズ

### プログラム呼び出し階層図

```
アプリケーションコード
    │
    ▼
std.http ★本機能
    │
    ├─ Client (lib/std/http/Client.zig)
    │      │
    │      ├─ ConnectionPool - 接続プール管理
    │      │      ├─ findConnection() - 既存接続検索
    │      │      ├─ acquire() - 接続取得
    │      │      └─ release() - 接続返却
    │      │
    │      ├─ Connection - 単一接続
    │      │      ├─ Plain - 非TLS接続
    │      │      └─ Tls - TLS接続
    │      │
    │      └─ Request - HTTPリクエスト
    │
    ├─ Server (lib/std/http/Server.zig)
    │      │
    │      ├─ receiveHead() - ヘッダー受信
    │      ├─ Request.Head.parse() - ヘッダーパース
    │      └─ respond() - レスポンス送信
    │
    ├─ HeadParser - ヘッダーパース
    │
    ├─ ChunkParser - チャンクパース
    │
    └─ Reader - HTTPボディ読み取り
           ├─ bodyReader() - 圧縮なし
           └─ bodyReaderDecompressing() - 圧縮展開
```

### データフロー図

```
[入力]                      [処理]                        [出力]

URI/Method  ───────────▶  Client  ────────────────────▶  Response
                              │
                              ├─ URI解析
                              ├─ 接続確立/プール取得
                              ├─ TLSハンドシェイク（HTTPS）
                              ├─ リクエスト送信
                              └─ レスポンス受信

受信データ  ───────────▶  Server  ────────────────────▶  Request
                              │
                              ├─ ヘッダー受信
                              ├─ Head.parse
                              │    ├─ method抽出
                              │    ├─ target抽出
                              │    ├─ version抽出
                              │    └─ ヘッダーフィールド抽出
                              │
                              └─ ボディ読み取り
                                   ├─ Content-Length
                                   └─ Transfer-Encoding: chunked

圧縮ボディ  ───────────▶  Reader.bodyReaderDecompressing  ▶  展開データ
                              │
                              ├─ gzip → std.compress.flate
                              ├─ deflate → std.compress.flate
                              └─ zstd → std.compress.zstd
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| http.zig | `lib/std/http.zig` | ソース | HTTPモジュールルート |
| Client.zig | `lib/std/http/Client.zig` | ソース | HTTPクライアント実装 |
| Server.zig | `lib/std/http/Server.zig` | ソース | HTTPサーバー実装 |
| HeadParser.zig | `lib/std/http/HeadParser.zig` | ソース | ヘッダーパーサー |
| ChunkParser.zig | `lib/std/http/ChunkParser.zig` | ソース | チャンクパーサー |
| HeaderIterator.zig | `lib/std/http/HeaderIterator.zig` | ソース | ヘッダーイテレーター |
| Io.zig | `lib/std/Io.zig` | ソース | I/O抽象化レイヤー |
| tls/Client.zig | `lib/std/crypto/tls/Client.zig` | ソース | TLSクライアント |
| Uri.zig | `lib/std/Uri.zig` | ソース | URI解析 |
| compress/ | `lib/std/compress/` | ソース | 圧縮・展開モジュール |
