# 機能設計書 51-std.crypto

## 概要

本ドキュメントは、Zig標準ライブラリの暗号化モジュール `std.crypto` の機能設計を記述する。このモジュールは、暗号化アルゴリズム、ハッシュ関数、デジタル署名、鍵導出関数などの包括的な暗号プリミティブを提供する。

### 本機能の処理概要

std.cryptoは、現代の暗号化ニーズに対応する包括的な暗号ライブラリを提供する。セキュリティが重要なアプリケーションにおいて、データの機密性、完全性、認証を確保するための基盤となる。

**業務上の目的・背景**：現代のソフトウェア開発において、セキュアな通信、データ保護、認証機能は必須である。std.cryptoは外部依存なしにこれらの暗号機能を提供し、Zigプログラムがセキュアなアプリケーションを構築できるようにする。TLS通信、パスワード保存、デジタル署名、暗号化ファイル処理など、様々なセキュリティ要件に対応する。

**機能の利用シーン**：
- HTTPS/TLS通信の実装時
- ユーザーパスワードの安全なハッシュ化と保存
- ファイルやメッセージの暗号化・復号化
- デジタル署名の生成と検証
- 乱数生成とキー導出
- メッセージ認証コード（MAC）の計算

**主要な処理内容**：
1. AEAD（認証付き暗号化）: AES-GCM、ChaCha20-Poly1305、AEGISなどの認証付き暗号化方式を提供
2. ハッシュ関数: SHA-2、SHA-3、BLAKE2、BLAKE3、MD5、SHA-1などのハッシュアルゴリズム
3. デジタル署名: Ed25519、ECDSA（P-256、P-384、secp256k1）
4. 鍵交換: X25519 Diffie-Hellman
5. 鍵カプセル化メカニズム（KEM）: ML-KEM（Kyber）
6. パスワードハッシュ: Argon2、bcrypt、scrypt、PBKDF2
7. 鍵導出関数: HKDF
8. メッセージ認証: HMAC、CMAC、SipHash

**関連システム・外部連携**：
- std.http: TLS通信でcryptoモジュールを使用
- std.crypto.Certificate: X.509証明書の検証
- std.crypto.tls: TLSプロトコルの実装

**権限による制御**：本モジュールは特別な権限制御を持たず、すべての機能がアプリケーションから直接利用可能である。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

計算処理 / データ変換 / セキュリティ

## 入力仕様

### 入力パラメータ

暗号化操作の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| key | []const u8 | Yes | 暗号鍵 | アルゴリズム固有の鍵長 |
| plaintext | []const u8 | Yes | 平文データ | 最大長はメモリ依存 |
| nonce | []const u8 | Yes | ナンス（使い捨て値） | アルゴリズム固有の長さ |
| ad | []const u8 | No | 追加認証データ | 任意長 |

### 入力データソース

プログラムからのメモリ上のデータ、ファイルから読み込んだデータ、ネットワークから受信したデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ciphertext | []u8 | 暗号化されたデータ |
| tag | [tag_len]u8 | 認証タグ（AEADの場合） |
| hash | [digest_len]u8 | ハッシュ値 |
| signature | [sig_len]u8 | デジタル署名 |

### 出力先

メモリ上のバッファ、ファイル、ネットワーク送信用バッファ

## 処理フロー

### 処理シーケンス

```
1. 鍵の初期化
   └─ 暗号コンテキストを鍵で初期化
2. データ処理
   └─ 入力データを暗号化/復号化/ハッシュ化
3. 結果の取得
   └─ 処理結果とタグ/署名を出力バッファに書き込み
4. 検証（復号時）
   └─ 認証タグまたは署名を検証
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[鍵の初期化]
    B --> C{処理タイプ}
    C -->|暗号化| D[AEADコンテキスト初期化]
    C -->|ハッシュ| E[ハッシュコンテキスト初期化]
    C -->|署名| F[署名鍵の生成/読込]
    D --> G[データ暗号化]
    G --> H[認証タグ生成]
    E --> I[データのハッシュ計算]
    I --> J[ダイジェスト取得]
    F --> K[データへの署名]
    H --> L[結果出力]
    J --> L
    K --> L
    L --> M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-51-01 | ナンス一意性 | 同一鍵でナンスを再利用してはならない | AEAD暗号化時 |
| BR-51-02 | 認証タグ検証 | 復号前に認証タグを検証する | AEAD復号時 |
| BR-51-03 | 定時実行 | サイドチャネル攻撃対策として一定時間で比較 | タイミング比較時 |
| BR-51-04 | メモリゼロ化 | 機密データ使用後はメモリをゼロクリア | 鍵/平文処理後 |

### 計算ロジック

AES-GCM暗号化の例:
```zig
// 鍵とナンスで初期化
const ctx = Aes256Gcm.init(key, nonce);
// 暗号化と認証タグ生成
ctx.encrypt(ciphertext, plaintext, ad, &tag);
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| AuthenticationFailed | 認証エラー | 認証タグが一致しない | データ改ざんの可能性、処理中止 |
| SignatureVerificationFailed | 署名エラー | 署名検証に失敗 | 署名者の確認、データ整合性確認 |
| IdentityElement | 鍵エラー | 無効な公開鍵 | 鍵の再生成を要求 |
| NonCanonical | 形式エラー | 非正規形式のデータ | 入力データの検証 |

### リトライ仕様

暗号操作は決定論的であり、同じ入力に対して同じエラーが発生するため、リトライは通常不要。

## トランザクション仕様

本機能はトランザクション制御を行わない（純粋な計算処理）。

## パフォーマンス要件

- AES-NI/ARM Crypto Extensions対応CPUでは、AESがハードウェアアクセラレーションされる
- ベクトル化により、複数ブロックの並列処理が可能
- サイドチャネル耐性モード（`SideChannelsMitigations`）の設定により、セキュリティとパフォーマンスのトレードオフを調整可能

## セキュリティ考慮事項

- 定数時間実装によるタイミング攻撃対策（`timing_safe`モジュール）
- `secureZero`関数による機密データのメモリゼロ化
- サイドチャネル緩和レベルの設定（none, basic, medium, full）
- 非推奨アルゴリズム（MD5, SHA-1）の存在は互換性のためであり、新規利用は非推奨

## 備考

- ハードウェアサポートの有無は `has_hardware_support` で確認可能
- `default_side_channels_mitigations` はデフォルトで `.medium`

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

暗号モジュールの全体構造と公開APIを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | crypto.zig | `lib/std/crypto.zig` | モジュール全体の構造と公開されている暗号プリミティブの一覧 |

**読解のコツ**: `pub const`宣言でサブモジュールをインポートしている。`aead`, `auth`, `hash`, `sign`などのカテゴリ分けを理解する。

**主要処理フロー**:
1. **8-69行目**: AEADカテゴリの定義（AEGIS、AES-GCM、ChaCha20-Poly1305など）
2. **71-95行目**: 認証（MAC）関数の定義（HMAC、SipHash、CMAC）
3. **97-110行目**: コア関数（AES、Keccak）
4. **134-149行目**: ハッシュ関数の定義
5. **179-195行目**: パスワードハッシュ関数の定義
6. **197-202行目**: デジタル署名関数の定義
7. **246-263行目**: サイドチャネル緩和レベルの定義

#### Step 2: AES実装を理解する

コア暗号プリミティブの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | aes.zig | `lib/std/crypto/aes.zig` | AES暗号化の実装とハードウェアアクセラレーション選択 |

**主要処理フロー**:
- **5-16行目**: ハードウェアサポートの検出とバックエンド選択
- **18-22行目**: ハードウェアサポートフラグの定義
- **24-29行目**: 公開API（Block、Aes128、Aes256）のエクスポート

#### Step 3: 認証付き暗号化を理解する

AEADの実装パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | aes_gcm.zig | `lib/std/crypto/aes_gcm.zig` | AES-GCMの実装 |
| 3-2 | chacha20.zig | `lib/std/crypto/chacha20.zig` | ChaCha20-Poly1305の実装 |

#### Step 4: ハッシュ関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | sha2.zig | `lib/std/crypto/sha2.zig` | SHA-256/512の実装 |
| 4-2 | blake3.zig | `lib/std/crypto/blake3.zig` | BLAKE3ハッシュの実装 |

#### Step 5: デジタル署名を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | ed25519.zig | `lib/std/crypto/25519/ed25519.zig` | Ed25519署名の実装 |
| 5-2 | ecdsa.zig | `lib/std/crypto/ecdsa.zig` | ECDSA署名の実装 |

### プログラム呼び出し階層図

```
std.crypto
    |
    +-- aead
    |      +-- aegis (AEGIS暗号)
    |      +-- aes_gcm (AES-GCM)
    |      +-- aes_gcm_siv (AES-GCM-SIV)
    |      +-- chacha_poly (ChaCha20-Poly1305)
    |
    +-- auth
    |      +-- hmac (HMAC)
    |      +-- siphash (SipHash)
    |      +-- cmac (CMAC)
    |
    +-- core
    |      +-- aes (AES基本ブロック暗号)
    |             +-- aesni.zig (x86_64ハードウェア実装)
    |             +-- armcrypto.zig (ARM実装)
    |             +-- soft.zig (ソフトウェア実装)
    |
    +-- hash
    |      +-- sha2 (SHA-256/512)
    |      +-- sha3 (SHA-3/Keccak)
    |      +-- blake2/blake3
    |
    +-- sign
    |      +-- Ed25519
    |      +-- ecdsa
    |
    +-- pwhash
           +-- argon2
           +-- bcrypt
           +-- scrypt
```

### データフロー図

```
[入力]               [処理]                    [出力]

平文データ ───────> AES-GCM暗号化 ─────────> 暗号文 + 認証タグ
   |                    |
   +-- 鍵 ─────────────+
   +-- ナンス ─────────+
   +-- 追加認証データ ─+

パスワード ───────> Argon2ハッシュ ─────────> パスワードハッシュ
   |                    |
   +-- ソルト ─────────+
   +-- パラメータ ─────+

メッセージ ───────> Ed25519署名 ──────────> デジタル署名
   |                    |
   +-- 秘密鍵 ─────────+
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| crypto.zig | `lib/std/crypto.zig` | ソース | メインモジュール、公開API |
| aes.zig | `lib/std/crypto/aes.zig` | ソース | AESブロック暗号 |
| aesni.zig | `lib/std/crypto/aes/aesni.zig` | ソース | AES-NI実装 |
| armcrypto.zig | `lib/std/crypto/aes/armcrypto.zig` | ソース | ARM暗号拡張実装 |
| soft.zig | `lib/std/crypto/aes/soft.zig` | ソース | ソフトウェアAES実装 |
| aes_gcm.zig | `lib/std/crypto/aes_gcm.zig` | ソース | AES-GCM AEAD |
| chacha20.zig | `lib/std/crypto/chacha20.zig` | ソース | ChaCha20ストリーム暗号 |
| sha2.zig | `lib/std/crypto/sha2.zig` | ソース | SHA-2ハッシュ |
| sha3.zig | `lib/std/crypto/sha3.zig` | ソース | SHA-3ハッシュ |
| blake3.zig | `lib/std/crypto/blake3.zig` | ソース | BLAKE3ハッシュ |
| ed25519.zig | `lib/std/crypto/25519/ed25519.zig` | ソース | Ed25519署名 |
| ecdsa.zig | `lib/std/crypto/ecdsa.zig` | ソース | ECDSA署名 |
| argon2.zig | `lib/std/crypto/argon2.zig` | ソース | Argon2パスワードハッシュ |
| timing_safe.zig | `lib/std/crypto/timing_safe.zig` | ソース | 定数時間比較 |
| errors.zig | `lib/std/crypto/errors.zig` | ソース | 暗号エラー定義 |
| tls.zig | `lib/std/crypto/tls.zig` | ソース | TLSプロトコル |
| Certificate.zig | `lib/std/crypto/Certificate.zig` | ソース | X.509証明書 |
