# 機能設計書 52-std.compress

## 概要

本ドキュメントは、Zig標準ライブラリの圧縮モジュール `std.compress` の機能設計を記述する。このモジュールは、deflate（gzip/zlib）、LZMA、LZMA2、XZ、Zstd などの圧縮・展開アルゴリズムを提供する。

### 本機能の処理概要

std.compressは、データ圧縮と展開のための包括的なアルゴリズム群を提供する。ファイルサイズの削減、ネットワーク転送の効率化、アーカイブファイルの処理など、様々な用途で利用される。

**業務上の目的・背景**：データ圧縮は、ストレージ効率の向上、ネットワーク帯域の節約、アーカイブファイル（.gz, .xz, .zst）の処理に不可欠である。std.compressは外部ライブラリに依存せず、純粋なZig実装でこれらの機能を提供し、クロスコンパイル環境でも一貫した動作を保証する。

**機能の利用シーン**：
- gzip/zlib形式のファイル読み書き
- HTTPレスポンスのgzip圧縮展開
- XZアーカイブの展開
- Zstandard圧縮データの処理
- ビルドシステムでの圧縮アーティファクト処理

**主要な処理内容**：
1. deflate/inflate: RFC 1951準拠のDEFLATE圧縮・展開（gzip/zlib基盤）
2. gzip: RFC 1952準拠のgzip形式圧縮・展開
3. zlib: RFC 1950準拠のzlib形式圧縮・展開
4. LZMA: Lempel-Ziv-Markov chain Algorithm圧縮・展開
5. LZMA2: LZMA改良版（XZで使用）
6. XZ: .xz形式アーカイブの展開
7. Zstd: Zstandard圧縮・展開

**関連システム・外部連携**：
- std.http: HTTPレスポンスのContent-Encoding処理
- std.fs: 圧縮ファイルの読み書き
- ビルドシステム: 圧縮アーティファクトの処理

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

データ変換 / 圧縮処理

## 入力仕様

### 入力パラメータ

圧縮・展開操作の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| reader | anytype | Yes | 入力データのReader | std.Io.Reader準拠 |
| writer | anytype | Yes | 出力データのWriter | std.Io.Writer準拠 |
| window_buffer | []u8 | Yes | 履歴ウィンウバッファ | 最小32KB (max_window_len) |
| container | Container | No | フォーマット種別 | .raw/.gzip/.zlib |

### 入力データソース

圧縮データストリーム、ファイル、メモリバッファ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| decompressed_data | []u8 | 展開されたデータ |
| compressed_data | []u8 | 圧縮されたデータ |
| checksum | u32 | CRC32またはAdler-32チェックサム |

### 出力先

メモリバッファ、ファイル、ネットワークストリーム

## 処理フロー

### 処理シーケンス

```
1. コンテナ解析（gzip/zlibの場合）
   └─ ヘッダーを読み取りメタデータを取得
2. 圧縮データの読み込み
   └─ 入力ストリームからデータを読み取り
3. 展開処理
   └─ アルゴリズムに従いデータを展開
4. チェックサム検証
   └─ CRC32またはAdler-32で整合性確認
5. フッター解析
   └─ 元データサイズの検証（gzipの場合）
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{コンテナ形式}
    B -->|gzip| C[gzipヘッダー解析]
    B -->|zlib| D[zlibヘッダー解析]
    B -->|raw| E[直接処理]
    C --> F[DEFLATEデータ展開]
    D --> F
    E --> F
    F --> G{チェックサム検証}
    G -->|成功| H[結果出力]
    G -->|失敗| I[エラー: チェックサム不一致]
    H --> J[終了]
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-52-01 | ウィンドウサイズ | 履歴バッファは最低32KB必要 | DEFLATE処理時 |
| BR-52-02 | チェックサム必須 | gzip/zlib形式ではチェックサム検証必須 | 展開完了時 |
| BR-52-03 | ストリーミング | 圧縮・展開はストリーミング処理可能 | 大容量データ処理時 |

### 計算ロジック

deflate展開の例:
```zig
// バッファの初期化
var buffer: [std.compress.flate.max_window_len]u8 = undefined;
// 展開器の初期化
var decompressor = std.compress.flate.Decompress.init(&buffer, .gzip);
// データの展開
const result = decompressor.read(output_buffer);
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| BadGzipHeader | フォーマットエラー | gzipヘッダーが不正 | 入力データの確認 |
| BadZlibHeader | フォーマットエラー | zlibヘッダーが不正 | 入力データの確認 |
| WrongGzipChecksum | 整合性エラー | CRC32が一致しない | データ破損の確認 |
| WrongZlibChecksum | 整合性エラー | Adler-32が一致しない | データ破損の確認 |
| WrongGzipSize | サイズエラー | 展開サイズが一致しない | データ破損の確認 |

### リトライ仕様

圧縮・展開処理は決定論的であり、同じ入力に対して同じエラーが発生するため、リトライは通常不要。

## トランザクション仕様

本機能はトランザクション制御を行わない（純粋なデータ変換処理）。

## パフォーマンス要件

- ストリーミング処理によりメモリ使用量を最小化
- 履歴ウィンドウサイズ（32KB×2）がメモリ要件の主要部分
- 圧縮レベルの選択により速度と圧縮率のトレードオフ調整可能

## セキュリティ考慮事項

- 展開サイズの検証によるZip爆弾対策
- チェックサム検証によるデータ改ざん検出
- 入力データの妥当性検証

## 備考

- max_window_len = 65536 (32KB × 2)
- history_len = 32768 (32KB)
- gzip形式はビッグエンディアン、zlib形式はリトルエンディアン

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

圧縮モジュールの全体構造とコンテナ形式を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | compress.zig | `lib/std/compress.zig` | モジュール全体の構造とサブモジュール |

**読解のコツ**: `pub const`宣言で各圧縮アルゴリズムのサブモジュールをインポートしている。

**主要処理フロー**:
1. **4行目**: flate（DEFLATE）モジュールのインポート
2. **5行目**: lzmaモジュールのインポート
3. **6行目**: lzma2モジュールのインポート
4. **7行目**: xzモジュールのインポート
5. **8行目**: zstdモジュールのインポート

#### Step 2: DEFLATE/gzip/zlibを理解する

最も基本的な圧縮形式の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | flate.zig | `lib/std/compress/flate.zig` | DEFLATE形式とコンテナ（gzip/zlib）の定義 |

**主要処理フロー**:
- **5-6行目**: ウィンドウサイズ定数の定義（max_window_len, history_len）
- **9-15行目**: Compress（圧縮）とDecompress（展開）のインポート
- **29-84行目**: Container enum（raw/gzip/zlib）の定義
- **60-83行目**: 各形式のヘッダーバイト列
- **86-140行目**: Hasherの実装（CRC32/Adler-32）

#### Step 3: 圧縮実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Compress.zig | `lib/std/compress/flate/Compress.zig` | DEFLATE圧縮の実装 |

#### Step 4: 展開実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Decompress.zig | `lib/std/compress/flate/Decompress.zig` | DEFLATE展開の実装 |

#### Step 5: 他の圧縮形式を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | lzma.zig | `lib/std/compress/lzma.zig` | LZMA圧縮の実装 |
| 5-2 | xz.zig | `lib/std/compress/xz.zig` | XZ形式の実装 |
| 5-3 | zstd.zig | `lib/std/compress/zstd.zig` | Zstandard圧縮の実装 |

### プログラム呼び出し階層図

```
std.compress
    |
    +-- flate
    |      +-- Compress (DEFLATE圧縮)
    |      +-- Decompress (DEFLATE展開)
    |      +-- Container
    |             +-- .raw (生DEFLATEストリーム)
    |             +-- .gzip (gzip形式)
    |             +-- .zlib (zlib形式)
    |
    +-- lzma
    |      +-- Decompress (LZMA展開)
    |
    +-- lzma2
    |      +-- Decompress (LZMA2展開)
    |
    +-- xz
    |      +-- Decompress (XZ展開)
    |
    +-- zstd
           +-- Decompress (Zstd展開)
```

### データフロー図

```
[入力]                    [処理]                       [出力]

圧縮データ ──────────────> gzip/zlibヘッダー解析 ───> メタデータ
(gzip/zlib形式)                  |
                                 v
                         DEFLATEストリーム展開 ───────> 元データ
                                 |
                                 v
                         チェックサム検証 ────────────> 検証結果
                                 |
                                 v
                         フッター解析 ────────────────> サイズ検証

元データ ────────────────> DEFLATE圧縮 ──────────────> 圧縮ストリーム
                                 |
                                 v
                         gzip/zlibヘッダー付与 ──────> 出力データ
                                 |
                                 v
                         チェックサム/フッター付与 ──> 最終出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| compress.zig | `lib/std/compress.zig` | ソース | メインモジュール |
| flate.zig | `lib/std/compress/flate.zig` | ソース | DEFLATE/gzip/zlibメイン |
| Compress.zig | `lib/std/compress/flate/Compress.zig` | ソース | DEFLATE圧縮実装 |
| Decompress.zig | `lib/std/compress/flate/Decompress.zig` | ソース | DEFLATE展開実装 |
| token.zig | `lib/std/compress/flate/token.zig` | ソース | トークン処理 |
| lzma.zig | `lib/std/compress/lzma.zig` | ソース | LZMAメイン |
| lzma2.zig | `lib/std/compress/lzma2.zig` | ソース | LZMA2メイン |
| xz.zig | `lib/std/compress/xz.zig` | ソース | XZメイン |
| Decompress.zig | `lib/std/compress/xz/Decompress.zig` | ソース | XZ展開実装 |
| zstd.zig | `lib/std/compress/zstd.zig` | ソース | Zstdメイン |
| Decompress.zig | `lib/std/compress/zstd/Decompress.zig` | ソース | Zstd展開実装 |
