# 機能設計書 53-std.json

## 概要

本ドキュメントは、Zig標準ライブラリのJSONモジュール `std.json` の機能設計を記述する。このモジュールは、RFC 8259準拠のJSONパース（解析）とシリアライズ（生成）機能を提供する。

### 本機能の処理概要

std.jsonは、JSONデータの解析と生成のための包括的なAPIを提供する。低レベルのスキャナーAPIから高レベルのパース・シリアライズAPIまで、様々な抽象度でJSONを扱うことができる。

**業務上の目的・背景**：JSONは現代のWeb API、設定ファイル、データ交換において最も広く使用されるデータ形式である。std.jsonは外部依存なしにJSONの完全なサポートを提供し、型安全なZig構造体へのデシリアライズ、動的な値としての解析、効率的なストリーミング処理を可能にする。

**機能の利用シーン**：
- REST APIレスポンスのパース
- 設定ファイル（.json）の読み込み
- データ構造のJSONシリアライズ
- JSON-RPCプロトコルの処理
- ログデータのJSON出力

**主要な処理内容**：
1. Scanner: 低レベルトークンスキャナー（ストリーミング対応）
2. parseFromSlice: スライスからZig型へのデシリアライズ
3. parseFromTokenSource: トークンソースからのパース
4. Value: 動的JSON値の表現
5. Stringify: Zig値からJSONへのシリアライズ
6. validate: JSON構文の検証

**関連システム・外部連携**：
- std.http: HTTPレスポンスボディのJSONパース
- std.fs: JSONファイルの読み書き
- ビルドシステム: build.zig.zonの処理

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

データ変換 / パース処理

## 入力仕様

### 入力パラメータ

JSON解析の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| T | type | Yes | デシリアライズ先の型 | comptime型 |
| allocator | Allocator | Yes | メモリアロケータ | 有効なアロケータ |
| input | []const u8 | Yes | JSON文字列 | 有効なUTF-8 |
| options | ParseOptions | No | パースオプション | - |

### 入力データソース

JSON文字列（メモリ上のスライス、ファイル、ネットワークストリーム）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| parsed_value | T | デシリアライズされたZig値 |
| Value | json.Value | 動的JSON値 |
| json_string | []const u8 | シリアライズされたJSON文字列 |

### 出力先

メモリバッファ、ファイル、ネットワークストリーム

## 処理フロー

### 処理シーケンス

```
1. 入力の準備
   └─ JSON文字列またはストリームの取得
2. トークン化（Scanner）
   └─ JSON文字列をトークン列に分解
3. パース/デシリアライズ
   └─ トークン列からZig値を構築
4. 結果の返却
   └─ Parsed構造体でラップして返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{パース方式}
    B -->|静的型| C[parseFromSlice]
    B -->|動的値| D[parseFromSlice with Value]
    B -->|ストリーム| E[Scanner + parseFromTokenSource]
    C --> F[Scannerでトークン化]
    D --> F
    E --> F
    F --> G{トークン種別}
    G -->|オブジェクト| H[オブジェクトパース]
    G -->|配列| I[配列パース]
    G -->|文字列/数値/等| J[プリミティブパース]
    H --> K[結果構築]
    I --> K
    J --> K
    K --> L{検証}
    L -->|成功| M[Parsed返却]
    L -->|失敗| N[エラー返却]
    M --> O[終了]
    N --> O
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-53-01 | UTF-8必須 | 入力はUTF-8エンコード必須 | パース時 |
| BR-53-02 | 重複キー許可 | オブジェクトの重複キーは出現順に処理 | オブジェクトパース時 |
| BR-53-03 | 最大値長制限 | default_max_value_lenで文字列長を制限 | メモリ保護 |
| BR-53-04 | サロゲートペア | \uエスケープでサロゲートペア必須 | Unicode処理時 |

### 計算ロジック

JSONパースの例:
```zig
const T = struct { a: i32 = -1, b: [2]u8 };
var parsed = try std.json.parseFromSlice(T, allocator, "{\"b\":\"xy\"}", .{});
defer parsed.deinit();
// parsed.value.a == -1 (デフォルト値)
// parsed.value.b == "xy"
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| SyntaxError | 構文エラー | 不正なJSON構文 | 入力データの修正 |
| UnexpectedEndOfInput | 入力エラー | 入力が途中で終了 | 完全なJSONの提供 |
| UnknownField | フィールドエラー | 未知のオブジェクトキー | オプションで無視可 |
| MissingField | フィールドエラー | 必須フィールドなし | デフォルト値設定 |
| Overflow | 数値エラー | 数値がオーバーフロー | 型の確認 |
| ValueTooLong | 長さエラー | 文字列が長すぎる | max_value_len調整 |

### リトライ仕様

パース処理は決定論的であり、同じ入力に対して同じエラーが発生するため、リトライは通常不要。

## トランザクション仕様

本機能はトランザクション制御を行わない（純粋なデータ変換処理）。

## パフォーマンス要件

- Scannerはストリーミング処理対応でメモリ使用量O(d)（dはネスト深度）
- パース結果のメモリ所有権はParsed構造体が管理
- Leakyバージョン（parseFromSliceLeaky等）でアロケータを省略可能

## セキュリティ考慮事項

- default_max_value_lenによる文字列長制限
- ネスト深度によるスタック消費の制御
- 不正なUTF-8シーケンスの検出と拒否

## 備考

- RFC 8259完全準拠
- ECMA-404との違い：孤立サロゲートを禁止
- default_max_value_len = 4096（調整可能）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

JSONモジュールの全体構造と公開APIを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | json.zig | `lib/std/json.zig` | モジュール全体の構造と公開API |

**読解のコツ**: テストコードを見ると使用例がわかりやすい。

**主要処理フロー**:
1. **1-10行目**: モジュールのドキュメントコメント（低レベルから高レベルAPIの説明）
2. **16-24行目**: Scannerの使用例テスト
3. **26-36行目**: parseFromSliceの使用例テスト
4. **38-42行目**: Valueの使用例テスト
5. **44-61行目**: Stringifyの使用例テスト
6. **63-79行目**: 公開型のエクスポート
7. **81-92行目**: パース関連関数のエクスポート
8. **94行目**: Stringifyのエクスポート

#### Step 2: Scannerを理解する

低レベルトークンスキャナーの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Scanner.zig | `lib/std/json/Scanner.zig` | トークンスキャナーの実装 |

**主要処理フロー**:
- **1-43行目**: RFC 8259準拠の詳細なドキュメント
- **45-61行目**: Scanner構造体のフィールド定義
- **64-68行目**: initStreaming（ストリーミングモード初期化）
- **76-82行目**: initCompleteInput（完全入力モード初期化）
- **115-172行目**: nextAllocMax（トークン取得とメモリ割り当て）

#### Step 3: 静的パースを理解する

Zig型へのデシリアライズ実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | static.zig | `lib/std/json/static.zig` | parseFromSlice等の実装 |

#### Step 4: 動的値を理解する

Value型の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | dynamic.zig | `lib/std/json/dynamic.zig` | Value, ObjectMap, Arrayの実装 |

#### Step 5: シリアライズを理解する

JSON文字列生成の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | Stringify.zig | `lib/std/json/Stringify.zig` | JSON出力の実装 |

### プログラム呼び出し階層図

```
std.json
    |
    +-- Scanner (低レベルスキャナー)
    |      +-- initStreaming (ストリーミング初期化)
    |      +-- initCompleteInput (完全入力初期化)
    |      +-- next (次トークン取得)
    |      +-- nextAlloc (メモリ割当付き取得)
    |
    +-- parseFromSlice (高レベルパース)
    |      +-- parseFromTokenSource
    |             +-- innerParse
    |
    +-- Value (動的値)
    |      +-- ObjectMap
    |      +-- Array
    |
    +-- Stringify (シリアライズ)
           +-- beginObject
           +-- objectField
           +-- write
           +-- endObject
```

### データフロー図

```
[入力]                    [処理]                       [出力]

JSON文字列 ─────────────> Scanner ──────────────────> Token列
 "{\"a\":1}"                  |
                              v
                         parseFromSlice ─────────────> Zig構造体
                              |                         struct { a: i32 }
                              v
                         メモリ管理 ──────────────────> Parsed(T)

Zig値 ──────────────────> Stringify ─────────────────> JSON文字列
 .{ .num = 123 }              |                         "{\"num\":123}"
                              v
                         フォーマット適用 ───────────> 整形済みJSON
                         (whitespace option)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| json.zig | `lib/std/json.zig` | ソース | メインモジュール、公開API |
| Scanner.zig | `lib/std/json/Scanner.zig` | ソース | 低レベルトークンスキャナー |
| static.zig | `lib/std/json/static.zig` | ソース | 静的型パース実装 |
| dynamic.zig | `lib/std/json/dynamic.zig` | ソース | 動的Value実装 |
| Stringify.zig | `lib/std/json/Stringify.zig` | ソース | JSONシリアライズ |
| hashmap.zig | `lib/std/json/hashmap.zig` | ソース | ArrayHashMap実装 |
| test.zig | `lib/std/json/test.zig` | テスト | ユニットテスト |
| JSONTestSuite_test.zig | `lib/std/json/JSONTestSuite_test.zig` | テスト | RFC準拠テスト |
