# 機能設計書 54-std.meta

## 概要

本ドキュメントは、Zig標準ライブラリのメタプログラミングモジュール `std.meta` の機能設計を記述する。このモジュールは、型の内省（イントロスペクション）、型変換、コンパイル時計算などのメタプログラミング機能を提供する。

### 本機能の処理概要

std.metaは、Zigのコンパイル時プログラミング機能を活用した型操作とリフレクションのユーティリティを提供する。ジェネリックプログラミング、シリアライゼーション、コード生成などで広く使用される。

**業務上の目的・背景**：型安全なジェネリックコードを書く際、型の構造を動的に検査・操作する必要がある。std.metaは、構造体のフィールド列挙、enum値の取得、関数シグネチャの解析など、メタプログラミングに必要なプリミティブを提供し、ボイラープレートコードを削減する。

**機能の利用シーン**：
- ジェネリック関数での型制約チェック
- 構造体フィールドの自動列挙
- enumから文字列/値への変換
- 関数引数の型タプル生成
- 型の等価比較
- コンパイル時の型情報取得

**主要な処理内容**：
1. stringToEnum: 文字列からenum値への変換
2. alignment: 型のアライメント取得
3. Child/Elem: パラメータ化型の子型取得
4. sentinel: センチネル値の取得
5. fields/fieldNames: フィールド情報の取得
6. FieldEnum: フィールド名からenum生成
7. Tag: タグ付きunionのタグ型取得
8. eql: 型の深い等価比較
9. hasUniqueRepresentation: ビット表現の一意性チェック

**関連システム・外部連携**：
- std.json: JSONシリアライズ/デシリアライズで型情報使用
- std.fmt: フォーマット処理で型情報使用
- std.testing: テストユーティリティで型比較使用

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

計算処理 / メタプログラミング / 型操作

## 入力仕様

### 入力パラメータ

メタプログラミング関数の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| T | type | Yes | 操作対象の型 | comptime型 |
| str | []const u8 | 場合による | 文字列（stringToEnum等） | 有効な文字列 |
| field | FieldEnum(T) | 場合による | フィールド指定 | 有効なフィールド |

### 入力データソース

コンパイル時に決定される型情報、文字列リテラル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| enum_value | ?T | enum変換結果（nullは変換失敗） |
| alignment | comptime_int | アライメント値 |
| child_type | type | 子型 |
| fields | []const StructField/UnionField/etc | フィールド情報配列 |
| is_equal | bool | 等価比較結果 |

### 出力先

コンパイル時に評価される値

## 処理フロー

### 処理シーケンス

```
1. コンパイル時型情報取得
   └─ @typeInfo でType情報を取得
2. 型に応じた処理分岐
   └─ switch文で型種別ごとに処理
3. 結果のコンパイル時評価
   └─ comptime値として結果を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{関数種別}
    B -->|stringToEnum| C[@typeInfo(T).enum]
    B -->|fields| D[@typeInfo(T)で分岐]
    B -->|Child| E[@typeInfo(T).pointer等]
    C --> F[フィールド名マッチング]
    D --> G[struct/union/enum/error_set判定]
    E --> H[child型抽出]
    F --> I[enum値またはnull返却]
    G --> J[フィールドスライス返却]
    H --> K[型返却]
    I --> L[終了]
    J --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-54-01 | コンパイル時評価 | すべての関数はコンパイル時に評価可能 | 常時 |
| BR-54-02 | 型制約 | 各関数は特定の型カテゴリのみ受け付ける | 型不一致でコンパイルエラー |
| BR-54-03 | StaticStringMap最適化 | enum要素100個以下でStaticStringMap使用 | stringToEnum |

### 計算ロジック

stringToEnumの使用例:
```zig
const E = enum { A, B, C };
const result = std.meta.stringToEnum(E, "B");
// result == E.B

const invalid = std.meta.stringToEnum(E, "X");
// invalid == null
```

fieldsの使用例:
```zig
const S = struct { a: u8, b: f32 };
const field_list = std.meta.fields(S);
// field_list[0].name == "a"
// field_list[0].type == u8
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| @compileError | コンパイルエラー | 不正な型が渡された | 正しい型カテゴリを使用 |

### リトライ仕様

コンパイル時に評価されるため、リトライの概念は適用されない。

## トランザクション仕様

本機能はトランザクション制御を行わない（純粋なコンパイル時計算）。

## パフォーマンス要件

- すべてコンパイル時に評価されるため、実行時コストはゼロ
- 大量のenum要素（100超）ではStaticStringMapの代わりにインラインループを使用

## セキュリティ考慮事項

- コンパイル時のみ動作するため、実行時セキュリティリスクは最小

## 備考

- @typeInfo組み込み関数のラッパーユーティリティ
- ほとんどの関数は`inline`で定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

metaモジュールの全体構造と主要関数を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | meta.zig | `lib/std/meta.zig` | メタプログラミングユーティリティ全体 |

**読解のコツ**: 各関数のテストコードを見ると使用例と期待動作がわかりやすい。

**主要処理フロー**:
1. **9行目**: TrailerFlagsのインポート
2. **17-44行目**: stringToEnum実装（StaticStringMapまたはインラインループ）
3. **56-69行目**: alignment関数（ポインタのアライメント取得）
4. **83-91行目**: Child関数（配列/ポインタ/optional等の子型取得）
5. **102-118行目**: Elem関数（メモリスパン型の要素型取得）
6. **134-150行目**: sentinel関数（センチネル値取得）
7. **170-209行目**: Sentinel関数（センチネル付き型生成）
8. **318-332行目**: fields関数（struct/union/enum/error_setのフィールド取得）
9. **363-371行目**: fieldInfo関数（特定フィールド情報取得）
10. **398-406行目**: fieldNames関数（フィールド名配列取得）
11. **438-448行目**: tags関数（enum/error_setの値配列取得）
12. **465-482行目**: FieldEnum関数（フィールド名enumの生成）
13. **571-577行目**: Tag関数（タグ付きunionのタグ型取得）
14. **594-597行目**: activeTag関数（unionのアクティブタグ取得）
15. **619-671行目**: eql関数（深い等価比較）
16. **747-753行目**: fieldIndex関数（フィールド名からインデックス取得）
17. **785-801行目**: ArgsTuple関数（関数引数型のタプル生成）
18. **877-879行目**: isError関数（error_unionの判定）
19. **888-897行目**: hasFn関数（関数宣言の存在チェック）
20. **917-925行目**: hasMethod関数（メソッドの存在チェック）
21. **962-1005行目**: hasUniqueRepresentation関数（ビット表現一意性チェック）

#### Step 2: 型情報の取得を理解する

@typeInfo組み込み関数との関係を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | meta.zig | `lib/std/meta.zig` | switch (@typeInfo(T)) パターン |

**読解のコツ**: `std.builtin.Type`の構造を理解すると、各分岐の意味がわかる。

### プログラム呼び出し階層図

```
std.meta
    |
    +-- 文字列変換
    |      +-- stringToEnum (文字列 -> enum)
    |
    +-- 型情報取得
    |      +-- alignment (アライメント)
    |      +-- Child (子型)
    |      +-- Elem (要素型)
    |      +-- sentinel (センチネル値)
    |      +-- containerLayout (レイアウト)
    |
    +-- フィールド操作
    |      +-- fields (フィールド配列)
    |      +-- fieldNames (フィールド名配列)
    |      +-- fieldInfo (特定フィールド)
    |      +-- fieldIndex (フィールドインデックス)
    |      +-- FieldEnum (フィールドenum生成)
    |
    +-- enum/union操作
    |      +-- tags (タグ配列)
    |      +-- Tag (タグ型)
    |      +-- activeTag (アクティブタグ)
    |
    +-- 比較・検査
    |      +-- eql (深い等価比較)
    |      +-- hasUniqueRepresentation (ビット一意性)
    |      +-- hasFn (関数存在)
    |      +-- hasMethod (メソッド存在)
    |      +-- isError (エラー判定)
    |
    +-- 型生成
           +-- ArgsTuple (引数タプル)
           +-- Sentinel (センチネル付き型)
           +-- DeclEnum (宣言enum)
```

### データフロー図

```
[入力]                    [処理]                       [出力]

型 T ───────────────────> @typeInfo(T) ──────────────> Type情報
                                 |
                                 v
                         switch文で分岐 ─────────────> 型固有情報
                                 |
                                 v
                         結果計算 ────────────────────> comptime値

文字列 "field" ──────────> stringToEnum ──────────────> ?Enum値

struct { a: T } ─────────> fields() ──────────────────> []StructField
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| meta.zig | `lib/std/meta.zig` | ソース | メインモジュール（1100行超） |
| trailer_flags.zig | `lib/std/meta/trailer_flags.zig` | ソース | TrailerFlags実装 |
| builtin.zig | (コンパイラ組み込み) | 組み込み | Type定義 |
