# 機能設計書 55-std.atomic

## 概要

本ドキュメントは、Zig標準ライブラリのアトミック操作モジュール `std.atomic` の機能設計を記述する。このモジュールは、マルチスレッド環境でのデータ競合を防ぐためのアトミック（不可分）操作を提供する。

### 本機能の処理概要

std.atomicは、スレッドセーフなプリミティブ値の読み書きとアトミック操作を提供する。並行プログラミングにおける同期プリミティブの基盤として機能し、ロックフリーデータ構造の実装に不可欠である。

**業務上の目的・背景**：マルチスレッドプログラミングでは、複数のスレッドが同じメモリ位置に同時にアクセスするとデータ競合が発生する。std.atomicは、CPUのアトミック命令を活用してスレッドセーフな操作を提供し、正確な並行処理を可能にする。

**機能の利用シーン**：
- 参照カウント（ref counting）の実装
- ロックフリーキューの実装
- スピンロックの実装
- フラグやカウンタの並行更新
- メモリバリアの挿入
- キャッシュラインパディングの最適化

**主要な処理内容**：
1. Value(T): アトミック値のラッパー型
2. load/store: メモリオーダリング付きの読み書き
3. swap: アトミックな値交換
4. cmpxchgWeak/cmpxchgStrong: Compare-and-Swap操作
5. fetchAdd/fetchSub: 加減算と旧値取得
6. fetchAnd/fetchOr/fetchXor: ビット演算と旧値取得
7. spinLoopHint: スピンループ最適化ヒント
8. cache_line: CPUキャッシュライン定数

**関連システム・外部連携**：
- std.Thread: スレッド管理
- std.os: OSレベルの同期プリミティブ
- compiler_rt: アトミック操作の低レベル実装

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

計算処理 / 同期処理 / 並行制御

## 入力仕様

### 入力パラメータ

アトミック操作の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| T | type | Yes | 値の型 | 整数、浮動小数点、ポインタ、enum |
| value | T | 場合による | 操作する値 | - |
| order | AtomicOrder | Yes | メモリオーダリング | 有効なオーダリング |

### 入力データソース

メモリ上の値、コンパイル時定数

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| old_value | T | 操作前の値（fetch系） |
| exchanged | ?T | CAS失敗時の現在値（null=成功） |
| bit | u1 | ビット操作の旧ビット値 |

### 出力先

メモリ（アトミック操作対象）

## 処理フロー

### 処理シーケンス

```
1. 操作の準備
   └─ メモリオーダリングの指定
2. アトミック操作実行
   └─ CPUのアトミック命令を使用
3. メモリバリア
   └─ 指定されたオーダリングに従いバリア挿入
4. 結果の返却
   └─ 旧値または成否を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{操作種別}
    B -->|load| C[アトミックロード]
    B -->|store| D[アトミックストア]
    B -->|fetchAdd| E[加算と旧値取得]
    B -->|cmpxchg| F[比較と交換]
    C --> G[値返却]
    D --> H[完了]
    E --> G
    F --> I{成功?}
    I -->|Yes| J[null返却]
    I -->|No| K[現在値返却]
    G --> L[終了]
    H --> L
    J --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-55-01 | オーダリング必須 | すべてのアトミック操作にオーダリング指定必須 | 常時 |
| BR-55-02 | 型制限 | 整数、浮動小数点、ポインタ、enum、boolのみ対応 | Value(T)使用時 |
| BR-55-03 | アライメント | 値は自然アライメントである必要がある | 常時 |

### 計算ロジック

参照カウントの実装例:
```zig
const RefCount = struct {
    count: std.atomic.Value(usize),

    fn ref(self: *RefCount) void {
        _ = self.count.fetchAdd(1, .monotonic);
    }

    fn unref(self: *RefCount) void {
        if (self.count.fetchSub(1, .release) == 1) {
            _ = self.count.load(.acquire);
            // 解放処理
        }
    }
};
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

本モジュールはエラーを返さない（すべてのアトミック操作は成功するか、CASの場合は現在値を返す）。

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | - | - |

### リトライ仕様

cmpxchgWeakは疑似的に失敗する可能性があるため、ループでのリトライが一般的なパターン。

## トランザクション仕様

本機能はトランザクション制御を行わない（低レベルの同期プリミティブ）。

## パフォーマンス要件

- CPUのアトミック命令を直接使用
- スピンループヒント（pause命令等）によるCPUリソース効率化
- キャッシュラインパディングによるfalse sharing回避

## セキュリティ考慮事項

- データ競合の防止による並行処理の安全性確保
- メモリオーダリングによる可視性保証

## 備考

- cache_lineはCPUアーキテクチャにより異なる（x86_64: 128、arm: 32等）
- spinLoopHintはアーキテクチャ固有の命令を使用（x86: pause、arm: isb等）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

アトミックモジュールの全体構造とValue型を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | atomic.zig | `lib/std/atomic.zig` | アトミック操作ユーティリティ全体 |

**読解のコツ**: Value(T)構造体の各メソッドがどのアトミック操作に対応するか理解する。

**主要処理フロー**:
1. **1-2行目**: モジュールドキュメント
2. **2-115行目**: Value(T)構造体の定義
3. **9-11行目**: init関数
4. **13-15行目**: load関数（アトミックロード）
5. **17-19行目**: store関数（アトミックストア）
6. **21-23行目**: swap関数（アトミック交換）
7. **25-43行目**: cmpxchgWeak/cmpxchgStrong関数（CAS操作）
8. **45-75行目**: fetch系関数（加減算、ビット演算）
9. **86-113行目**: bit系関数（ビット設定/リセット/トグル）
10. **347-398行目**: spinLoopHint関数（スピンループ最適化）
11. **406-485行目**: cacheLineForCpu関数（キャッシュラインサイズ）
12. **493行目**: cache_line定数

#### Step 2: メモリオーダリングを理解する

AtomicOrderの各値の意味を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | builtin.zig | (コンパイラ組み込み) | AtomicOrder enum |

**メモリオーダリングの種類**:
- `.monotonic`: 単一変数の原子性のみ保証
- `.acquire`: 後続の読み込みがこの操作より前に並び替えられない
- `.release`: 先行の書き込みがこの操作より後に並び替えられない
- `.acq_rel`: acquireとreleaseの両方
- `.seq_cst`: 最も強い順序保証

### プログラム呼び出し階層図

```
std.atomic
    |
    +-- Value(T) (アトミック値ラッパー)
    |      +-- init (初期化)
    |      +-- load (読み込み)
    |      +-- store (書き込み)
    |      +-- swap (交換)
    |      +-- cmpxchgWeak (弱いCAS)
    |      +-- cmpxchgStrong (強いCAS)
    |      +-- fetchAdd (加算)
    |      +-- fetchSub (減算)
    |      +-- fetchMin/fetchMax (最小/最大)
    |      +-- fetchAnd/fetchOr/fetchXor/fetchNand (ビット演算)
    |      +-- bitSet/bitReset/bitToggle (ビット操作)
    |      +-- rmw (汎用RMW)
    |
    +-- spinLoopHint (スピンループヒント)
    |
    +-- cache_line (キャッシュライン定数)
    |
    +-- cacheLineForCpu (CPU別キャッシュライン)
```

### データフロー図

```
[スレッドA]               [共有メモリ]                [スレッドB]

値 X ───────────────────> アトミックストア ───────────> 可視化
(release)                  Value.raw
                              |
                              v
                          メモリバリア
                              |
                              v
読み取り <─────────────── アトミックロード <────────── 要求
(acquire)                  Value.raw

fetchAdd(1) ────────────> read-modify-write ─────────> 旧値 + 結果
                          (原子的)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| atomic.zig | `lib/std/atomic.zig` | ソース | メインモジュール（503行） |
| atomics.zig | `lib/compiler_rt/atomics.zig` | ソース | コンパイラランタイムのアトミック実装 |
| builtin.zig | (コンパイラ) | 組み込み | AtomicOrder, AtomicRmwOp定義 |
