# 機能設計書 56-std.posix

## 概要

本ドキュメントは、Zig標準ライブラリのPOSIXモジュール `std.posix` の機能設計を記述する。このモジュールは、POSIX互換APIを提供し、Unix系システムでの低レベルOS操作を可能にする。

### 本機能の処理概要

std.posixは、POSIX標準に準拠したシステムコールラッパーを提供する。ファイル操作、プロセス管理、シグナル処理、ソケット通信など、OSレベルの操作を統一的なインターフェースで実行できる。

**業務上の目的・背景**：Unix系OSでの低レベルプログラミングには、POSIXシステムコールへのアクセスが必要である。std.posixは、Linux、macOS、BSD系OSなど様々なPOSIX互換システムで動作する移植性の高いコードを書くための基盤を提供する。直接libcを呼び出すよりも高い抽象度でありながら、std.fsやstd.netよりも低レベルな制御が可能。

**機能の利用シーン**：
- ファイルディスクリプタの直接操作
- プロセスの生成と管理
- シグナルハンドリング
- ソケットプログラミング
- メモリマッピング（mmap）
- 端末制御（termios）
- システム情報の取得

**主要な処理内容**：
1. ファイル操作: open, close, read, write, lseek, fstat
2. プロセス管理: fork, exec, wait, kill, getpid
3. シグナル: sigaction, sigprocmask, raise
4. ソケット: socket, bind, listen, accept, connect
5. メモリ: mmap, munmap, mprotect
6. 時刻: clock_gettime, nanosleep
7. 端末: tcgetattr, tcsetattr

**関連システム・外部連携**：
- std.fs: 高レベルファイルシステムAPI（内部でposixを使用）
- std.net: 高レベルネットワークAPI（内部でposixを使用）
- std.os.linux: Linux固有のシステムコール
- libc: libcリンク時はlibcを経由

**権限による制御**：POSIXシステムコールはOS権限に従う（root権限、ファイルパーミッション等）。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

システムコール / OS操作 / 低レベルI/O

## 入力仕様

### 入力パラメータ

システムコールの一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| fd | fd_t | 場合による | ファイルディスクリプタ | 有効なFD |
| path | []const u8 | 場合による | ファイルパス | 有効なパス |
| buf | []u8 | 場合による | データバッファ | - |
| flags | O | 場合による | オープンフラグ | - |
| mode | mode_t | 場合による | パーミッション | - |

### 入力データソース

ファイルシステム、ネットワーク、メモリ、プロセス情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| fd | fd_t | 新しいファイルディスクリプタ |
| bytes_read | usize | 読み取りバイト数 |
| pid | pid_t | プロセスID |
| stat | Stat | ファイル情報 |

### 出力先

返り値、出力バッファ、errnoによるエラー情報

## 処理フロー

### 処理シーケンス

```
1. 引数の準備
   └─ Zig型からC互換型への変換
2. システムコール実行
   └─ libc経由または直接syscall
3. エラー処理
   └─ errnoをZigエラーに変換
4. 結果の返却
   └─ 成功値またはエラー
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{libc使用?}
    B -->|Yes| C[libc関数呼び出し]
    B -->|No| D[直接syscall]
    C --> E{errno == 0?}
    D --> E
    E -->|Yes| F[成功値返却]
    E -->|No| G[errnoをZigエラーに変換]
    G --> H[エラー返却]
    F --> I[終了]
    H --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-56-01 | libc選択 | link_libcまたはWindows/WASIでlibcを使用 | use_libc条件 |
| BR-56-02 | errno変換 | errnoをZigのエラー型に変換 | すべてのsyscall |
| BR-56-03 | パス変換 | ZigスライスをC文字列に変換 | パス引数 |
| BR-56-04 | EINTR再試行 | EINTRは自動的に再試行 | read/write等 |

### 計算ロジック

ファイル読み取りの例:
```zig
pub fn read(fd: fd_t, buf: []u8) ReadError!usize {
    const max_count = switch (native_os) {
        .linux => 0x7ffff000,
        .macos => maxInt(i32),
        else => maxInt(isize),
    };
    while (true) {
        const rc = system.read(fd, buf.ptr, @min(buf.len, max_count));
        switch (errno(rc)) {
            .SUCCESS => return @intCast(rc),
            .INTR => continue,
            .AGAIN => return error.WouldBlock,
            // ...
        }
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| AccessDenied | 権限エラー | EACCES/EPERM | 権限確認 |
| FileNotFound | ファイルエラー | ENOENT | パス確認 |
| WouldBlock | ブロックエラー | EAGAIN/EWOULDBLOCK | 再試行またはノンブロッキング処理 |
| ConnectionResetByPeer | 接続エラー | ECONNRESET | 再接続 |
| InputOutput | I/Oエラー | EIO | デバイス確認 |
| SystemResources | リソースエラー | ENOMEM/ENOBUFS | リソース解放 |

### リトライ仕様

EINTRエラーは内部で自動リトライ。EAGAINはWouldBlockエラーとして呼び出し元に返却。

## トランザクション仕様

POSIXレベルではトランザクション概念なし。ファイルロック（flock）で排他制御可能。

## パフォーマンス要件

- システムコールのオーバーヘッドは避けられない
- 読み取り上限はOS依存（Linux: 0x7ffff000）
- ノンブロッキングI/Oで効率的な並行処理

## セキュリティ考慮事項

- パス文字列の検証（パストラバーサル対策）
- ファイルパーミッションの適切な設定
- 信頼できない入力のサニタイズ

## 備考

- Windows非対応（Windows APIは別モジュール）
- WASIでは一部機能が制限される
- スレッドキャンセルは非サポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

POSIXモジュールの全体構造と型定義を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | posix.zig | `lib/std/posix.zig` | POSIXラッパー全体 |

**読解のコツ**: `system`変数でlibc/linuxを切り替えている点に注目。

**主要処理フロー**:
1. **1-15行目**: モジュールドキュメント
2. **34-38行目**: use_libc定数（libc使用判定）
3. **45-63行目**: system変数（libc/linux/plan9の選択）
4. **65-206行目**: 型エイリアス（AF, AT, E, O, PROT, MAP等）
5. **283-289行目**: errno関数
6. **300-313行目**: close関数
7. **405-448行目**: read関数
8. **450-466行目**: openat関数
9. **474-524行目**: openatZ関数

#### Step 2: 主要システムコールを理解する

代表的なシステムコールラッパーの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | posix.zig | `lib/std/posix.zig` | read/write/open等の実装 |

**読解のコツ**: while(true)ループでEINTRをハンドリングしているパターンが多い。

### プログラム呼び出し階層図

```
std.posix
    |
    +-- system (抽象化レイヤー)
    |      +-- std.c (libc)
    |      +-- std.os.linux (Linux syscall)
    |      +-- std.os.plan9 (Plan 9)
    |
    +-- ファイル操作
    |      +-- close
    |      +-- read
    |      +-- write
    |      +-- openat/openatZ
    |      +-- lseek
    |
    +-- プロセス管理
    |      +-- kill
    |      +-- raise
    |      +-- reboot
    |
    +-- メモリ管理
    |      +-- mmap
    |      +-- munmap
    |      +-- mprotect
    |
    +-- ソケット
           +-- socket
           +-- bind
           +-- listen
           +-- accept
           +-- connect
```

### データフロー図

```
[アプリケーション]          [std.posix]              [カーネル]

read(fd, buf) ─────────> errno(system.read()) ────> システムコール
                               |                         |
                               v                         v
                         errno変換 <───────────────── 結果/errno
                               |
                               v
                         Zigエラー/成功値 ─────────> 呼び出し元
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| posix.zig | `lib/std/posix.zig` | ソース | メインモジュール |
| test.zig | `lib/std/posix/test.zig` | テスト | POSIXテスト |
| linux.zig | `lib/std/os/linux.zig` | ソース | Linux syscall |
| c.zig | `lib/std/c.zig` | ソース | libc互換レイヤー |
| windows.zig | `lib/std/os/windows.zig` | ソース | Windows API（参考） |
