# 機能設計書 57-std.Build

## 概要

本ドキュメントは、Zig標準ライブラリのビルドシステムモジュール `std.Build` の機能設計を記述する。このモジュールは、ZigのネイティブビルドシステムのコアAPIを提供し、build.zigファイルでのプロジェクト構成を可能にする。

### 本機能の処理概要

std.Buildは、Zigプロジェクトのビルド定義とタスク管理のためのフレームワークを提供する。コンパイル設定、依存関係管理、カスタムビルドステップの定義など、ビルドプロセス全体を制御する。

**業務上の目的・背景**：従来のビルドシステム（Make、CMake等）は設定が複雑で、クロスコンパイルが困難な場合が多い。std.Buildは、Zig言語自体でビルド設定を記述でき、Zigの強力なコンパイル時機能を活用した型安全で再現性のあるビルドを実現する。また、Zigの卓越したクロスコンパイル能力と統合されている。

**機能の利用シーン**：
- build.zigでのプロジェクト構成
- 実行ファイル/ライブラリのビルド定義
- 依存パッケージの管理
- テストの実行
- カスタムビルドステップの定義
- インストール先の指定

**主要な処理内容**：
1. Build: ビルドグラフの構築と管理
2. Step: ビルドステップの抽象化
3. Module: Zigモジュールの定義
4. Cache: コンパイルキャッシュ管理
5. LazyPath: 遅延評価されるパス
6. addExecutable/addStaticLibrary/addSharedLibrary: ターゲット追加
7. dependency: 依存パッケージの解決

**関連システム・外部連携**：
- Zigコンパイラ: ビルドステップの実行
- build.zig.zon: パッケージマニフェスト
- グローバルキャッシュ: 依存パッケージの保存

**権限による制御**：ファイルシステムへのアクセス権限に従う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 28 | プロジェクト初期化画面 | 補助機能 | ビルドシステムAPIを使用したbuild.zig生成 |

## 機能種別

ビルド管理 / 依存解決 / タスク実行

## 入力仕様

### 入力パラメータ

Build作成の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| graph | *Graph | Yes | 共有状態 | 有効なGraph |
| build_root | Cache.Directory | Yes | build.zigのディレクトリ | 有効なパス |
| cache_root | Cache.Directory | Yes | キャッシュルート | 有効なパス |
| available_deps | AvailableDeps | No | 利用可能な依存関係 | - |

### 入力データソース

build.zig、build.zig.zon、コマンドライン引数

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| artifacts | []Artifact | ビルド成果物 |
| install_path | []const u8 | インストール先パス |
| cache_dir | Cache.Directory | キャッシュディレクトリ |

### 出力先

ファイルシステム（zig-outディレクトリ等）

## 処理フロー

### 処理シーケンス

```
1. Build初期化
   └─ グラフと設定の初期化
2. ビルドステップ定義
   └─ addExecutable等でターゲット追加
3. 依存関係解決
   └─ dependency()で依存パッケージ取得
4. ビルドグラフ構築
   └─ ステップ間の依存関係設定
5. 実行
   └─ zigビルドランナーがステップを実行
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[Build.create]
    B --> C[build.zigのbuild関数呼び出し]
    C --> D{ターゲット追加}
    D -->|Executable| E[addExecutable]
    D -->|Library| F[addStaticLibrary]
    D -->|Test| G[addTest]
    E --> H[依存関係設定]
    F --> H
    G --> H
    H --> I[installArtifact]
    I --> J[ビルドグラフ完成]
    J --> K[ステップ実行]
    K --> L[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-57-01 | ステップ依存 | ステップは依存関係順に実行 | 常時 |
| BR-57-02 | キャッシュ活用 | 変更がないステップはスキップ | インクリメンタルビルド |
| BR-57-03 | 遅延評価 | LazyPathはステップ実行時に解決 | パス参照時 |
| BR-57-04 | 依存ハッシュ | 依存パッケージはハッシュで識別 | dependency() |

### 計算ロジック

build.zigの典型例:
```zig
pub fn build(b: *std.Build) void {
    const target = b.standardTargetOptions(.{});
    const optimize = b.standardOptimizeOption(.{});

    const exe = b.addExecutable(.{
        .name = "myapp",
        .root_source_file = b.path("src/main.zig"),
        .target = target,
        .optimize = optimize,
    });

    b.installArtifact(exe);

    const run_cmd = b.addRunArtifact(exe);
    const run_step = b.step("run", "Run the application");
    run_step.dependOn(&run_cmd.step);
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| OutOfMemory | メモリエラー | メモリ不足 | メモリ解放 |
| PkgConfigCrashed | 依存エラー | pkg-config失敗 | pkg-config確認 |
| PkgConfigNotInstalled | 依存エラー | pkg-configなし | インストール |
| ReadFailure | I/Oエラー | ファイル読み取り失敗 | パス確認 |

### リトライ仕様

ビルドステップは通常リトライせず、エラー時は中断。

## トランザクション仕様

本機能はトランザクション制御を行わない。キャッシュにより部分的な再実行は可能。

## パフォーマンス要件

- インクリメンタルビルドによる高速な再ビルド
- 並列ステップ実行のサポート
- キャッシュによるコンパイル結果の再利用

## セキュリティ考慮事項

- 依存パッケージのハッシュ検証
- サンドボックス環境でのビルド（将来）

## 備考

- arenaアロケータ使用でメモリ管理を簡素化
- GraphはすべてのBuildインスタンス間で共有

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Buildモジュールの全体構造とコア型を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Build.zig | `lib/std/Build.zig` | ビルドシステムのコアAPI |

**読解のコツ**: Build構造体のフィールドを理解し、Graphとの関係を把握する。

**主要処理フロー**:
1. **1-17行目**: インポートと基本定義
2. **19-26行目**: サブモジュールのエクスポート（Cache, Step, Module等）
3. **27-100行目**: Build構造体のフィールド定義
4. **102-108行目**: ReleaseMode enum定義
5. **110-136行目**: Graph構造体（共有状態）
6. **259-323行目**: create関数（Build初期化）

#### Step 2: Stepを理解する

ビルドステップの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Step.zig | `lib/std/Build/Step.zig` | ビルドステップの抽象化 |

#### Step 3: Moduleを理解する

Zigモジュールの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Module.zig | `lib/std/Build/Module.zig` | モジュール定義 |

#### Step 4: Cacheを理解する

キャッシュメカニズムを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Cache.zig | `lib/std/Build/Cache.zig` | コンパイルキャッシュ |

### プログラム呼び出し階層図

```
std.Build
    |
    +-- Graph (共有状態)
    |      +-- cache (キャッシュ管理)
    |      +-- host (ホストターゲット)
    |      +-- zig_exe (Zigコンパイラパス)
    |
    +-- Build (ビルド定義)
    |      +-- addExecutable
    |      +-- addStaticLibrary
    |      +-- addSharedLibrary
    |      +-- addTest
    |      +-- addRunArtifact
    |      +-- step
    |      +-- installArtifact
    |      +-- dependency
    |
    +-- Step (ビルドステップ)
    |      +-- Compile
    |      +-- Run
    |      +-- WriteFile
    |      +-- InstallArtifact
    |      +-- InstallFile
    |
    +-- Module (モジュール定義)
    |
    +-- Cache (キャッシュ管理)
           +-- Directory
           +-- Hash
```

### データフロー図

```
[入力]                    [処理]                       [出力]

build.zig ──────────────> Build.create ──────────────> Build
                              |
                              v
addExecutable() ─────────> Step.Compile ─────────────> Artifact
                              |
                              v
依存関係設定 ────────────> ビルドグラフ構築 ──────────> 実行順序
                              |
                              v
zig build ───────────────> ステップ実行 ──────────────> zig-out/
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Build.zig | `lib/std/Build.zig` | ソース | メインモジュール |
| Step.zig | `lib/std/Build/Step.zig` | ソース | ビルドステップ基底 |
| Module.zig | `lib/std/Build/Module.zig` | ソース | モジュール定義 |
| Cache.zig | `lib/std/Build/Cache.zig` | ソース | キャッシュ管理 |
| Watch.zig | `lib/std/Build/Watch.zig` | ソース | ファイル監視 |
| Fuzz.zig | `lib/std/Build/Fuzz.zig` | ソース | ファジングサポート |
| WebServer.zig | `lib/std/Build/WebServer.zig` | ソース | ビルドサーバー |
| abi.zig | `lib/std/Build/abi.zig` | ソース | ABI情報 |
