# 機能設計書 58-std.Target

## 概要

本ドキュメントは、Zig標準ライブラリのターゲット情報モジュール `std.Target` の機能設計を記述する。このモジュールは、コンパイルターゲット（CPU、OS、ABI）の情報と操作を提供する。

### 本機能の処理概要

std.Targetは、Zigがサポートする様々なコンパイルターゲットの情報を表現・管理する。クロスコンパイルの設定、条件付きコンパイル、プラットフォーム固有のコード分岐など、多様な用途で使用される。

**業務上の目的・背景**：Zigは「クロスコンパイルをファーストクラス」としており、任意のターゲット向けにコードをコンパイルできる。std.Targetは、この強力なクロスコンパイル能力の基盤として、CPUアーキテクチャ、OS、ABIの組み合わせを統一的に表現し、ターゲット依存の設定を管理する。

**機能の利用シーン**：
- ビルド時のターゲット指定（-target）
- コンパイル時のプラットフォーム分岐（@import("builtin")）
- 実行ファイル拡張子の決定
- ライブラリパス/プレフィックスの決定
- ABI互換性の判定
- CPUフィーチャーの検出

**主要な処理内容**：
1. Cpu: CPUアーキテクチャと機能の表現
2. Os: オペレーティングシステムとバージョン情報
3. Abi: Application Binary Interface
4. ObjectFormat: 出力形式（ELF、Mach-O、COFF等）
5. DynamicLinker: 動的リンカのパス
6. Query: ターゲットのクエリ（部分指定）

**関連システム・外部連携**：
- std.Build: ビルドシステムでのターゲット指定
- Zigコンパイラ: コンパイル時のターゲット情報
- @import("builtin"): コンパイル時定数としてのターゲット情報

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

本機能はライブラリモジュールであり、直接的な画面関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリ機能として他の画面から間接的に利用される |

## 機能種別

データ構造 / 設定管理 / 条件分岐

## 入力仕様

### 入力パラメータ

Target関連の一般的なパラメータ:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| cpu | Cpu | Yes | CPUアーキテクチャ | 有効なCpu |
| os | Os | Yes | オペレーティングシステム | 有効なOs |
| abi | Abi | Yes | ABI | 有効なAbi |
| ofmt | ObjectFormat | No | 出力形式 | - |

### 入力データソース

コマンドライン引数（-target）、build.zig、@import("builtin")

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| exe_ext | [:0]const u8 | 実行ファイル拡張子 |
| lib_prefix | [:0]const u8 | ライブラリプレフィックス |
| lib_suffix | [:0]const u8 | ライブラリサフィックス |
| dynamic_linker | DynamicLinker | 動的リンカパス |

### 出力先

コンパイル時定数、ビルド設定

## 処理フロー

### 処理シーケンス

```
1. ターゲット情報の取得
   └─ コマンドライン/-targetまたはホストから
2. 正規化
   └─ デフォルト値の適用、矛盾の解決
3. 派生情報の計算
   └─ 拡張子、プレフィックス、リンカパス等
4. ビルド設定への反映
   └─ コンパイラオプションの決定
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{ターゲット指定}
    B -->|明示的| C[Query解析]
    B -->|ホスト| D[ネイティブ検出]
    C --> E[CPU解析]
    D --> E
    E --> F[OS解析]
    F --> G[ABI解析]
    G --> H[ObjectFormat決定]
    H --> I[DynamicLinker決定]
    I --> J[Target完成]
    J --> K[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-58-01 | ABI互換性 | ABIはCPU/OSと互換性が必要 | ターゲット構築時 |
| BR-58-02 | バージョン範囲 | OSバージョンは最小/最大で範囲指定 | OS依存コード |
| BR-58-03 | デフォルト選択 | 未指定項目にはデフォルト値を適用 | Query解決時 |
| BR-58-04 | CPU機能検出 | 実行時CPU機能はランタイムで検出可能 | ネイティブ実行時 |

### 計算ロジック

実行ファイル拡張子の決定例:
```zig
pub fn exeFileExt(tag: Tag, arch: Cpu.Arch) [:0]const u8 {
    return switch (tag) {
        .windows => ".exe",
        .uefi => ".efi",
        .plan9 => arch.plan9Ext(),
        else => switch (arch) {
            .wasm32, .wasm64 => ".wasm",
            else => "",
        },
    };
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidOperatingSystemVersion | パースエラー | 不正なOSバージョン文字列 | 形式確認 |

### リトライ仕様

ターゲット解析は決定論的であり、リトライは通常不要。

## トランザクション仕様

本機能はトランザクション制御を行わない（純粋なデータ構造）。

## パフォーマンス要件

- すべてコンパイル時に評価可能
- ランタイムのCPU機能検出は起動時に1回のみ

## セキュリティ考慮事項

- ターゲット情報による攻撃面の特定を考慮
- セキュリティ機能（スタックプロテクタ等）のターゲット依存設定

## 備考

- Queryは部分指定を許容（未指定はnull）
- Targetは完全に解決された状態

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Targetモジュールの全体構造とコア型を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Target.zig | `lib/std/Target.zig` | ターゲット情報の表現 |

**読解のコツ**: Target構造体のフィールドとOs、Cpu、Abi enumの関係を理解する。

**主要処理フロー**:
1. **1-5行目**: モジュールドキュメント
2. **6-10行目**: Target構造体のフィールド（cpu, os, abi, ofmt, dynamic_linker）
3. **12行目**: Query型のインポート
4. **14-216行目**: Os構造体とOs.Tag enum
5. **18-89行目**: Os.Tag enumの値（各OS名）
6. **78-89行目**: isDarwin(), isBSD()ヘルパー
7. **99-134行目**: exeFileExt(), staticLibSuffix(), dynamicLibSuffix()
8. **220-314行目**: WindowsVersion enum
9. **316-346行目**: HurdVersionRange, LinuxVersionRange
10. **372-431行目**: VersionRange union

#### Step 2: CPUアーキテクチャを理解する

Cpu型の詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Target.zig | `lib/std/Target.zig` | Cpu構造体とArch enum |

#### Step 3: Queryを理解する

部分指定のターゲットクエリを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Query.zig | `lib/std/Target/Query.zig` | ターゲットクエリ |

### プログラム呼び出し階層図

```
std.Target
    |
    +-- cpu: Cpu
    |      +-- arch: Arch (x86_64, aarch64, etc.)
    |      +-- model: *const Model
    |      +-- features: FeatureSet
    |
    +-- os: Os
    |      +-- tag: Tag (linux, windows, macos, etc.)
    |      +-- version_range: VersionRange
    |             +-- semver (SemVer範囲)
    |             +-- linux (Linux固有)
    |             +-- windows (Windows固有)
    |
    +-- abi: Abi
    |      +-- none, gnu, musl, msvc, etc.
    |
    +-- ofmt: ObjectFormat
    |      +-- elf, macho, coff, wasm, etc.
    |
    +-- dynamic_linker: DynamicLinker
           +-- パス文字列
```

### データフロー図

```
[入力]                    [処理]                       [出力]

-target x86_64-linux ──> Query.parse ────────────────> Query
                              |
                              v
                         resolveDefaults ────────────> Target
                              |
                              v
                         派生情報計算 ────────────────> 拡張子等
                              |
                              v
builtin.target ──────────────────────────────────────> @import("builtin")
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Target.zig | `lib/std/Target.zig` | ソース | メインモジュール |
| Query.zig | `lib/std/Target/Query.zig` | ソース | ターゲットクエリ |
| x86.zig | `lib/std/Target/x86.zig` | ソース | x86 CPU定義 |
| aarch64.zig | `lib/std/Target/aarch64.zig` | ソース | ARM64 CPU定義 |
| riscv.zig | `lib/std/Target/riscv.zig` | ソース | RISC-V CPU定義 |
