# 機能設計書 59-compiler_rt

## 概要

本ドキュメントは、Zigのコンパイラランタイムライブラリ `compiler_rt` の機能設計を記述する。このモジュールは、コンパイラが生成するコードが依存する低レベルのランタイム関数を提供する。

### 本機能の処理概要

compiler_rtは、CPUがネイティブにサポートしていない演算（ソフトウェア浮動小数点、大きな整数演算等）をソフトウェアで実装したランタイムライブラリである。LLVMのcompiler-rtやGCCのlibgccに相当する。

**業務上の目的・背景**：すべてのCPUがすべての演算をハードウェアでサポートしているわけではない。特に、64ビット整数演算を32ビットCPUで行う場合や、浮動小数点演算をFPUなしで行う場合、ソフトウェアによるエミュレーションが必要となる。compiler_rtは、これらのエミュレーション関数を提供し、Zigコードが様々なターゲットで動作することを可能にする。

**機能の利用シーン**：
- 32ビットターゲットでの64ビット整数演算
- FPUなしターゲットでの浮動小数点演算
- 128ビット整数演算
- 複素数演算
- スタックプローブ（大きなスタック割り当て）
- アトミック操作のフォールバック

**主要な処理内容**：
1. 整数演算: 乗算、除算、剰余、シフト、比較
2. 浮動小数点変換: 整数⇔浮動小数点、精度変換
3. 浮動小数点演算: 加減乗除、比較、否定
4. 数学関数: sin, cos, tan, exp, log, sqrt
5. メモリ操作: memcpy, memset, memmove, memcmp
6. スタックプローブ: __chkstk等
7. アトミック操作: ロックベースのフォールバック

**関連システム・外部連携**：
- Zigコンパイラ: ランタイム関数の呼び出し生成
- リンカ: ランタイムライブラリのリンク
- 各種libc: 一部関数はlibcに委譲

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 実行ファイルビルド画面 | 補助機能 | コンパイラランタイムの組み込み |

## 機能種別

ランタイムライブラリ / エミュレーション / 低レベル演算

## 入力仕様

### 入力パラメータ

各ランタイム関数に固有（例：除算関数）:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| a | i64/u64/f32/f64等 | Yes | 被演算子 | - |
| b | i64/u64/f32/f64等 | Yes | 演算子 | ゼロ除算チェック |

### 入力データソース

コンパイラ生成コードからの暗黙的呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | 演算結果型 | 演算結果 |

### 出力先

呼び出し元への返り値

## 処理フロー

### 処理シーケンス

```
1. 関数呼び出し
   └─ コンパイラが生成したcall命令
2. 引数の前処理
   └─ 必要に応じて正規化
3. ソフトウェア演算
   └─ アルゴリズムに従い計算
4. 結果の返却
   └─ 適切な型で返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{演算種別}
    B -->|整数演算| C[整数処理ルーチン]
    B -->|浮動小数点| D[浮動小数点処理ルーチン]
    B -->|メモリ| E[メモリ操作ルーチン]
    B -->|アトミック| F[アトミック操作ルーチン]
    C --> G[結果返却]
    D --> G
    E --> G
    F --> G
    G --> H[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-59-01 | IEEE 754準拠 | 浮動小数点演算はIEEE 754に準拠 | 浮動小数点演算 |
| BR-59-02 | 最適化禁止 | panic関数は最適化されない | ランタイムエラー時 |
| BR-59-03 | リンケージ | weak linkageでユーザー定義に置換可能 | エクスポート時 |

### 計算ロジック

64ビット除算（32ビットターゲット）の例:
```zig
fn __divdi3(a: i64, b: i64) i64 {
    // 符号処理
    // 64ビット値を32ビット演算で除算
    // 結果を64ビットで返却
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| Division by zero | 実行時エラー | ゼロ除算 | panic発生 |
| Stack overflow | 実行時エラー | スタック不足 | スタック拡大または終了 |

### リトライ仕様

ランタイム関数はリトライしない（エラー時はpanicまたはトラップ）。

## トランザクション仕様

本機能はトランザクション制御を行わない。

## パフォーマンス要件

- ハードウェア命令が利用可能な場合は使用しない（コンパイラが判断）
- ソフトウェア実装は可能な限り効率的に
- インライン展開可能な関数はインライン

## セキュリティ考慮事項

- スタックプローブによるスタックオーバーフロー検出
- SSP（Stack Smashing Protector）のサポート

## 備考

- macOSではlibSystemに同等関数が存在
- Windowsではmsvcrtに同等関数が存在
- UEFIでは_fltused変数が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

compiler_rtモジュールの全体構造とインポートを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | compiler_rt.zig | `lib/compiler_rt.zig` | ランタイムライブラリのエントリポイント |

**読解のコツ**: comptime ブロックで各種ルーチンをインポートしている。

**主要処理フロー**:
1. **1-10行目**: panic設定（安全性 vs サイズのトレードオフ）
2. **22-55行目**: 整数ルーチンのインポート
3. **56-165行目**: 浮動小数点変換ルーチン
4. **166-182行目**: 浮動小数点比較ルーチン
5. **184-232行目**: 浮動小数点算術ルーチン
6. **233-251行目**: 数学関数ルーチン
7. **253-285行目**: 追加ルーチン（memcpy, スタックプローブ等）
8. **287-293行目**: Windows/UEFI固有の_fltused

#### Step 2: 整数演算を理解する

整数演算の実装パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | divti3.zig | `lib/compiler_rt/divti3.zig` | 128ビット除算 |
| 2-2 | mulXi3.zig | `lib/compiler_rt/mulXi3.zig` | 大きな整数乗算 |

#### Step 3: 浮動小数点演算を理解する

浮動小数点演算の実装パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | addf3.zig | `lib/compiler_rt/addf3.zig` | 浮動小数点加算 |
| 3-2 | mulf3.zig | `lib/compiler_rt/mulf3.zig` | 浮動小数点乗算 |
| 3-3 | divdf3.zig | `lib/compiler_rt/divdf3.zig` | 倍精度除算 |

#### Step 4: メモリ操作を理解する

メモリ操作関数の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | memcpy.zig | `lib/compiler_rt/memcpy.zig` | メモリコピー |
| 4-2 | memset.zig | `lib/compiler_rt/memset.zig` | メモリ設定 |

### プログラム呼び出し階層図

```
compiler_rt.zig (エントリポイント)
    |
    +-- 整数ルーチン
    |      +-- count0bits.zig (clz, ctz)
    |      +-- parity.zig (パリティ)
    |      +-- popcount.zig (ポップカウント)
    |      +-- shift.zig (シフト)
    |      +-- mulXi3.zig (乗算)
    |      +-- divti3.zig (除算)
    |
    +-- 浮動小数点変換
    |      +-- extendf.zig (精度拡張)
    |      +-- truncf.zig (精度縮小)
    |      +-- int_from_float.zig (整数変換)
    |      +-- float_from_int.zig (浮動小数点変換)
    |
    +-- 浮動小数点演算
    |      +-- addf3.zig (加算)
    |      +-- mulf3.zig (乗算)
    |      +-- divdf3.zig (除算)
    |      +-- comparef.zig (比較)
    |
    +-- 数学関数
    |      +-- sin.zig, cos.zig, tan.zig
    |      +-- exp.zig, log.zig, log2.zig
    |      +-- sqrt.zig, fma.zig
    |
    +-- その他
           +-- memcpy.zig, memset.zig
           +-- stack_probe.zig
           +-- atomics.zig
           +-- ssp.zig (スタックプロテクタ)
```

### データフロー図

```
[コンパイル時]            [リンク時]                    [実行時]

Zigソースコード ──────> Zigコンパイラ ──────────────> マシンコード
                            |
                            v
                      ランタイム関数呼び出し挿入
                            |
                            v
compiler_rt.a ─────────> リンカ ──────────────────> 実行ファイル
                            |
                            v
                      ランタイム関数実行 ─────────> 結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| compiler_rt.zig | `lib/compiler_rt.zig` | ソース | エントリポイント |
| common.zig | `lib/compiler_rt/common.zig` | ソース | 共通定義 |
| divti3.zig | `lib/compiler_rt/divti3.zig` | ソース | 128ビット除算 |
| mulXi3.zig | `lib/compiler_rt/mulXi3.zig` | ソース | 大整数乗算 |
| addf3.zig | `lib/compiler_rt/addf3.zig` | ソース | 浮動小数点加算 |
| mulf3.zig | `lib/compiler_rt/mulf3.zig` | ソース | 浮動小数点乗算 |
| sin.zig | `lib/compiler_rt/sin.zig` | ソース | sin関数 |
| cos.zig | `lib/compiler_rt/cos.zig` | ソース | cos関数 |
| memcpy.zig | `lib/compiler_rt/memcpy.zig` | ソース | メモリコピー |
| memset.zig | `lib/compiler_rt/memset.zig` | ソース | メモリ設定 |
| stack_probe.zig | `lib/compiler_rt/stack_probe.zig` | ソース | スタックプローブ |
| atomics.zig | `lib/compiler_rt/atomics.zig` | ソース | アトミック操作 |
| ssp.zig | `lib/compiler_rt/ssp.zig` | ソース | スタックプロテクタ |
