# 機能設計書 60-ubsan_rt

## 概要

本ドキュメントは、Zigの未定義動作サニタイザランタイム `ubsan_rt` の機能設計を記述する。このモジュールは、C/C++コードにおける未定義動作を実行時に検出・報告するためのランタイムハンドラを提供する。

### 本機能の処理概要

ubsan_rtは、LLVMの未定義動作サニタイザ（UBSan）が生成するチェックコードから呼び出されるハンドラ関数群を実装する。整数オーバーフロー、ゼロ除算、不正なアライメント、境界外アクセスなど、様々な未定義動作を検出してpanicを発生させる。

**業務上の目的・背景**：C/C++言語には多くの未定義動作が存在し、これらは深刻なセキュリティ脆弱性やバグの原因となる。UBSanは、これらの未定義動作を実行時に検出するためのコンパイラ機能であり、ubsan_rtはZigにおけるそのランタイムサポートを提供する。

**機能の利用シーン**：
- `-fsanitize=undefined`オプションでコンパイルされたC/C++コードの実行
- 整数演算のオーバーフロー検出
- 型不正の検出
- ポインタ操作の異常検出
- シフト演算の範囲外検出

**主要な処理内容**：
1. 整数オーバーフロー: 加算、減算、乗算、除算のオーバーフロー
2. 型不整合: nullポインタアクセス、アライメント違反
3. シフト演算: 範囲外シフト
4. 配列境界外: インデックス範囲外アクセス
5. ポインタオーバーフロー: ポインタ演算のオーバーフロー
6. 浮動小数点変換: 範囲外のfloatキャスト
7. 到達不能: unreachableコードへの到達

**関連システム・外部連携**：
- LLVMバックエンド: UBSanチェックコード生成
- Zigコンパイラ: ランタイムハンドラのリンク
- std.debug: panic処理

**権限による制御**：本モジュールは特別な権限制御を持たない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 実行ファイルビルド画面 | 補助機能 | サニタイザランタイムの組み込み |

## 機能種別

ランタイムライブラリ / サニタイザ / デバッグ支援

## 入力仕様

### 入力パラメータ

各ハンドラ関数に固有（例：オーバーフローハンドラ）:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| data | *const OverflowData | Yes | エラー情報構造体 | 有効なポインタ |
| lhs_handle | ValueHandle | Yes | 左辺値 | - |
| rhs_handle | ValueHandle | Yes | 右辺値 | - |

### 入力データソース

コンパイラ生成のサニタイザチェックコードからの暗黙的呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | noreturn | すべてのハンドラはpanicを発生 |

### 出力先

標準エラー出力（panic経由）

## 処理フロー

### 処理シーケンス

```
1. サニタイザチェック失敗
   └─ コンパイラ生成のチェックコードが異常を検出
2. ハンドラ呼び出し
   └─ 対応するubsanハンドラが呼び出される
3. エラー情報解析
   └─ ソース位置、型情報、値を取得
4. panicメッセージ生成
   └─ 人間可読なエラーメッセージを構築
5. panic発生
   └─ プログラムを異常終了
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{未定義動作種別}
    B -->|整数オーバーフロー| C[overflowHandler]
    B -->|ゼロ除算| D[divRemHandler]
    B -->|型不整合| E[typeMismatch]
    B -->|シフト範囲外| F[shiftOob]
    B -->|配列境界外| G[outOfBounds]
    B -->|ポインタオーバーフロー| H[pointerOverflow]
    C --> I[panicメッセージ生成]
    D --> I
    E --> I
    F --> I
    G --> I
    H --> I
    I --> J[panic発生]
    J --> K[プログラム終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-60-01 | noreturn保証 | すべてのハンドラは復帰しない | 常時 |
| BR-60-02 | weak linkage | ユーザー定義ハンドラで置換可能 | エクスポート時 |
| BR-60-03 | hidden visibility | シンボルは非公開 | リンク時 |
| BR-60-04 | abort版提供 | 各ハンドラに_abort版も存在 | ランタイム |

### 計算ロジック

オーバーフローハンドラの例:
```zig
fn overflowHandler(
    data: *const OverflowData,
    lhs_handle: ValueHandle,
    rhs_handle: ValueHandle,
) callconv(.c) noreturn {
    const lhs: Value = .{ .handle = lhs_handle, .td = data.td };
    const rhs: Value = .{ .handle = rhs_handle, .td = data.td };
    const signed_str = if (data.td.isSigned()) "signed" else "unsigned";
    panic(
        @returnAddress(),
        "{s} integer overflow: {f} + {f} cannot be represented in type {s}",
        .{ signed_str, lhs, rhs, data.td.getName() },
    );
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を直接行わない。

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| add_overflow | 整数オーバーフロー | 加算がオーバーフロー | コード修正 |
| sub_overflow | 整数オーバーフロー | 減算がオーバーフロー | コード修正 |
| mul_overflow | 整数オーバーフロー | 乗算がオーバーフロー | コード修正 |
| divrem_overflow | ゼロ除算/オーバーフロー | ゼロ除算または-1除算 | コード修正 |
| shift_out_of_bounds | シフト範囲外 | シフト量が不正 | コード修正 |
| out_of_bounds | 配列境界外 | インデックスが範囲外 | コード修正 |
| type_mismatch_v1 | 型不整合 | nullまたはアライメント違反 | コード修正 |
| pointer_overflow | ポインタオーバーフロー | ポインタ演算不正 | コード修正 |
| float_cast_overflow | 浮動小数点変換エラー | 値が範囲外 | コード修正 |
| builtin_unreachable | 到達不能コード到達 | unreachableに到達 | コード修正 |

### リトライ仕様

ランタイムハンドラはリトライしない（すべてpanicで終了）。

## トランザクション仕様

本機能はトランザクション制御を行わない。

## パフォーマンス要件

- ハンドラはエラー発生時のみ実行（ホットパスではない）
- パニックメッセージの生成は一度きり
- 通常実行時のオーバーヘッドはチェックコードのみ（ランタイム呼び出しなし）

## セキュリティ考慮事項

- 未定義動作の検出によるセキュリティ脆弱性の早期発見
- 整数オーバーフロー攻撃の防止
- バッファオーバーフローの検出
- nullポインタ参照の検出

## 備考

- powerpc、riscv64バックエンドではubsanビルド不可（can_build_ubsan条件）
- 一部のハンドラ（dynamic_type_cache_miss等）はItanium C++ ABI依存のため未実装
- `-fsanitize=function`は特定の理由で無効化されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ubsan_rtモジュールの基本データ構造を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ubsan_rt.zig | `lib/ubsan_rt.zig` | サニタイザランタイムの全実装 |

**読解のコツ**: 各ハンドラに渡されるデータ構造（SourceLocation, TypeDescriptor, Value等）を理解する。

**主要処理フロー**:
1. **6-10行目**: SourceLocation構造体（ファイル名、行、列）
2. **12-44行目**: TypeDescriptor構造体（型種別、サイズ、名前）
3. **46-135行目**: Value構造体とその操作（整数/浮動小数点値の取得）
4. **137-172行目**: OverflowDataとオーバーフローハンドラ
5. **174-189行目**: 否定ハンドラ（negationHandler）
6. **191-212行目**: 除算/剰余ハンドラ（divRemHandler）
7. **214-273行目**: アライメント仮定ハンドラ
8. **275-320行目**: シフト範囲外ハンドラ
9. **322-344行目**: 配列境界外ハンドラ
10. **346-402行目**: ポインタオーバーフローハンドラ
11. **404-473行目**: 型不整合ハンドラ
12. **475-485行目**: 到達不能/missing returnハンドラ
13. **617-643行目**: ハンドラエクスポート関数
14. **652-685行目**: comptimeでのハンドラ登録

#### Step 2: 整数値処理を理解する

Value型による整数/浮動小数点値の解釈を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ubsan_rt.zig | `lib/ubsan_rt.zig` (48-135行目) | Valueの値取得メソッド |

**読解のコツ**: インライン値とポインタ経由値の判定ロジックを理解する。

#### Step 3: ハンドラ登録を理解する

comptimeでのハンドラエクスポートパターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ubsan_rt.zig | `lib/ubsan_rt.zig` (617-685行目) | exportHandler/exportHandlerWithAbort |

### プログラム呼び出し階層図

```
ubsan_rt.zig (サニタイザランタイム)
    |
    +-- データ構造
    |      +-- SourceLocation (ソース位置情報)
    |      +-- TypeDescriptor (型記述子)
    |      +-- Value (値ハンドル)
    |      +-- ValueHandle (不透明ポインタ)
    |
    +-- オーバーフローハンドラ
    |      +-- add_overflow (加算)
    |      +-- sub_overflow (減算)
    |      +-- mul_overflow (乗算)
    |      +-- negate_overflow (否定)
    |      +-- divrem_overflow (除算/剰余)
    |
    +-- 型チェックハンドラ
    |      +-- type_mismatch_v1 (型不整合)
    |      +-- alignment_assumption (アライメント仮定)
    |      +-- load_invalid_value (不正値ロード)
    |
    +-- 範囲チェックハンドラ
    |      +-- shift_out_of_bounds (シフト範囲外)
    |      +-- out_of_bounds (配列境界外)
    |      +-- vla_bound_not_positive (VLAサイズ不正)
    |
    +-- ポインタ/浮動小数点ハンドラ
    |      +-- pointer_overflow (ポインタ演算)
    |      +-- float_cast_overflow (floatキャスト)
    |
    +-- その他ハンドラ
           +-- builtin_unreachable (到達不能)
           +-- missing_return (返り値なし)
           +-- nonnull_return_v1 (非nullリターン)
           +-- nonnull_arg (非null引数)
           +-- invalid_builtin (不正ビルトイン)
```

### データフロー図

```
[コンパイル時]                 [リンク時]                    [実行時]

C/C++ソースコード              ubsan_rt.zig
        |                           |
        v                           v
-fsanitize=undefined          シンボルエクスポート
        |                           |
        v                           v
チェックコード生成 ─────────> __ubsan_handle_* ─────────> ハンドラ関数
        |                           |                         |
        v                           v                         v
条件分岐挿入                  weak linkage              未定義動作検出
        |                                                     |
        v                                                     v
実行ファイル ─────────────────────────────────────────> panic終了
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ubsan_rt.zig | `lib/ubsan_rt.zig` | ソース | サニタイザランタイム全体 |
| std.debug | `lib/std/debug.zig` | ソース | panic関数 |
| std.mem | `lib/std/mem.zig` | ソース | isAligned等のユーティリティ |
| std.math | `lib/std/math.zig` | ソース | Log2Int等の型 |
