# 機能設計書 69-changelist

## 概要

本ドキュメントは、ZigコンパイラにおけるZIRチェンジリスト機能（changelist）の設計を記述する。この機能は、2つのZigソースファイル間のZIR（Zig Intermediate Representation）命令マッピングを計算するデバッグツールである。

### 本機能の処理概要

changelistコマンドは、2つのZigソースファイルを入力として受け取り、それぞれをZIRにコンパイルした後、旧ZIRの各命令が新ZIRのどの命令に対応するかを計算・表示する。これはインクリメンタルコンパイルのデバッグや検証に使用される。

**業務上の目的・背景**：Zigコンパイラはインクリメンタルコンパイルをサポートしており、ソースコードの変更時に再コンパイルが必要な部分を最小化する。このツールは、インクリメンタルコンパイルの基盤となるZIR命令マッピングアルゴリズムの動作を確認・デバッグするために使用される。

**機能の利用シーン**：
- インクリメンタルコンパイル機能の開発・デバッグ
- ZIR生成の変更影響分析
- コンパイラ内部動作の理解

**主要な処理内容**：
1. 旧ソースファイルのパースとZIR生成
2. 新ソースファイルのパースとZIR生成
3. Zcu.mapOldZirToNewによる命令マッピング計算
4. マッピング結果の表示

**関連システム・外部連携**：
- Zigコンパイラのフロントエンド（AST、ZIR生成）
- Zcuモジュール（インクリメンタルコンパイル）

**権限による制御**：デバッグビルドでのみ有効。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コマンドラインツール（画面なし） |

## 機能種別

デバッグツール / コンパイラ内部機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| old_source | []const u8 | Yes | 旧ソースファイルパス | 有効なZigファイル |
| new_source | []const u8 | Yes | 新ソースファイルパス | 有効なZigファイル |

### 入力データソース

- 旧バージョンのZigソースファイル
- 新バージョンのZigソースファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| instruction_mappings | []const u8 | 命令マッピングの表示（%old => %new形式） |

### 出力先

- 標準出力

## 処理フロー

### 処理シーケンス

```
1. 引数取得
   ├─ old_source_path取得
   └─ new_source_path取得

2. 旧ソース処理
   ├─ ファイル読み込み
   ├─ Ast.parse
   ├─ AstGen.generate
   └─ エラーチェック

3. 新ソース処理
   ├─ ファイル読み込み
   ├─ Ast.parse
   ├─ AstGen.generate
   └─ エラーチェック

4. マッピング計算
   └─ Zcu.mapOldZirToNew

5. 結果表示
   └─ 各マッピングエントリを出力
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数取得]
    B --> C[旧ソースファイル読み込み]
    C --> D[Ast.parse - 旧]
    D --> E[AstGen.generate - 旧]
    E --> F{ASTGen失敗?}
    F -->|Yes| G[エラー表示]
    G --> H[exit 1]
    F -->|No| I[新ソースファイル読み込み]
    I --> J[Ast.parse - 新]
    J --> K[AstGen.generate - 新]
    K --> L{ASTGen失敗?}
    L -->|Yes| M[エラー表示]
    M --> H
    L -->|No| N[Zcu.mapOldZirToNew]
    N --> O[マッピング反復]
    O --> P[%old => %new 出力]
    P --> Q{次エントリ?}
    Q -->|Yes| O
    Q -->|No| R[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-69-01 | デバッグ専用 | デバッグビルドでのみ利用可能 | dev.check(.changelist_command) |
| BR-69-02 | エラー時終了 | ZIR生成エラー時はexit(1) | loweringFailed() |
| BR-69-03 | カラー出力 | カラー設定はauto | Color.auto |

### 計算ロジック

**ZIR命令マッピング（Zcu.mapOldZirToNew）**：
- 旧ZIRの各命令について、対応する新ZIR命令を特定
- 命令の種類と位置情報を基に対応関係を計算
- AutoHashMapUnmanagedで結果を格納

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ファイルオープンエラー | ソースファイルが存在しない | パス確認 |
| - | ファイル読み込みエラー | 読み込み失敗 | ファイル確認 |
| - | ZIR生成エラー | 構文/意味エラー | ソース修正 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- メモリ: アリーナアロケータ使用で効率的なメモリ管理
- 2つのZIR生成とマッピング計算を実行

## セキュリティ考慮事項

- デバッグビルド専用機能
- ファイルアクセス権限確認

## 備考

- このコマンドはデバッグビルドでのみ有効
- 出力形式: `%{old_index} => %{new_index}`
- インクリメンタルコンパイルの内部動作理解に有用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: コマンド定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | main.zig | `src/main.zig` | cmdChangelist関数 |

**読解のコツ**:
- デバッグビルドチェック（dev.check）の目的を理解
- Ast.parseとAstGen.generateの役割を把握
- Zcu.mapOldZirToNewがマッピングの核心

**主要処理フロー**:
- **6572-6638行目**: cmdChangelist関数
  - 6573行目: dev.check(.changelist_command) - デバッグ専用確認
  - 6578-6588行目: 旧ソース読み込みとパース
  - 6589-6596行目: 新ソース読み込み
  - 6598-6620行目: ZIR生成とエラーチェック
  - 6622-6637行目: マッピング計算と表示

#### Step 2: ZIRマッピングを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Zcu.zig | `src/Zcu.zig` | mapOldZirToNew関数 |

**読解のコツ**:
- 命令インデックス間のマッピングアルゴリズム
- インクリメンタルコンパイルとの関連

### プログラム呼び出し階層図

```
cmdChangelist
    │
    ├─ dev.check(.changelist_command)
    │
    ├─ 旧ソース処理
    │      ├─ Io.Dir.cwd().openFile
    │      ├─ Io.File.Reader
    │      ├─ std.zig.readSourceFileToEndAlloc
    │      ├─ Ast.parse
    │      └─ AstGen.generate
    │             └─ loweringFailed → ErrorBundle
    │
    ├─ 新ソース処理
    │      ├─ Io.Dir.cwd().openFile
    │      ├─ Io.File.Reader
    │      ├─ std.zig.readSourceFileToEndAlloc
    │      ├─ Ast.parse
    │      └─ AstGen.generate
    │             └─ loweringFailed → ErrorBundle
    │
    ├─ Zcu.mapOldZirToNew
    │      └─ inst_map (AutoHashMapUnmanaged)
    │
    └─ 結果出力
           └─ stdout_bw.print
```

### データフロー図

```
[入力]                          [処理]                              [出力]

old_source.zig ─────────────▶ readSourceFileToEndAlloc
                                    │
                                    ▼
                              Ast.parse ──────────────▶ old_tree
                                    │
                                    ▼
                            AstGen.generate ──────────▶ old_zir
                                    │
                                    │
new_source.zig ─────────────▶ readSourceFileToEndAlloc
                                    │
                                    ▼
                              Ast.parse ──────────────▶ new_tree
                                    │
                                    ▼
                            AstGen.generate ──────────▶ new_zir
                                    │
                                    │
                    ┌───────────────┴───────────────┐
                    │                               │
                    ▼                               ▼
                old_zir                         new_zir
                    │                               │
                    └───────────┬───────────────────┘
                                │
                                ▼
                        Zcu.mapOldZirToNew
                                │
                                ▼
                    ┌───────────────────────┐
                    │ inst_map              │
                    │ Zir.Inst.Index =>     │
                    │ Zir.Inst.Index        │
                    └───────────────────────┘
                                │
                                ▼
                    stdout: %old => %new ─────────▶ 標準出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.zig | `src/main.zig` | ソース | cmdChangelist実装 |
| Zcu.zig | `src/Zcu.zig` | ソース | mapOldZirToNew実装 |
| AstGen.zig | `lib/std/zig/AstGen.zig` | ソース | ZIR生成 |
| Ast.zig | `lib/std/zig/Ast.zig` | ソース | AST構造 |
| Zir.zig | `lib/std/zig/Zir.zig` | ソース | ZIR構造 |
