# 機能設計書 70-dump-zir

## 概要

本ドキュメントは、ZigコンパイラにおけるZIRダンプ機能（dump-zir）の設計を記述する。この機能は、キャッシュされたZIR（Zig Intermediate Representation）ファイルの内容を人間が読める形式でダンプするデバッグツールである。

### 本機能の処理概要

dump-zirコマンドは、Zigコンパイラがキャッシュに保存したZIRファイルを入力として受け取り、命令数、サイズ統計、および各命令の詳細をテキスト形式で出力する。これはコンパイラの中間表現をデバッグ・分析するために使用される。

**業務上の目的・背景**：ZIR（Zig Intermediate Representation）はZigコンパイラの内部表現であり、ASTとSema（意味解析）の間の中間形式である。このツールにより、コンパイラ開発者はZIRの内容を確認し、コード生成やセマンティック解析の問題をデバッグできる。

**機能の利用シーン**：
- コンパイラ内部の動作デバッグ
- ZIR生成の正確性検証
- インクリメンタルコンパイルキャッシュの分析
- コンパイラ最適化の効果測定

**主要な処理内容**：
1. ZIRキャッシュファイルの読み込み
2. ZIR統計情報の計算・表示
3. ZIR命令のテキスト形式での出力

**関連システム・外部連携**：
- Zigコンパイラのキャッシュシステム
- print_zir.zigモジュール

**権限による制御**：デバッグビルドでのみ有効。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コマンドラインツール（画面なし） |

## 機能種別

デバッグツール / コンパイラ内部機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| cache_file | []const u8 | Yes | ZIRキャッシュファイルパス | 有効なファイル |

### 入力データソース

- ZIRキャッシュファイル（.zigキャッシュディレクトリ内）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| total_bytes | usize | ZIR全体のバイト数 |
| instructions | usize | 命令数と命令バイト数 |
| string_table | usize | 文字列テーブルバイト数 |
| extra_data | usize | 追加データ項目数とバイト数 |
| zir_text | []const u8 | ZIR命令のテキスト表現 |

### 出力先

- 標準出力

## 処理フロー

### 処理シーケンス

```
1. 引数取得
   └─ cache_fileパス取得

2. ファイル読み込み
   ├─ ファイルオープン
   └─ Zcu.loadZirCache

3. 統計情報表示
   ├─ 命令バイト数計算
   ├─ 追加データバイト数計算
   └─ 合計バイト数表示

4. ZIRテキスト出力
   └─ print_zir.renderAsText
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数取得]
    B --> C[cache_file取得]
    C --> D[ファイルオープン]
    D --> E{オープン成功?}
    E -->|No| F[fatal: unable to open]
    F --> G[終了]
    E -->|Yes| H[Zcu.loadZirCache]
    H --> I[統計計算]
    I --> J[instruction_bytes計算]
    J --> K[extra_bytes計算]
    K --> L[total_bytes計算]
    L --> M[統計ヘッダー出力]
    M --> N[print_zir.renderAsText]
    N --> O[flush]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-70-01 | デバッグ専用 | デバッグビルドでのみ利用可能 | dev.check(.dump_zir_command) |
| BR-70-02 | バイト計算 | 命令サイズはリリースサイズで計算 | @sizeOf(Zir.Inst.Tag) + 8 |
| BR-70-03 | ストリーミング出力 | バッファリングされた出力を使用 | writerStreaming |

### 計算ロジック

**命令バイト数計算**：
```
instruction_bytes = zir.instructions.len * (@sizeOf(Zir.Inst.Tag) + 8)
```
- `@sizeOf(Zir.Inst.Tag)` + 8（Zir.Inst.Dataのリリースサイズ）

**追加データバイト数計算**：
```
extra_bytes = zir.extra.len * @sizeOf(u32)
```

**合計バイト数計算**：
```
total_bytes = @sizeOf(Zir) + instruction_bytes + extra_bytes + zir.string_bytes.len
```

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ファイルオープンエラー | キャッシュファイルが存在しない | パス確認 |
| - | ZIRロードエラー | キャッシュ形式不正 | キャッシュ再生成 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- ストリーミング出力による効率的なメモリ使用
- 大きなZIRファイルにも対応

## セキュリティ考慮事項

- デバッグビルド専用機能
- ファイルアクセス権限確認

## 備考

- 出力形式はコメント形式のヘッダー（# で始まる）
- Bi（バイナリ単位）でサイズ表示
- キャッシュファイルは通常`.zig-cache`ディレクトリに存在

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: コマンド定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | main.zig | `src/main.zig` | cmdDumpZir関数 |

**読解のコツ**:
- デバッグビルドチェック（dev.check）の目的を理解
- Zcu.loadZirCacheの役割を把握
- print_zir.renderAsTextが出力の核心

**主要処理フロー**:
- **6528-6569行目**: cmdDumpZir関数
  - 6529行目: dev.check(.dump_zir_command) - デバッグ専用確認
  - 6533行目: cache_file引数取得
  - 6535-6538行目: ファイルオープン
  - 6540行目: Zcu.loadZirCache - ZIRの読み込み
  - 6543-6563行目: 統計情報の計算と出力
  - 6567行目: print_zir.renderAsText - ZIRのテキスト出力

#### Step 2: ZIR構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Zir.zig | `lib/std/zig/Zir.zig` | ZIR構造体とInst |

**読解のコツ**:
- `Zir`構造体のフィールド（instructions, extra, string_bytes）
- `Zir.Inst`の構造（Tag, Data）

#### Step 3: ZIR出力を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | print_zir.zig | `src/print_zir.zig` | renderAsText関数 |

**読解のコツ**:
- ZIR命令のテキスト表現形式
- 命令の種類ごとの出力処理

### プログラム呼び出し階層図

```
cmdDumpZir
    │
    ├─ dev.check(.dump_zir_command)
    │
    ├─ args[0] (cache_file)
    │
    ├─ Io.Dir.cwd().openFile
    │      └─ fatal (エラー時)
    │
    ├─ Zcu.loadZirCache
    │      └─ zir: Zir
    │
    ├─ 統計計算
    │      ├─ instruction_bytes
    │      │      └─ @sizeOf(Zir.Inst.Tag) + 8
    │      ├─ extra_bytes
    │      │      └─ @sizeOf(u32) * extra.len
    │      └─ total_bytes
    │
    ├─ stdout_bw.print (統計ヘッダー)
    │      └─ # Total ZIR bytes: ...
    │
    └─ print_zir.renderAsText
           └─ 各命令のテキスト出力
```

### データフロー図

```
[入力]                          [処理]                              [出力]

.zig-cache/xxx.zir ─────────▶ Io.Dir.cwd().openFile
                                    │
                                    ▼
                            Zcu.loadZirCache
                                    │
                                    ▼
                    ┌───────────────────────────────┐
                    │ Zir構造体                     │
                    │ ├─ instructions: []Inst      │
                    │ ├─ extra: []u32              │
                    │ └─ string_bytes: []u8        │
                    └───────────────────────────────┘
                                    │
                    ┌───────────────┴───────────────┐
                    │                               │
                    ▼                               ▼
            統計計算                        print_zir.renderAsText
                    │                               │
                    ▼                               ▼
    # Total ZIR bytes: xxxx              各命令テキスト
    # Instructions: xxxx                      │
    # String Table: xxxx                      │
    # Extra Data: xxxx                        │
                    │                               │
                    └───────────────┬───────────────┘
                                    │
                                    ▼
                                標準出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.zig | `src/main.zig` | ソース | cmdDumpZir実装 |
| Zcu.zig | `src/Zcu.zig` | ソース | loadZirCache実装 |
| print_zir.zig | `src/print_zir.zig` | ソース | renderAsText実装 |
| Zir.zig | `lib/std/zig/Zir.zig` | ソース | ZIR構造定義 |
