# 画面設計書 30-ZIRダンプ画面

## 概要

本ドキュメントは、Zigコンパイラの `zig dump-zir` コマンドによるZIRダンプ画面の設計仕様を定義する。この画面は、キャッシュされたZIRファイルの内容を人間が読める形式でダンプ・表示するデバッグ機能を提供する。

### 本画面の処理概要

ZIRダンプ画面は、コンパイラがキャッシュしたZIR（Zig Intermediate Representation）ファイルの内容を解析・表示するデバッグツールである。ZIRのサイズ情報と命令内容をテキスト形式で出力し、コンパイラの動作理解やデバッグを支援する。

**業務上の目的・背景**：ZIRはZigコンパイラの中間表現であり、ASTからネイティブコードへの変換過程で使用される。この画面は、コンパイラ開発者がZIRの内容を確認し、コンパイラの動作を理解・デバッグするために使用される。インクリメンタルコンパイル機能の開発においても、キャッシュされたZIRの検証に活用される。

**画面へのアクセス方法**：ターミナルから `zig dump-zir <cache_file>` コマンドを実行する。このコマンドはデバッグビルドでのみ利用可能。

**主要な操作・処理内容**：
1. キャッシュファイルを開く
2. ZIRデータをロード
3. ZIRの統計情報（サイズ、命令数等）を計算
4. 統計情報をヘッダーとして出力
5. ZIR命令をテキスト形式でレンダリング
6. 結果を標準出力に出力

**画面遷移**：本コマンドは独立したデバッグ機能であり、他の画面への遷移は発生しない。

**権限による表示制御**：このコマンドはデバッグビルド（`build_options.enable_debug_extensions = true`）でのみ利用可能。リリースビルドでは表示されない。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 70 | dump-zir | 主機能 | キャッシュされたZIRを含むファイルのダンプ処理 |
| 47 | ZIR | 補助機能 | Zig中間表現の解析処理 |

## 画面種別

デバッグツール（CLI出力）

## URL/ルーティング

コマンド: `zig dump-zir <cache_file>`

**注意**: デバッグモード限定コマンド

## 入出力項目

### 入力項目（コマンドライン引数）

| No | 項目名 | 型 | 必須 | 説明 |
|----|--------|-----|------|------|
| 1 | cache_file | ファイルパス | ○ | キャッシュされたZIRファイルのパス |

## 表示項目

### ヘッダー情報

| No | 項目名 | フォーマット | 説明 |
|----|--------|-------------|------|
| 1 | Total ZIR bytes | {Bi} | ZIR全体のバイト数 |
| 2 | Instructions | {d} ({Bi}) | 命令数とバイト数 |
| 3 | String Table Bytes | {Bi} | 文字列テーブルのバイト数 |
| 4 | Extra Data Items | {d} ({Bi}) | 追加データ項目数とバイト数 |

### ZIR命令ダンプ

`print_zir.zig` の `renderAsText` 関数により、ZIR命令がテキスト形式でレンダリングされる。

### 出力例

```
# Total ZIR bytes:    1.5KiB
# Instructions:       42 (672B)
# String Table Bytes: 256B
# Extra Data Items:   15 (60B)

[ZIR命令のテキスト表現]
```

## イベント仕様

### 1-コマンド実行

ユーザーが `zig dump-zir <cache_file>` を実行すると:

1. `dev.check(.dump_zir_command)` で機能有効性を確認
2. 引数からキャッシュファイルパスを取得
3. ファイルを開く
4. `Zcu.loadZirCache` でZIRデータをロード
5. ZIRのサイズ情報を計算:
   - 命令バイト数 = 命令数 * (Tag + Data)サイズ
   - 追加データバイト数 = extra.len * 4
   - 合計バイト数 = 上記 + 文字列テーブル + ヘッダー
6. 統計情報をヘッダーとして出力
7. `print_zir.renderAsText` でZIR命令をテキスト出力
8. 出力バッファをフラッシュ

## データベース更新仕様

本機能はデータベースを使用しない。

## メッセージ仕様

| 種別 | メッセージ | 発生条件 |
|------|----------|----------|
| エラー | "unable to open zir cache file for dumping '{s}': {s}" | キャッシュファイルが開けない |

## 例外処理

| エラー種別 | 説明 | 対処 |
|-----------|------|------|
| ファイルオープン失敗 | キャッシュファイルにアクセスできない | fatal でエラー出力して終了 |
| ZIRロード失敗 | ファイル形式が不正 | エラーを伝播 |
| メモリ不足 | 大きなZIRでメモリ枯渇 | エラーを伝播 |

## 備考

- このコマンドはデバッグビルドでのみ利用可能
- ZIRキャッシュファイルはコンパイル時にグローバルキャッシュに保存される
- サイズ計算ではリリースビルドのサイズを想定（デバッグ安全タグを除外）
- 出力はバッファリングされ、最後にフラッシュされる
- `print_zir.zig` モジュールがZIR命令のテキストレンダリングを担当

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ZIRの構造とキャッシュ形式を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | std.zig.Zir | 標準ライブラリ | ZIR構造（instructions, extra, string_bytes） |
| 1-2 | Zcu | `src/Zcu.zig` | `loadZirCache` 関数の理解 |

**読解のコツ**: ZIRは `instructions`（命令配列）、`extra`（追加データ）、`string_bytes`（文字列テーブル）から構成される。

#### Step 2: エントリーポイントを理解する

main.zigでのコマンドディスパッチを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | main.zig | `src/main.zig` | `zig dump-zir` コマンドの分岐処理（385-386行目） |

**主要処理フロー**:
1. **385行目**: デバッグ拡張有効かつコマンド名が "dump-zir" かを判定
2. **386行目**: `cmdDumpZir` 関数を呼び出し

#### Step 3: dump-zirコマンドの実装を理解する

cmdDumpZir関数の詳細を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | main.zig | `src/main.zig` | `cmdDumpZir` 関数（6528-6569行目） |

**主要処理フロー**:
- **6529行目**: `dev.check(.dump_zir_command)` で機能確認
- **6533行目**: 引数からキャッシュファイルパスを取得
- **6535-6538行目**: ファイルを開く
- **6540行目**: `Zcu.loadZirCache` でZIRをロード
- **6541-6542行目**: 標準出力ライターを初期化
- **6544-6563行目**: サイズ情報を計算・出力
- **6567行目**: `print_zir.renderAsText` でZIR命令を出力
- **6568行目**: 出力バッファをフラッシュ

#### Step 4: ZIRレンダリングを理解する

print_zir.zigのレンダリング処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | print_zir.zig | `src/print_zir.zig` | `renderAsText` 関数 |

### プログラム呼び出し階層図

```
main() [src/main.zig:171]
    │
    ├─ build_options.enable_debug_extensions チェック [385行目]
    │
    └─ cmdDumpZir() [src/main.zig:6528]
           │
           ├─ dev.check(.dump_zir_command) [6529行目]
           │
           ├─ Io.Dir.cwd().openFile() [6535行目]
           │
           ├─ Zcu.loadZirCache() [6540行目]
           │
           ├─ サイズ計算 [6544-6563行目]
           │      ├─ instruction_bytes計算
           │      ├─ extra_bytes計算
           │      └─ total_bytes計算
           │
           ├─ サイズ情報出力 [6552-6563行目]
           │
           ├─ print_zir.renderAsText() [6567行目]
           │
           └─ stdout_bw.flush() [6568行目]
```

### データフロー図

```
[入力]                    [処理]                           [出力]

cache_file (ファイル)
      │
      ▼
   openFile() ───▶ loadZirCache() ───┐
                                      │
                                      ▼
                              サイズ計算 ─────────▶ ヘッダー出力
                                      │               │
                                      ▼               │
                              renderAsText() ────────┼─▶ 標準出力
                                                     │
                                                     ▼
                                               ZIR命令テキスト
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.zig | `src/main.zig` | ソース | エントリーポイント、cmdDumpZir関数 |
| Zcu.zig | `src/Zcu.zig` | ソース | loadZirCache関数 |
| print_zir.zig | `src/print_zir.zig` | ソース | ZIRテキストレンダリング |
| std.zig.Zir | 標準ライブラリ | ソース | ZIR構造定義 |

### ZIRサイズ計算の詳細

```zig
// src/main.zig:6544-6550
const instruction_bytes = zir.instructions.len *
    // @sizeOf(Zir.Inst.Data) はデバッグ安全タグを含む可能性があるため
    // リリースサイズを想定して手動計算
    (@sizeOf(Zir.Inst.Tag) + 8);
const extra_bytes = zir.extra.len * @sizeOf(u32);
const total_bytes = @sizeOf(Zir) + instruction_bytes + extra_bytes +
    zir.string_bytes.len * @sizeOf(u8);
```
