# 通知設計書 13-fatal

## 概要

本ドキュメントは、Zigコンパイラにおける「fatal」通知の設計仕様を定義する。fatalは、プログラムを即座に終了させる重大なエラーを通知するための致命的エラー機能である。

### 本通知の処理概要

fatalは、回復不可能な重大なエラーが発生した際に、エラーメッセージを標準エラー出力に表示し、プロセスを即座に終了させる機能である。`std.process.fatal`関数を通じて呼び出される。

**業務上の目的・背景**：コンパイラやビルドシステムには、続行不可能な致命的状態が発生する場合がある。例えば、必須の設定ファイルが見つからない、キャッシュディレクトリが破損している、または内部の一貫性チェックが失敗した場合などである。このような状況では、不正な状態で処理を続行するよりも、明確なエラーメッセージと共に即座に終了することで、さらなる問題の発生を防ぎ、デバッグを容易にする。

**通知の送信タイミング**：プログラム実行中の任意の時点で、回復不可能なエラーが検出された瞬間に発生する。初期化処理、ファイルI/O、リソース確保など、様々なフェーズで発生しうる。

**通知の受信者**：コンパイラを実行している開発者、CI/CDシステム、またはビルドスクリプトがエラーメッセージの受信者となる。終了コードによって自動化システムはビルド失敗を検知できる。

**通知内容の概要**：エラーの種類と原因を説明するメッセージが含まれる。ファイルパス、環境変数名、期待される設定など、問題解決に必要なコンテキスト情報も含まれることが多い。

**期待されるアクション**：開発者は、エラーメッセージを確認し、根本原因（ファイルの欠如、設定の誤り、環境の問題など）を特定して修正する必要がある。自動化システムは、ビルド失敗を適切に報告し、必要に応じて通知を行う。

## 通知種別

致命的エラー / プロセス終了通知

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（即時プロセス終了） |
| 優先度 | 最高（即時終了） |
| リトライ | なし（回復不可能） |

### 送信先決定ロジック

`std.process.fatal`関数はフォーマット済みメッセージを標準エラー出力に書き込み、プロセスを終了コード1（または指定されたコード）で終了させる。出力先の変更は不可能。

## 通知テンプレート

### エラーメッセージの場合

| 項目 | 内容 |
|-----|------|
| 出力先 | 標準エラー出力（stderr） |
| 形式 | テキストメッセージ |

### 本文テンプレート

```
error: {エラーメッセージ}
```

または詳細なコンテキスト付き：

```
{コンテキスト説明}: {エラー詳細}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| format | フォーマット文字列 | コンパイル時定数 | Yes |
| args | フォーマット引数 | 実行時値 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 初期化処理 | 必須リソース未検出 | キャッシュディレクトリが開けない | fatal呼び出し |
| 初期化処理 | 設定エラー | zig libディレクトリがキャッシュと同一 | fatal呼び出し |
| ファイルI/O | 致命的I/Oエラー | ファイル操作の失敗 | fatal呼び出し |
| 内部エラー | 一貫性チェック失敗 | assertと同等の状況 | fatal呼び出し |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | fatalは常に実行される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[致命的エラー検出] --> B[std.process.fatal呼び出し]
    B --> C[メッセージフォーマット]
    C --> D[stderr書き込み]
    D --> E[プロセス終了 exit code 1]
```

## データベース参照・更新仕様

### 参照テーブル一覧

なし（fatalは即座にプロセスを終了するため、データベース操作は行わない）

### 更新テーブル一覧

なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ディレクトリアクセス失敗 | キャッシュ/libディレクトリ未検出 | 環境変数または引数で正しいパス指定 |
| 設定衝突 | libディレクトリとキャッシュが同一 | 異なるパスを指定 |
| cwd取得失敗 | カレントディレクトリ不明 | ファイルシステム確認 |
| 無限ループ検出 | Zigがシステムccとして設定 | CC環境変数を修正 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | なし（すべて致命的） |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- エラーメッセージにはファイルパスが含まれる場合がある
- 環境変数名が露出する可能性がある
- 内部状態の詳細が含まれる場合がある

## 備考

- `fatal`は`noreturn`型の関数であり、呼び出し後のコードは実行されない
- WASI環境では若干異なる動作をする可能性がある
- デバッグビルドではより詳細な情報が出力される場合がある

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

fatalは特別なデータ構造を持たないが、`std.process.fatal`関数の実装を理解する必要がある。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | process.zig | `lib/std/process.zig` | `fatal`関数の定義。フォーマット文字列と引数を受け取り、stderrに出力後プロセス終了 |

**読解のコツ**: `fatal`は`@compileError`や`@panic`とは異なり、ユーザー向けの明確なエラーメッセージを出力することが目的。

#### Step 2: 使用箇所を理解する

Compilation.zigとmain.zigでfatalがどのように使用されているかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Compilation.zig | `src/Compilation.zig` | 14行目: `const fatal = std.process.fatal;`のインポート |
| 2-2 | Compilation.zig | `src/Compilation.zig` | 770-808行目: Directories.init内でのfatal使用例 |
| 2-3 | main.zig | `src/main.zig` | 66行目: `const fatal = std.process.fatal;`のインポート |

**主要処理フロー**:
1. **771行目**: cwd取得失敗時のfatal
2. **778行目**: zig libディレクトリ検出失敗時のfatal
3. **804-808行目**: ディレクトリ衝突時のfatal

#### Step 3: 具体的なエラーケースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | main.zig | `src/main.zig` | 237-244行目: CC環境変数による無限ループ防止 |
| 3-2 | Zcu/PerThread.zig | `src/Zcu/PerThread.zig` | 195-207行目: キャッシュディレクトリ削除時のfatal |

**主要処理フロー**:
- **195行目**: macOSでのファイルレース条件対処
- **237行目**: システムccがZig自身の場合の検出

### プログラム呼び出し階層図

```
main
    │
    ├─ Directories.init
    │      ├─ introspect.getResolvedCwd
    │      │      └─ 失敗時 → fatal
    │      ├─ introspect.findZigLibDirFromSelfExe
    │      │      └─ 失敗時 → fatal
    │      └─ ディレクトリ衝突チェック
    │             └─ 衝突時 → fatal
    │
    ├─ CC環境変数チェック
    │      └─ 無限ループ検出 → fatal
    │
    └─ Compilation.update
           └─ 各種処理
                  └─ 致命的エラー → fatal
```

### データフロー図

```
[入力]                      [処理]                         [出力]

エラー状況 ───────▶ fatal(format, args) ───────▶ stderr
                          │
                          ▼
                    プロセス終了
                          │
                          ▼
                    exit code: 1
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| process.zig | `lib/std/process.zig` | ソース | std.process.fatal関数の定義 |
| Compilation.zig | `src/Compilation.zig` | ソース | 初期化処理でのfatal使用 |
| main.zig | `src/main.zig` | ソース | エントリーポイントでのfatal使用 |
| PerThread.zig | `src/Zcu/PerThread.zig` | ソース | ファイル処理でのfatal使用 |
| introspect.zig | `src/introspect.zig` | ソース | パス解決処理 |
