# 通知設計書 7-SourceLocationError

## 概要

本ドキュメントは、Zigコンパイラにおけるソース位置エラー（SourceLocationError）通知の設計を記述するものである。

### 本通知の処理概要

本通知は、Zigコンパイラのリンク処理中に特定のソースコード位置に関連するエラーを報告する機能である。ソース位置情報により、ユーザーはエラーの発生箇所を正確に特定できる。

**業務上の目的・背景**：リンク処理において発生したエラーをソースコード上の具体的な位置と関連付けて報告するために必要である。これにより、ユーザーは問題箇所を迅速に特定し、修正を行うことができる。

**通知の送信タイミング**：リンク処理中にソース位置情報を伴うエラー条件が検出された時点でDiagsシステムにエラーが追加され、`error.LinkFailure`が返される。

**通知の受信者**：コンパイラを実行しているユーザー。エラーメッセージは最終的に標準エラー出力（stderr）に出力される。

**通知内容の概要**：エラーメッセージとソース位置情報（ファイル、行、列）が含まれる。

**期待されるアクション**：ユーザーはエラーメッセージとソース位置を確認し、該当箇所のソースコードを修正する。

## 通知種別

診断メッセージ（ErrorBundle経由での出力）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（エラー蓄積後にバッチ出力） |
| 優先度 | 高 |
| リトライ | 無し |

### 送信先決定ロジック

エラーメッセージは`link.Diags`構造体に蓄積され、ソース位置情報が`Diags.SourceLocation`として添付される。最終的に`Compilation.getAllErrorsAlloc()`によってErrorBundleに変換されて出力される。

## 通知テンプレート

### エラー出力の場合

| 項目 | 内容 |
|-----|------|
| 出力先 | stderr（ErrorBundle経由） |
| 形式 | 構造化エラーメッセージ（ソース位置付き） |

### 本文テンプレート

```
{ファイルパス}:{行}:{列}: error: {エラーメッセージ}
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | 診断メッセージのため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| msg | エラーメッセージ | format引数で生成 | Yes |
| source_location | ソース位置情報 | Diags.SourceLocation | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| リンク処理 | Diags.failSourceLocation呼び出し | ソース位置付きエラー発生時 | 汎用ソース位置エラー |
| Wasmリンク | diags.failSourceLocation呼び出し | Wasm固有エラー発生時 | WasmリンカーでのSourceLocation使用 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | エラー発生時は常にDiagsに追加される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ソース位置付きエラー発生] --> B[failSourceLocation呼び出し]
    B --> C[addErrorSourceLocation呼び出し]
    C --> D[メッセージフォーマット]
    D --> E[SourceLocation付きでDiags.msgsに追加]
    E --> F[error.LinkFailure返却]
    F --> G[呼び出し元でエラーハンドリング]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| なし | - | 診断メッセージのためDB参照なし |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| なし | - | 診断メッセージのためDB更新なし |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| OutOfMemory | メッセージ生成時のメモリ不足 | setAllocFailure()で記録 |
| LinkFailure | ソース位置付きリンクエラー | Diagsにエラー追加後に返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0回 |
| リトライ間隔 | なし |
| リトライ対象エラー | なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（リンク処理中は常時発生可能）

## セキュリティ考慮事項

- エラーメッセージにファイルパスやソースコード情報が含まれるが、これはユーザーの作業環境情報であり、外部送信されない
- エラー出力は標準エラー出力のみで、ネットワーク経由での送信は行われない

## 備考

- SourceLocationはunion型で、現在は`.none`と`.wasm`の2つのバリアントを持つ
- Wasm以外のリンカーでもソース位置情報をサポートする拡張が可能

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Diags.SourceLocation型の定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | link.zig | `src/link.zig` | 43-46行目: SourceLocation union型の定義 |

**読解のコツ**: SourceLocationはunion(enum)型で、noneまたはwasm固有の位置情報を持つ。

#### Step 2: エントリーポイントを理解する

failSourceLocation関数とaddErrorSourceLocation関数の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | link.zig | `src/link.zig` | 219-223行目: failSourceLocation関数 |
| 2-2 | link.zig | `src/link.zig` | 230-240行目: addErrorSourceLocation関数 |

**主要処理フロー**:
1. **219行目**: `failSourceLocation` - ソース位置付きでエラーを追加し`error.LinkFailure`を返す
2. **221行目**: `addErrorSourceLocation(diags, sl, format, args);` - ソース位置付きエラーの追加
3. **222行目**: `return error.LinkFailure;` - エラー値の返却

#### Step 3: 使用箇所

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | link/Wasm.zig | `src/link/Wasm.zig` | 572行目: Wasm固有のソース位置エラー |

**主要処理フロー**:
- **572行目**: `return diags.failSourceLocation(.{ .wasm = sl }, format, args);`

### プログラム呼び出し階層図

```
link.Diags.failSourceLocation(sl, format, args)
    │
    ├─ addErrorSourceLocation(diags, sl, format, args)
    │      │
    │      ├─ std.fmt.allocPrint(gpa, format, args)
    │      │      └─ エラーメッセージ生成
    │      │
    │      ├─ diags.mutex.lockUncancelable(io)
    │      │
    │      └─ addErrorLockedFallible(diags, sl, eu_main_msg)
    │             └─ diags.msgs.append(.{.msg, .source_location = sl})
    │
    └─ return error.LinkFailure
```

### データフロー図

```
[入力]               [処理]                    [出力]

ソース位置 ───▶ failSourceLocation() ───▶ error.LinkFailure
エラー情報            │
                     └─ Diags.msgsに蓄積
                           （ソース位置情報付き）

                         │
                         ▼

             Compilation.getAllErrorsAlloc()
                         │
                         ▼

                 ErrorBundle（stderr出力）
                 file.zig:10:5: error: ...
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| link.zig | `src/link.zig` | ソース | SourceLocation型、failSourceLocation関数の定義 |
| link/Wasm.zig | `src/link/Wasm.zig` | ソース | WasmリンカーでのSourceLocation使用 |
| Compilation.zig | `src/Compilation.zig` | ソース | getAllErrorsAlloc、ErrorBundle生成 |
