# Zig Programming Language 業務運用マニュアル

## 目次

**【基本操作】**
1. [システム概要](#1-システム概要)
2. [インストールとセットアップ](#2-インストールとセットアップ)
3. [ヘルプとバージョン確認](#3-ヘルプとバージョン確認)

**【開発ワークフローに沿った機能】**
4. [【STEP1】プロジェクト初期化（新規プロジェクト作成）](#4-step1プロジェクト初期化新規プロジェクト作成)
5. [【STEP2】コーディングとコード品質管理](#5-step2コーディングとコード品質管理)
6. [【STEP3】ビルドとコンパイル](#6-step3ビルドとコンパイル)
7. [【STEP4】テスト実行](#7-step4テスト実行)
8. [【STEP5】実行とデバッグ](#8-step5実行とデバッグ)

**【サポート機能】**
9. [パッケージ管理](#9-パッケージ管理)
10. [C/C++連携](#10-cc連携)
11. [情報表示ツール](#11-情報表示ツール)

**【運用ガイド】**
12. [日常開発の流れ](#12-日常開発の流れ)
13. [よくある質問と対処法](#13-よくある質問と対処法)
14. [トラブルシューティング](#14-トラブルシューティング)

---

## 1. システム概要

### 1.1 Zig Programming Languageとは

Zigは、堅牢性、最適化、再利用可能なソフトウェアを維持するための汎用プログラミング言語およびツールチェーンです。C/C++との相互運用性を重視し、LLVMをバックエンドとして使用することで、高性能なネイティブコードを生成します。

**主な特徴：**
- 自己ホスト型コンパイラ（Zig言語自身で書かれている）
- 強力なコンパイル時評価機能
- C/C++との高い相互運用性
- クロスコンパイルの容易さ
- メモリ安全性を重視した設計

### 1.2 本システムで行う主要業務

本ツールチェーンは、主に以下の開発業務を支援します：

| 業務カテゴリ | 具体的な業務内容 |
|------------|----------------|
| **プロジェクト管理** | プロジェクト初期化、パッケージ管理、依存関係解決 |
| **コード開発** | コンパイル、ビルド、コード品質チェック、フォーマット |
| **テスト・品質保証** | ユニットテスト実行、構文チェック、バグ最小化 |
| **C/C++連携** | translate-c、ドロップインC/C++コンパイラ、互換ツール |
| **情報取得** | 環境情報表示、ドキュメント閲覧、ターゲット確認 |

### 1.3 開発ワークフロー概要

本ツールチェーンにおける開発は、以下の流れで進行します：

```
┌─────────────────────────────────────────────────────────────────┐
│                     開発ワークフロー                             │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│  【STEP 1】zig init でプロジェクトを初期化                      │
│      ↓                                                          │
│  【STEP 2】コードを記述し、zig fmt でフォーマット               │
│      ↓                                                          │
│  【STEP 3】zig build または zig build-exe でビルド              │
│      ↓                                                          │
│  【STEP 4】zig test でテストを実行                              │
│      ↓                                                          │
│  【STEP 5】zig run で実行、デバッグ                             │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

**ユーザー別の主な業務：**

| ユーザー種別 | 主な業務 |
|-------------|---------|
| **一般開発者** | コード記述、ビルド、テスト実行、デバッグ |
| **プロジェクトリーダー** | プロジェクト初期化、依存関係管理、CI/CD設定 |
| **C/C++移行担当者** | translate-c、C/C++コンパイラ互換機能の利用 |

### 1.4 主な機能

| 機能 | 説明 | ワークフローでの位置 |
|------|------|---------------------|
| init | プロジェクト初期化 | STEP 1 |
| fmt | コードフォーマット | STEP 2 |
| ast-check | 構文チェック | STEP 2 |
| build | プロジェクトビルド | STEP 3 |
| build-exe | 実行ファイル作成 | STEP 3 |
| test | テスト実行 | STEP 4 |
| run | 即時実行 | STEP 5 |
| fetch | パッケージ取得 | 随時利用 |
| translate-c | CからZigへ変換 | 随時利用 |

### 1.5 ユーザー種別

| ユーザー種別 | 説明 |
|-------------|------|
| 開発者 | Zigを使用してソフトウェアを開発するユーザー |
| システム管理者 | Zigツールチェーンのインストール・管理を行うユーザー |
| C/C++開発者 | 既存のC/C++プロジェクトにZigを統合するユーザー |

### 1.6 対応環境

**対応プラットフォーム：**
- Linux（x86_64、aarch64、他多数）
- macOS（x86_64、aarch64）
- Windows（x86_64）
- FreeBSD、NetBSD、その他

**必要条件：**
- システムC/C++ツールチェーン（オプション）
- LLVM 21.x（オプション、非LLVMビルドも可能）

---

## 2. インストールとセットアップ

### 2.1 ビルド済みバイナリのインストール

1. [公式ダウンロードページ](https://ziglang.org/download/)にアクセスします
2. プラットフォームに応じたバイナリをダウンロードします
3. アーカイブを展開します

**入力項目：**

| 項目 | 説明 |
|------|------|
| プラットフォーム | Linux/macOS/Windows等を選択 |
| アーキテクチャ | x86_64/aarch64等を選択 |

4. PATHに `zig` 実行ファイルのディレクトリを追加します

### 2.2 ソースからのビルド（LLVM使用）

```bash
mkdir build
cd build
cmake ..
make install
```

`stage3/bin/zig` にコンパイラが生成されます。

### 2.3 ソースからのビルド（LLVM無し）

```bash
cc -o bootstrap bootstrap.c
./bootstrap
```

カレントディレクトリに `zig2` 実行ファイルが生成されます。

> **注意**: この方法ではリリースモード最適化など一部機能が制限されます。

### 2.4 インストール確認

```bash
zig version
```

バージョン番号が表示されればインストール成功です。

---

## 3. ヘルプとバージョン確認

### 3.1 ヘルプの表示

コマンドの使用方法を確認するには以下を実行します：

```bash
zig help
```

または：

```bash
zig -h
zig --help
```

**表示内容：**
- 利用可能なコマンド一覧
- 各コマンドの簡単な説明
- グローバルオプション

### 3.2 サブコマンドのヘルプ

各サブコマンドの詳細ヘルプを表示するには：

```bash
zig build --help
zig build-exe --help
zig test --help
```

### 3.3 バージョン確認

```bash
zig version
```

### 3.4 Zenの表示

Zigの設計哲学を表示するには：

```bash
zig zen
```

---

## 4. 【STEP1】プロジェクト初期化（新規プロジェクト作成）

> **開発ワークフローでの位置**: 新規プロジェクトを開始する最初のステップです。

### 4.1 プロジェクトの作成

1. プロジェクト用のディレクトリを作成します
2. 「zig init」コマンドを実行します

```bash
mkdir my-project
cd my-project
zig init
```

**生成されるファイル：**

| ファイル | 説明 |
|---------|------|
| build.zig | ビルドシステム設定ファイル |
| build.zig.zon | パッケージマニフェスト |
| src/main.zig | メインソースファイル |
| src/root.zig | ライブラリルートファイル |

### 4.2 生成されるプロジェクト構造

```
my-project/
├── build.zig          # ビルド設定
├── build.zig.zon      # パッケージマニフェスト
└── src/
    ├── main.zig       # 実行ファイルのエントリポイント
    └── root.zig       # ライブラリのルート
```

### 4.3 注意点

- 既存のファイルがあるディレクトリで実行すると、既存ファイルを上書きしません
- プロジェクト名はディレクトリ名から自動的に設定されます

---

## 5. 【STEP2】コーディングとコード品質管理

> **開発ワークフローでの位置**: コードを記述し、品質を維持するステップです。

### 5.1 コードフォーマット

Zigソースコードを標準形式に整形します：

```bash
zig fmt src/main.zig
```

**ディレクトリ全体をフォーマット：**

```bash
zig fmt src/
```

**オプション：**

| オプション | 説明 |
|-----------|------|
| `--check` | フォーマットが必要かどうかをチェックのみ |
| `--exclude` | 除外するファイル/ディレクトリを指定 |

### 5.2 構文チェック（AST検査）

コンパイル前に構文エラーを検出します：

```bash
zig ast-check src/main.zig
```

**用途：**
- エディタ統合での即時エラー検出
- CI/CDパイプラインでの事前チェック

### 5.3 バグレポート最小化

バグを再現する最小限のコードを生成します：

```bash
zig reduce buggy_file.zig
```

> **注意**: この機能はデバッグ支援用です。

---

## 6. 【STEP3】ビルドとコンパイル

> **開発ワークフローでの位置**: コードをコンパイルして成果物を生成するステップです。

### 6.1 プロジェクトビルド

build.zigに基づいてプロジェクト全体をビルドします：

```bash
zig build
```

**主なオプション：**

| オプション | 説明 |
|-----------|------|
| `-Doptimize=ReleaseSafe` | 最適化モードを指定 |
| `-Dtarget=x86_64-linux` | ターゲットプラットフォームを指定 |
| `--release` | リリースビルド |

### 6.2 実行ファイルのビルド

Zigソースから直接実行ファイルを生成します：

```bash
zig build-exe src/main.zig
```

**クロスコンパイル例：**

```bash
zig build-exe src/main.zig -target x86_64-linux
zig build-exe src/main.zig -target aarch64-macos
zig build-exe src/main.zig -target x86_64-windows
```

### 6.3 ライブラリのビルド

静的/動的ライブラリを生成します：

```bash
zig build-lib src/lib.zig
```

**オプション：**

| オプション | 説明 |
|-----------|------|
| `-dynamic` | 動的ライブラリを生成 |
| `-shared` | 共有ライブラリを生成 |

### 6.4 オブジェクトファイルのビルド

オブジェクトファイルのみを生成します：

```bash
zig build-obj src/main.zig
```

### 6.5 最適化オプション

| オプション | 説明 | 用途 |
|-----------|------|------|
| Debug | デフォルト、最適化なし | 開発時 |
| ReleaseSafe | 最適化 + 安全性チェック | テスト環境 |
| ReleaseFast | 最大速度最適化 | 本番環境（速度重視） |
| ReleaseSmall | 最小サイズ最適化 | 組み込み等 |

---

## 7. 【STEP4】テスト実行

> **開発ワークフローでの位置**: コードの品質を検証するステップです。

### 7.1 テストの実行

Zigソースファイル内のテストを実行します：

```bash
zig test src/main.zig
```

**プロジェクト全体のテスト：**

```bash
zig build test
```

### 7.2 テストフィルタ

特定のテストのみを実行します：

```bash
zig test src/main.zig --test-filter "test_name"
```

### 7.3 テストオブジェクトの生成

テストをオブジェクトファイルとして出力します：

```bash
zig test-obj src/main.zig
```

> **用途**: 他のビルドシステムとの統合時に使用します。

### 7.4 テストの書き方

```zig
const std = @import("std");
const testing = std.testing;

test "basic test" {
    try testing.expectEqual(1 + 1, 2);
}
```

---

## 8. 【STEP5】実行とデバッグ

> **開発ワークフローでの位置**: ビルドした成果物を実行し、動作確認するステップです。

### 8.1 即時実行

ビルドと実行を一度に行います：

```bash
zig run src/main.zig
```

**引数を渡す場合：**

```bash
zig run src/main.zig -- arg1 arg2
```

### 8.2 ビルド後の実行

```bash
zig build
./zig-out/bin/my-project
```

### 8.3 デバッグビルド

```bash
zig build -Doptimize=Debug
```

### 8.4 デバッガの使用

LLDB/GDB用のpretty printerスクリプトが提供されています：

```bash
lldb ./zig-out/bin/my-project
# LLDB内で
command script import tools/lldb_pretty_printers.py
```

---

## 9. パッケージ管理

### 9.1 パッケージの取得

依存パッケージをグローバルキャッシュにダウンロードします：

```bash
zig fetch https://example.com/package.tar.gz
```

**ハッシュ付きで取得：**

```bash
zig fetch --save https://example.com/package.tar.gz
```

### 9.2 依存関係の追加

build.zig.zonに依存関係を追加します：

```zig
.dependencies = .{
    .package_name = .{
        .url = "https://example.com/package.tar.gz",
        .hash = "1220...",
    },
},
```

### 9.3 依存関係の解決

```bash
zig build
```

ビルド時に依存関係が自動的に解決されます。

---

## 10. C/C++連携

### 10.1 CからZigへの変換

CコードをZigコードに変換します：

```bash
zig translate-c header.h > translated.zig
```

**オプション：**

| オプション | 説明 |
|-----------|------|
| `-I<path>` | インクルードパスを追加 |
| `-D<macro>` | マクロを定義 |

### 10.2 ドロップインCコンパイラ

ZigをCコンパイラとして使用します：

```bash
zig cc -o output source.c
```

### 10.3 ドロップインC++コンパイラ

ZigをC++コンパイラとして使用します：

```bash
zig c++ -o output source.cpp
```

### 10.4 互換ツール

| コマンド | 説明 |
|---------|------|
| `zig ar` | アーカイバ（静的ライブラリ作成） |
| `zig ranlib` | アーカイブインデックス生成 |
| `zig lib` | Windows lib.exe互換 |
| `zig dlltool` | DLLインポートライブラリ生成 |
| `zig objcopy` | オブジェクトファイル変換 |
| `zig rc` | Windowsリソースコンパイラ |

---

## 11. 情報表示ツール

### 11.1 環境情報

```bash
zig env
```

**表示内容：**
- ライブラリパス
- 標準ライブラリパス
- キャッシュディレクトリ
- バージョン情報

### 11.2 標準ライブラリドキュメント

ブラウザでドキュメントを表示します：

```bash
zig std
```

### 11.3 libc情報

```bash
zig libc
```

### 11.4 ターゲット一覧

利用可能なコンパイルターゲットを表示します：

```bash
zig targets
```

---

## 12. 日常開発の流れ

本セクションでは、開発ワークフローに沿った日常業務の流れを説明します。

```
開発ワークフロー概要（再掲）
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
【STEP 1】zig init でプロジェクトを初期化 → 4章参照
    ↓
【STEP 2】コードを記述し、zig fmt でフォーマット → 5章参照
    ↓
【STEP 3】zig build または zig build-exe でビルド → 6章参照
    ↓
【STEP 4】zig test でテストを実行 → 7章参照
    ↓
【STEP 5】zig run で実行、デバッグ → 8章参照
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
```

### 12.1 新規プロジェクト開始

```bash
mkdir my-project
cd my-project
zig init
```

### 12.2 開発サイクル

1. **コード編集**: src/main.zigを編集
2. **フォーマット**: `zig fmt src/`
3. **ビルド確認**: `zig build`
4. **テスト実行**: `zig build test`
5. **実行確認**: `zig run src/main.zig`

### 12.3 リリース準備

```bash
# リリースビルド
zig build -Doptimize=ReleaseSafe

# クロスコンパイル（複数プラットフォーム向け）
zig build-exe src/main.zig -target x86_64-linux
zig build-exe src/main.zig -target x86_64-windows
zig build-exe src/main.zig -target aarch64-macos
```

---

## 13. よくある質問と対処法

### 13.1 ビルドに関するQ&A

**Q: ビルドが失敗します**

A: 以下を確認してください：
1. `zig version` でZigが正しくインストールされているか確認
2. build.zigファイルが存在するか確認
3. ソースファイルに構文エラーがないか `zig ast-check` で確認

**Q: クロスコンパイルができません**

A: 以下を確認してください：
1. ターゲットが正しいか `zig targets` で確認
2. `-target` オプションの形式: `<arch>-<os>-<abi>`

### 13.2 パッケージに関するQ&A

**Q: 依存パッケージがダウンロードできません**

A: 以下を確認してください：
1. ネットワーク接続を確認
2. URLが正しいか確認
3. `zig fetch` でハッシュを取得して build.zig.zon に追加

### 13.3 C/C++連携に関するQ&A

**Q: translate-cで変換エラーが発生します**

A: 以下を確認してください：
1. 必要なインクルードパスを `-I` オプションで指定
2. 必要なマクロを `-D` オプションで定義
3. システムヘッダーのパスが正しいか確認

---

## 14. トラブルシューティング

### 14.1 エラーメッセージと対処法

| エラーメッセージ | 原因 | 対処法 |
|----------------|------|--------|
| FileNotFound | ソースファイルが見つからない | パスを確認、ファイル存在を確認 |
| SyntaxError | 構文エラー | `zig ast-check` でエラー箇所を特定 |
| UnresolvedDependency | 依存パッケージが解決できない | `zig fetch` でパッケージを取得 |
| TargetNotSupported | ターゲットが非対応 | `zig targets` でサポート確認 |

### 14.2 よくあるトラブルと解決方法

**コンパイル時間が長い**

- `-fincremental` オプションでインクリメンタルコンパイルを有効化
- `-j` オプションで並列ビルドスレッド数を調整

**メモリ不足エラー**

- 大規模プロジェクトでは十分なメモリを確保
- ビルドキャッシュをクリア: `rm -rf zig-cache`

**キャッシュの問題**

- キャッシュをクリア: `rm -rf zig-cache ~/.cache/zig`

### 14.3 サポートへの連絡方法

- **公式ドキュメント**: https://ziglang.org/documentation/
- **GitHub Issues**: https://github.com/ziglang/zig/issues
- **Discord**: Zig公式Discordコミュニティ
- **IRC**: libera.chat #zig

---

## 改訂履歴

| 版数 | 改訂日 | 改訂内容 |
|-----|--------|---------|
| 1.0 | 2026-02-01 | 初版作成 |

---

*本マニュアルは、Zig Programming Languageのエンドユーザー向け業務運用マニュアルです。*
*ツールチェーンの更新により、コマンドや動作が変更される場合があります。*
