# UI Design System Implementation

This document outlines a complete implementation of the UI design system shown in the screenshot, with a focus on the mint green and light gray color scheme. The design system is built for Next.js and follows WCAG 2.0 contrast guidelines.

## Table of Contents

1. [Theme Variables (theme.css)](#theme-variables)
2. [Global Styles (globals.css)](#global-styles)
3. [Implementation Guide](#implementation-guide)
   - [Typography](#typography)
   - [Color System](#color-system)
   - [Layout & Spacing](#layout--spacing)
   - [Components](#components)
   - [Accessibility](#accessibility)
4. [Usage Examples](#usage-examples)

## Theme Variables

Create a `theme.css` file with the following CSS variables:

```css
:root {
  /* Colors - Green */
  --green-1: #f9fefc;
  --green-2: #f2fcf8;
  --green-3: #def9ef;
  --green-4: #c9f4e4;
  --green-5: #b3ecd8;
  --green-6: #9be1c9;
  --green-7: #7ad1b5;
  --green-8: #42bd9b;
  --green-9: #6bdfbc;
  --green-10: #61d4b2;
  --green-11: #007c5f;
  --green-12: #144235;

  /* Colors - Gray */
  --gray-1: #fcfcfd;
  --gray-2: #f9f9fb;
  --gray-3: #f0f0f4;
  --gray-4: #e7e8ed;
  --gray-5: #e0e1e7;
  --gray-6: #d8d9e0;
  --gray-7: #cdced7;
  --gray-8: #b9bbc7;
  --gray-9: #8c8d99;
  --gray-10: #81828d;
  --gray-11: #62636c;
  --gray-12: #1f1f25;

  /* Functional Colors */
  --primary: var(--green-9);
  --primary-hover: var(--green-10);
  --primary-active: var(--green-11);
  --primary-contrast: white;
  
  --background: white;
  --foreground: var(--gray-12);
  
  --muted: var(--gray-3);
  --muted-foreground: var(--gray-11);
  
  --accent: var(--green-3);
  --accent-foreground: var(--green-11);
  
  --border: var(--gray-6);
  --input: var(--gray-7);
  --ring: var(--green-7);
  
  --success: #10b981;
  --warning: #f59e0b;
  --danger: #ef4444;
  --info: #3b82f6;
  
  /* Typography */
  --font-sans: 'Roboto', 'Noto Sans JP', sans-serif;
  --font-mono: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, monospace;
  
  /* Spacing (8px Grid System) */
  --space-1: 0.25rem; /* 4px */
  --space-2: 0.5rem;  /* 8px */
  --space-3: 0.75rem; /* 12px */
  --space-4: 1rem;    /* 16px */
  --space-5: 1.25rem; /* 20px */
  --space-6: 1.5rem;  /* 24px */
  --space-8: 2rem;    /* 32px */
  --space-10: 2.5rem; /* 40px */
  --space-12: 3rem;   /* 48px */
  --space-16: 4rem;   /* 64px */
  
  /* Border Radius */
  --radius-sm: 0.125rem; /* 2px */
  --radius: 0.375rem;    /* 6px */
  --radius-md: 0.5rem;   /* 8px */
  --radius-lg: 0.75rem;  /* 12px */
  --radius-full: 9999px;
  
  /* Shadows */
  --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
  --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
  --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
  --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
  
  /* Animation */
  --transition-fast: 150ms cubic-bezier(0.4, 0, 0.2, 1);
  --transition: 200ms cubic-bezier(0.4, 0, 0.2, 1);
  --transition-slow: 300ms cubic-bezier(0.4, 0, 0.2, 1);
}

/* Dark theme (for future implementation) */
.dark {
  --background: var(--gray-12);
  --foreground: var(--gray-1);
  
  --muted: var(--gray-11);
  --muted-foreground: var(--gray-4);
  
  --accent: var(--green-11);
  --accent-foreground: var(--green-3);
  
  --border: var(--gray-10);
  --input: var(--gray-10);
  --ring: var(--green-10);
}
```

## Global Styles

Create a `globals.css` file with the following styles:

```css
@import './theme.css';

/* Import fonts from Google Fonts */
@import url('https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&family=Noto+Sans+JP:wght@300;400;500;700&display=swap');

/* Base Reset */
*, *::before, *::after {
  box-sizing: border-box;
}

* {
  margin: 0;
  padding: 0;
}

html, body {
  height: 100%;
}

html {
  font-size: 16px;
  line-height: 1.5;
  -webkit-font-smoothing: antialiased;
  -moz-osx-font-smoothing: grayscale;
  text-rendering: optimizeLegibility;
}

body {
  font-family: var(--font-sans);
  background-color: var(--background);
  color: var(--foreground);
  line-height: 1.5;
}

img, picture, video, canvas, svg {
  display: block;
  max-width: 100%;
}

input, button, textarea, select {
  font: inherit;
}

p, h1, h2, h3, h4, h5, h6 {
  overflow-wrap: break-word;
}

/* Typography */
h1, .h1 {
  font-size: 2.25rem; /* 36px */
  font-weight: 700;
  line-height: 1.2;
  margin-bottom: var(--space-4);
}

h2, .h2 {
  font-size: 1.75rem; /* 28px */
  font-weight: 700;
  line-height: 1.3;
  margin-bottom: var(--space-3);
}

h3, .h3 {
  font-size: 1.5rem; /* 24px */
  font-weight: 600;
  line-height: 1.4;
  margin-bottom: var(--space-2);
}

h4, .h4 {
  font-size: 1.25rem; /* 20px */
  font-weight: 600;
  line-height: 1.4;
  margin-bottom: var(--space-2);
}

h5, .h5 {
  font-size: 1.125rem; /* 18px */
  font-weight: 500;
  line-height: 1.5;
  margin-bottom: var(--space-2);
}

h6, .h6 {
  font-size: 1rem; /* 16px */
  font-weight: 500;
  line-height: 1.5;
  margin-bottom: var(--space-2);
}

p {
  margin-bottom: var(--space-4);
}

small, .text-sm {
  font-size: 0.875rem; /* 14px */
}

.text-xs {
  font-size: 0.75rem; /* 12px */
}

.text-lg {
  font-size: 1.125rem; /* 18px */
}

.text-xl {
  font-size: 1.25rem; /* 20px */
}

.text-2xl {
  font-size: 1.5rem; /* 24px */
}

.font-light {
  font-weight: 300;
}

.font-normal {
  font-weight: 400;
}

.font-medium {
  font-weight: 500;
}

.font-bold {
  font-weight: 700;
}

/* Links */
a {
  color: var(--primary);
  text-decoration: none;
  transition: color var(--transition-fast);
}

a:hover {
  color: var(--primary-hover);
  text-decoration: underline;
}

/* Buttons */
.btn {
  display: inline-flex;
  align-items: center;
  justify-content: center;
  border-radius: var(--radius);
  font-weight: 500;
  font-size: 0.875rem;
  line-height: 1.5;
  padding: var(--space-2) var(--space-4);
  border: 1px solid transparent;
  cursor: pointer;
  transition: all var(--transition-fast);
  white-space: nowrap;
}

.btn:focus {
  outline: 2px solid transparent;
  outline-offset: 2px;
  box-shadow: 0 0 0 2px var(--background), 0 0 0 4px var(--ring);
}

.btn:disabled, .btn[disabled] {
  opacity: 0.5;
  cursor: not-allowed;
}

.btn-primary {
  background-color: var(--primary);
  color: var(--primary-contrast);
  border-color: var(--primary);
}

.btn-primary:hover {
  background-color: var(--primary-hover);
  border-color: var(--primary-hover);
}

.btn-primary:active {
  background-color: var(--primary-active);
  border-color: var(--primary-active);
}

.btn-secondary {
  background-color: var(--gray-3);
  color: var(--gray-12);
  border-color: var(--gray-3);
}

.btn-secondary:hover {
  background-color: var(--gray-4);
  border-color: var(--gray-4);
}

.btn-secondary:active {
  background-color: var(--gray-5);
  border-color: var(--gray-5);
}

.btn-outline {
  background-color: transparent;
  color: var(--foreground);
  border-color: var(--border);
}

.btn-outline:hover {
  background-color: var(--gray-2);
}

.btn-ghost {
  background-color: transparent;
  color: var(--foreground);
  border-color: transparent;
}

.btn-ghost:hover {
  background-color: var(--gray-2);
}

.btn-sm {
  padding: var(--space-1) var(--space-2);
  font-size: 0.75rem;
}

.btn-lg {
  padding: var(--space-3) var(--space-6);
  font-size: 1rem;
}

/* Forms */
.form-group {
  margin-bottom: var(--space-4);
}

.form-label {
  display: block;
  font-size: 0.875rem;
  font-weight: 500;
  margin-bottom: var(--space-1);
  color: var(--foreground);
}

.form-input, .form-select, .form-textarea {
  display: block;
  width: 100%;
  padding: var(--space-2) var(--space-3);
  font-size: 0.875rem;
  line-height: 1.5;
  color: var(--foreground);
  background-color: var(--background);
  border: 1px solid var(--border);
  border-radius: var(--radius);
  transition: border-color var(--transition-fast), box-shadow var(--transition-fast);
}

.form-input:focus, .form-select:focus, .form-textarea:focus {
  outline: none;
  border-color: var(--primary);
  box-shadow: 0 0 0 3px var(--green-4);
}

.form-input::placeholder, .form-textarea::placeholder {
  color: var(--muted-foreground);
  opacity: 1;
}

.form-textarea {
  min-height: 100px;
  resize: vertical;
}

.form-select {
  appearance: none;
  background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='currentColor' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Cpath d='M6 9l6 6 6-6'/%3E%3C/svg%3E");
  background-repeat: no-repeat;
  background-position: right var(--space-3) center;
  background-size: 16px 16px;
  padding-right: var(--space-8);
}

.form-checkbox, .form-radio {
  margin-right: var(--space-2);
  cursor: pointer;
}

.form-checkbox-label, .form-radio-label {
  display: flex;
  align-items: center;
  font-size: 0.875rem;
  cursor: pointer;
}

/* Card */
.card {
  background-color: var(--background);
  border-radius: var(--radius-lg);
  box-shadow: var(--shadow);
  overflow: hidden;
}

.card-header {
  padding: var(--space-4) var(--space-6);
  border-bottom: 1px solid var(--border);
}

.card-title {
  font-size: 1.125rem;
  font-weight: 600;
  margin: 0;
}

.card-content {
  padding: var(--space-4) var(--space-6);
}

.card-footer {
  padding: var(--space-4) var(--space-6);
  border-top: 1px solid var(--border);
}

/* Badge */
.badge {
  display: inline-flex;
  align-items: center;
  justify-content: center;
  border-radius: var(--radius-full);
  padding: 0 var(--space-2);
  font-size: 0.75rem;
  font-weight: 500;
  line-height: 1.4;
  height: 1.25rem;
  white-space: nowrap;
}

.badge-primary {
  background-color: var(--primary);
  color: var(--primary-contrast);
}

.badge-secondary {
  background-color: var(--gray-3);
  color: var(--gray-11);
}

.badge-success {
  background-color: var(--success);
  color: white;
}

.badge-warning {
  background-color: var(--warning);
  color: white;
}

.badge-danger {
  background-color: var(--danger);
  color: white;
}

.badge-info {
  background-color: var(--info);
  color: white;
}

/* Table */
.table {
  width: 100%;
  border-collapse: collapse;
}

.table th, .table td {
  padding: var(--space-3) var(--space-4);
  text-align: left;
  border-bottom: 1px solid var(--border);
}

.table th {
  font-weight: 500;
  background-color: var(--gray-2);
}

.table tbody tr:hover {
  background-color: var(--gray-1);
}

/* Alert */
.alert {
  padding: var(--space-3) var(--space-4);
  border-radius: var(--radius);
  border-left: 4px solid transparent;
  margin-bottom: var(--space-4);
}

.alert-success {
  background-color: rgba(16, 185, 129, 0.1);
  border-left-color: var(--success);
  color: var(--success);
}

.alert-warning {
  background-color: rgba(245, 158, 11, 0.1);
  border-left-color: var(--warning);
  color: var(--warning);
}

.alert-danger {
  background-color: rgba(239, 68, 68, 0.1);
  border-left-color: var(--danger);
  color: var(--danger);
}

.alert-info {
  background-color: rgba(59, 130, 246, 0.1);
  border-left-color: var(--info);
  color: var(--info);
}

/* Toggle Switch */
.toggle {
  position: relative;
  display: inline-block;
  width: 40px;
  height: 24px;
}

.toggle input {
  opacity: 0;
  width: 0;
  height: 0;
}

.toggle-slider {
  position: absolute;
  cursor: pointer;
  top: 0;
  left: 0;
  right: 0;
  bottom: 0;
  background-color: var(--gray-6);
  transition: var(--transition);
  border-radius: var(--radius-full);
}

.toggle-slider:before {
  position: absolute;
  content: "";
  height: 18px;
  width: 18px;
  left: 3px;
  bottom: 3px;
  background-color: white;
  transition: var(--transition);
  border-radius: 50%;
}

.toggle input:checked + .toggle-slider {
  background-color: var(--primary);
}

.toggle input:checked + .toggle-slider:before {
  transform: translateX(16px);
}

.toggle input:focus + .toggle-slider {
  box-shadow: 0 0 0 2px var(--green-4);
}

/* Calendar */
.calendar {
  width: 100%;
  font-size: 0.875rem;
}

.calendar-header {
  display: flex;
  align-items: center;
  justify-content: space-between;
  padding: var(--space-2);
}

.calendar-weekdays {
  display: grid;
  grid-template-columns: repeat(7, 1fr);
  text-align: center;
  font-weight: 500;
  font-size: 0.75rem;
  color: var(--gray-11);
  padding: var(--space-2) 0;
}

.calendar-grid {
  display: grid;
  grid-template-columns: repeat(7, 1fr);
  gap: 2px;
}

.calendar-day {
  aspect-ratio: 1/1;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  border-radius: var(--radius);
  transition: var(--transition-fast);
}

.calendar-day:hover:not(.calendar-day-disabled):not(.calendar-day-selected) {
  background-color: var(--gray-3);
}

.calendar-day-today {
  font-weight: 600;
}

.calendar-day-selected {
  background-color: var(--primary);
  color: white;
}

.calendar-day-disabled {
  color: var(--gray-6);
  cursor: not-allowed;
}

/* Progress Bar */
.progress {
  height: 8px;
  overflow: hidden;
  background-color: var(--gray-3);
  border-radius: var(--radius-full);
}

.progress-bar {
  height: 100%;
  background-color: var(--primary);
  border-radius: var(--radius-full);
  transition: width var(--transition-slow);
}

/* Utility Classes */
.text-center {
  text-align: center;
}

.text-right {
  text-align: right;
}

.text-primary {
  color: var(--primary);
}

.text-muted {
  color: var(--muted-foreground);
}

.text-success {
  color: var(--success);
}

.text-warning {
  color: var(--warning);
}

.text-danger {
  color: var(--danger);
}

.text-info {
  color: var(--info);
}

.bg-primary {
  background-color: var(--primary);
}

.bg-muted {
  background-color: var(--muted);
}

.rounded {
  border-radius: var(--radius);
}

.rounded-lg {
  border-radius: var(--radius-lg);
}

.rounded-full {
  border-radius: var(--radius-full);
}

.shadow {
  box-shadow: var(--shadow);
}

.shadow-lg {
  box-shadow: var(--shadow-lg);
}

.container {
  width: 100%;
  padding-right: var(--space-4);
  padding-left: var(--space-4);
  margin-right: auto;
  margin-left: auto;
}

@media (min-width: 640px) {
  .container {
    max-width: 640px;
  }
}

@media (min-width: 768px) {
  .container {
    max-width: 768px;
  }
}

@media (min-width: 1024px) {
  .container {
    max-width: 1024px;
  }
}

@media (min-width: 1280px) {
  .container {
    max-width: 1280px;
  }
}

.flex {
  display: flex;
}

.flex-col {
  flex-direction: column;
}

.items-center {
  align-items: center;
}

.justify-center {
  justify-content: center;
}

.justify-between {
  justify-content: space-between;
}

.gap-1 {
  gap: var(--space-1);
}

.gap-2 {
  gap: var(--space-2);
}

.gap-4 {
  gap: var(--space-4);
}

.p-2 {
  padding: var(--space-2);
}

.p-4 {
  padding: var(--space-4);
}

.py-2 {
  padding-top: var(--space-2);
  padding-bottom: var(--space-2);
}

.px-4 {
  padding-left: var(--space-4);
  padding-right: var(--space-4);
}

.m-2 {
  margin: var(--space-2);
}

.m-4 {
  margin: var(--space-4);
}

.my-2 {
  margin-top: var(--space-2);
  margin-bottom: var(--space-2);
}

.mt-4 {
  margin-top: var(--space-4);
}

.mb-4 {
  margin-bottom: var(--space-4);
}

.w-full {
  width: 100%;
}

.h-full {
  height: 100%;
}

.grid {
  display: grid;
}

.grid-cols-2 {
  grid-template-columns: repeat(2, minmax(0, 1fr));
}

.grid-cols-3 {
  grid-template-columns: repeat(3, minmax(0, 1fr));
}

.grid-cols-4 {
  grid-template-columns: repeat(4, minmax(0, 1fr));
}

.sr-only {
  position: absolute;
  width: 1px;
  height: 1px;
  padding: 0;
  margin: -1px;
  overflow: hidden;
  clip: rect(0, 0, 0, 0);
  white-space: nowrap;
  border-width: 0;
}
```

## Implementation Guide

### Typography

For Next.js, configure the fonts in your `_app.js` or `layout.js` (for Next.js 13+) using the Next.js font system:

```javascript
// In _app.js (Next.js 12) or layout.js (Next.js 13+)
import { Roboto, Noto_Sans_JP } from 'next/font/google';

// Configure the fonts
const roboto = Roboto({
  weight: ['300', '400', '500', '700'],
  subsets: ['latin'],
  display: 'swap',
  variable: '--font-roboto',
});

const notoSansJP = Noto_Sans_JP({
  weight: ['300', '400', '500', '700'],
  subsets: ['latin'],
  display: 'swap',
  variable: '--font-noto-sans-jp',
});

// Apply the fonts to the body
function MyApp({ Component, pageProps }) {
  return (
    <div className={`${roboto.variable} ${notoSansJP.variable}`}>
      <Component {...pageProps} />
    </div>
  );
}

export default MyApp;
```

### Color System

The color system is designed with accessibility in mind, following WCAG 2.0 contrast guidelines. The primary color scheme uses mint/teal green colors for accent with various shades of gray for the supporting UI.

The colors are organized as:
- Primary color (mint/teal green): Used for buttons, links, and interactive elements
- Gray scale: Used for text, backgrounds, and borders
- Semantic colors: For alerts, status indicators (success, warning, danger, info)

### Layout & Spacing

The design system uses an 8-point grid system for spacing and layout:

- All spacing values are multiples of 4px or 8px
- Margin and padding use this consistent scale
- Component sizing follows this grid

Container classes provide responsive layouts with appropriate breakpoints:
- 640px (small)
- 768px (medium)
- 1024px (large)
- 1280px (extra large)

### Components

#### Buttons

The design provides four button variants:
- Primary: Mint green background, white text
- Secondary: Light gray background, dark text
- Outline: Transparent background with border
- Ghost: Transparent background without border

Each button has three states:
- Default
- Hover
- Active/Pressed

#### Forms

Form elements include:
- Text inputs
- Textareas
- Select dropdowns
- Checkboxes and radio buttons
- Toggle switches

All form elements have consistent styling with focus states that are accessible.

#### Cards

Cards are used to group related content:
- Card header with title
- Card content area
- Optional card footer

#### Alerts

Alert components for four different types of messages:
- Success (green)
- Warning (yellow/orange)
- Danger (red)
- Info (blue)

#### Calendar

A styled calendar component similar to the one in the screenshot:
- Month navigation
- Day grid
- Selected, today, and disabled states

### Accessibility

The design system follows WCAG 2.0 guidelines for accessibility:
- Sufficient color contrast between text and background
- Focus states for interactive elements
- Semantic HTML structure
- Screen reader considerations (sr-only class)

## Usage Examples

### Card Example

```jsx
<div className="card">
  <div className="card-header">
    <h3 className="card-title">Total Revenue</h3>
  </div>
  <div className="card-content">
    <h2 className="text-2xl font-bold">$15,231.89</h2>
    <p className="text-sm text-muted">+20.1% from last month</p>
    {/* Chart component would go here */}
  </div>
</div>
```

### Form Example

```jsx
<form>
  <div className="form-group">
    <label className="form-label" htmlFor="email">Email</label>
    <input 
      className="form-input" 
      type="email" 
      id="email" 
      placeholder="email@example.com" 
    />
  </div>
  
  <div className="form-group">
    <label className="form-label" htmlFor="password">Password</label>
    <input 
      className="form-input" 
      type="password" 
      id="password" 
    />
  </div>
  
  <div className="form-group">
    <label className="toggle">
      <input type="checkbox" />
      <span className="toggle-slider"></span>
    </label>
    <span className="ml-2">Remember me</span>
  </div>
  
  <button className="btn btn-primary">Sign In</button>
</form>
```

### Calendar Example

```jsx
<div className="calendar">
  <div className="calendar-header">
    <button className="btn btn-ghost btn-sm">&lt;</button>
    <span>June 2023</span>
    <button className="btn btn-ghost btn-sm">&gt;</button>
  </div>
  
  <div className="calendar-weekdays">
    <div>Su</div>
    <div>Mo</div>
    <div>Tu</div>
    <div>We</div>
    <div>Th</div>
    <div>Fr</div>
    <div>Sa</div>
  </div>
  
  <div className="calendar-grid">
    {/* Example days */}
    <div className="calendar-day calendar-day-selected">5</div>
    <div className="calendar-day">6</div>
    <div className="calendar-day">7</div>
    <div className="calendar-day">8</div>
    <div className="calendar-day">9</div>
    <div className="calendar-day">10</div>
    <div className="calendar-day">11</div>
    <div className="calendar-day">12</div>
    <div className="calendar-day calendar-day-today">13</div>
    {/* ... and so on */}
  </div>
</div>
```

### Stats Card Example

```jsx
<div className="card">
  <div className="card-header">
    <h3 className="card-title">Subscriptions</h3>
  </div>
  <div className="card-content">
    <h2 className="text-2xl font-bold">+2350</h2>
    <p className="text-sm text-muted">+80.1% from last month</p>
    <div className="mt-4">
      {/* Bar chart would go here */}
    </div>
  </div>
</div>
```

### Progress Example

```jsx
<div>
  <div className="flex justify-between mb-2">
    <span className="text-sm font-medium">Progress</span>
    <span className="text-sm text-muted">75%</span>
  </div>
  <div className="progress">
    <div className="progress-bar" style={{ width: '75%' }}></div>
  </div>
</div>
```

### Alert Examples

```jsx
<div className="alert alert-success">
  <strong>Success!</strong> Your changes have been saved.
</div>

<div className="alert alert-warning">
  <strong>Warning!</strong> Your subscription will expire soon.
</div>

<div className="alert alert-danger">
  <strong>Error!</strong> There was a problem processing your request.
</div>

<div className="alert alert-info">
  <strong>Info:</strong> New features are available.
</div>
```

### Table Example

```jsx
<table className="table">
  <thead>
    <tr>
      <th>Status</th>
      <th>Email</th>
      <th>Amount</th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td>
        <span className="badge badge-success">Success</span>
      </td>
      <td>ken99@yahoo.com</td>
      <td>$316.00</td>
      <td className="text-right">...</td>
    </tr>
    <tr>
      <td>
        <span className="badge badge-success">Success</span>
      </td>
      <td>abe45@gmail.com</td>
      <td>$242.00</td>
      <td className="text-right">...</td>
    </tr>
    <tr>
      <td>
        <span className="badge badge-warning">Processing</span>
      </td>
      <td>monserrat44@gmail.com</td>
      <td>$837.00</td>
      <td className="text-right">...</td>
    </tr>
  </tbody>
</table>
```

## TailwindCSS Integration

If you want to use this design system with TailwindCSS, you can extend the Tailwind configuration to match the design variables:

```javascript
// tailwind.config.js
module.exports = {
  content: [
    './pages/**/*.{js,ts,jsx,tsx}',
    './components/**/*.{js,ts,jsx,tsx}',
  ],
  theme: {
    extend: {
      colors: {
        // Green
        'green-1': '#f9fefc',
        'green-2': '#f2fcf8',
        'green-3': '#def9ef',
        'green-4': '#c9f4e4',
        'green-5': '#b3ecd8',
        'green-6': '#9be1c9',
        'green-7': '#7ad1b5',
        'green-8': '#42bd9b',
        'green-9': '#6bdfbc',
        'green-10': '#61d4b2',
        'green-11': '#007c5f',
        'green-12': '#144235',
        
        // Gray
        'gray-1': '#fcfcfd',
        'gray-2': '#f9f9fb',
        'gray-3': '#f0f0f4',
        'gray-4': '#e7e8ed',
        'gray-5': '#e0e1e7',
        'gray-6': '#d8d9e0',
        'gray-7': '#cdced7',
        'gray-8': '#b9bbc7',
        'gray-9': '#8c8d99',
        'gray-10': '#81828d',
        'gray-11': '#62636c',
        'gray-12': '#1f1f25',
        
        // Semantic
        primary: 'var(--green-9)',
        success: '#10b981',
        warning: '#f59e0b',
        danger: '#ef4444',
        info: '#3b82f6',
      },
      borderRadius: {
        'sm': '0.125rem',
        DEFAULT: '0.375rem',
        'md': '0.5rem',
        'lg': '0.75rem',
        'full': '9999px',
      },
      fontFamily: {
        sans: ['var(--font-roboto)', 'var(--font-noto-sans-jp)', 'sans-serif'],
        mono: ['ui-monospace', 'SFMono-Regular', 'Menlo', 'Monaco', 'Consolas', 'monospace'],
      },
      boxShadow: {
        sm: '0 1px 2px 0 rgb(0 0 0 / 0.05)',
        DEFAULT: '0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1)',
        md: '0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1)',
        lg: '0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1)',
      },
      spacing: {
        // 8-point grid system
        '1': '0.25rem', // 4px
        '2': '0.5rem',  // 8px
        '3': '0.75rem', // 12px
        '4': '1rem',    // 16px
        '5': '1.25rem', // 20px
        '6': '1.5rem',  // 24px
        '8': '2rem',    // 32px
        '10': '2.5rem', // 40px
        '12': '3rem',   // 48px
        '16': '4rem',   // 64px
      },
    },
  },
  plugins: [],
}
```

### Component Implementation with TailwindCSS

With Tailwind properly configured, you can build components that match your design system without writing custom CSS:

```jsx
// Button component using Tailwind
function Button({ children, variant = 'primary', size = 'md', ...props }) {
  const variantClasses = {
    primary: 'bg-green-9 text-white hover:bg-green-10 active:bg-green-11',
    secondary: 'bg-gray-3 text-gray-12 hover:bg-gray-4 active:bg-gray-5',
    outline: 'bg-transparent border border-gray-6 hover:bg-gray-2',
    ghost: 'bg-transparent hover:bg-gray-2',
  };

  const sizeClasses = {
    sm: 'py-1 px-2 text-xs',
    md: 'py-2 px-4 text-sm',
    lg: 'py-3 px-6 text-base',
  };

  return (
    <button
      className={`rounded font-medium transition-colors focus:outline-none focus:ring-2 focus:ring-green-4 focus:ring-offset-2 ${variantClasses[variant]} ${sizeClasses[size]}`}
      {...props}
    >
      {children}
    </button>
  );
}
```

## Responsive Design Considerations

The design system is built with responsiveness in mind:

1. **Mobile-first approach**: All components are designed to work well on small screens first
2. **Responsive containers**: The `container` class has breakpoints for different screen sizes
3. **Flexible components**: Cards, tables, and forms adapt to their containers
4. **Grid system**: Using CSS Grid and Flexbox for responsive layouts

## Best Practices for Implementation

1. **Consistency**: Use the design system variables consistently across your application
2. **Accessibility**: Ensure sufficient contrast and proper focus states
3. **Component Reusability**: Create reusable components based on this design system
4. **Documentation**: Document component usage for your team
5. **Theme Extension**: Consider adding a dark mode theme using the variables

## Final Notes

This design system provides a comprehensive foundation for building a user interface that matches the provided screenshot. The CSS variables and utility classes can be used directly, or integrated with a framework like TailwindCSS for a more utility-first approach.

For a complete implementation, you would create React components for each UI element, utilizing these styles and ensuring they follow accessibility best practices.disabled">28</div>
    <div className="calendar-day calendar-day-disabled">29</div>
    <div className="calendar-day calendar-day-disabled">30</div>
    <div className="calendar-day calendar-day-disabled">31</div>
    <div className="calendar-day">1</div>
    <div className="calendar-day">2</div>
    <div className="calendar-day">3</div>
    <div className="calendar-day">4</div>
    <div className="calendar-day calendar-day-